------------------------------------------------------------------------------
--	Common LUA support functions specific to Civilization 6
------------------------------------------------------------------------------

include( "ToolTipHelper" );


-- ===========================================================================
--	CONSTANTS
-- ===========================================================================

local TUTORIAL_UUID	:string = "17462E0F-1EE1-4819-AAAA-052B5896B02A";

ProductionType = {
	BUILDING	= "BUILDING",
	DISTRICT	= "DISTRICT",
	PROJECT		= "PROJECT",
	UNIT		= "UNIT"
}


-- ===========================================================================
--	FUNCTIONS
-- ===========================================================================


-- ===========================================================================
--	Return the inline text-icon for a given yield
--	yieldType	A database YIELD_TYPE
--	returns		The [ICON_yield] string
-- ===========================================================================
function GetYieldTextIcon( yieldType:string )
	local  iconString:string = "";
	if		yieldType == nil or yieldType == ""	then
		iconString="Error:NIL";
	elseif	GameInfo.Yields[yieldType] ~= nil and GameInfo.Yields[yieldType].IconString ~= nil and GameInfo.Yields[yieldType].IconString ~= "" then
		iconString=GameInfo.Yields[yieldType].IconString;
	else
		iconString = "Unknown:"..yieldType; 
	end			
	return iconString;
end


-- ===========================================================================
--	Return the inline entry for a yield's color
-- ===========================================================================
function GetYieldTextColor( yieldType:string )
	if		yieldType == nil or yieldType == "" then return "[COLOR:255,255,255,255]NIL ";
	elseif	yieldType == "YIELD_FOOD"			then return "[COLOR:ResFoodLabelCS]";
	elseif	yieldType == "YIELD_PRODUCTION"		then return "[COLOR:ResProductionLabelCS]";
	elseif	yieldType == "YIELD_GOLD"			then return "[COLOR:ResGoldLabelCS]";
	elseif	yieldType == "YIELD_SCIENCE"		then return "[COLOR:ResScienceLabelCS]";
	elseif	yieldType == "YIELD_CULTURE"		then return "[COLOR:ResCultureLabelCS]";
	elseif	yieldType == "YIELD_FAITH"			then return "[COLOR:ResFaithLabelCS]";
	else											 return "[COLOR:255,255,255,0]ERROR ";
	end				
end

-- ===========================================================================
--	Return a string with +/- or 0 based on any value.
-- ===========================================================================
function toPlusMinusString( value:number )
	if(value == 0) then
		return "0";
	else
		return Locale.ToNumber(value, "+#,###.#;-#,###.#");
	end
end

-- ===========================================================================
--	Return a string with +/- or 0 based on any value.
-- ===========================================================================
function toPlusMinusNoneString( value:number )
	if(value == 0) then
		return " ";
	else
		return Locale.ToNumber(value, "+#,###.#;-#,###.#");
	end
end


-- ===========================================================================
--	Return a string with a yield icon and a +/- based on yield amount.
-- ===========================================================================
function GetYieldString( yieldType:string, amount:number )
	return GetYieldTextIcon(yieldType)..GetYieldTextColor(yieldType)..toPlusMinusString(amount).."[ENDCOLOR]";
end

-- ===========================================================================
--	Move a unit to X,Y
-- ===========================================================================
function MoveUnitToPlot( kUnit:table, plotX:number, plotY:number )
	if kUnit ~= nil then
		local tParameters:table = {};
		tParameters[UnitOperationTypes.PARAM_X] = plotX;
		tParameters[UnitOperationTypes.PARAM_Y] = plotY;		
		
		-- Will this start a war?  Note, we are ignoring destinations in the for that will start a war, the unit will be allowed to move until they are adjacent.
		-- We may want to also skip the war check if the move will take more than one turn to get to the destination.
		local eAttackingPlayer:number = kUnit:GetOwner();
		local eUnitComponentID:table = kUnit:GetComponentID();
		local bWillStartWar = PlayersVisibility[eAttackingPlayer]:IsVisible(plotX, plotY) and CombatManager.IsAttackChangeWarState(eUnitComponentID, plotX, plotY);
		if (bWillStartWar)then
			local eDefendingPlayer = CombatManager.GetBestDefender(eUnitComponentID, plotX, plotY );
			if (eDefendingPlayer == nil) then
				local pPlot = Map.GetPlot(plotX, plotY);
				eDefendingPlayer = pPlot:GetOwner();
			end
			-- Create the action specific parameters 
			if (eDefendingPlayer ~= nil and eDefendingPlayer ~= -1) then
				LuaEvents.Civ6Common_ConfirmWarDialog(eAttackingPlayer, eDefendingPlayer, WarTypes.SURPRISE_WAR);
			end
		else
			RequestMoveOperation(kUnit, tParameters, plotX, plotY);
		end
	end			
end

-- ===========================================================================
--  Requests an operation based on the type of unit and parameters
-- ===========================================================================
function RequestMoveOperation( kUnit:table, tParameters:table, plotX:number, plotY:number )
	-- Air units move and attack slightly differently than land and naval units
	if ( GameInfo.Units[kUnit:GetUnitType()].Domain == "DOMAIN_AIR" ) then
		tParameters[UnitOperationTypes.PARAM_MODIFIERS] = UnitOperationMoveModifiers.ATTACK;
		if (UnitManager.CanStartOperation( kUnit, UnitOperationTypes.AIR_ATTACK, nil, tParameters) ) then
			UnitManager.RequestOperation(kUnit, UnitOperationTypes.AIR_ATTACK, tParameters);
		elseif (UnitManager.CanStartOperation( kUnit, UnitOperationTypes.DEPLOY, nil, tParameters) ) then
			UnitManager.RequestOperation(kUnit, UnitOperationTypes.DEPLOY, tParameters);
		end
	else
		tParameters[UnitOperationTypes.PARAM_MODIFIERS] = UnitOperationMoveModifiers.NONE;
		if (UnitManager.CanStartOperation( kUnit, UnitOperationTypes.RANGE_ATTACK, nil, tParameters) ) then
			UnitManager.RequestOperation(kUnit, UnitOperationTypes.RANGE_ATTACK, tParameters);
		else
			-- Allow for attacking and don't early out if the destination is blocked, etc., but is in the fog.
			tParameters[UnitOperationTypes.PARAM_MODIFIERS] = UnitOperationMoveModifiers.ATTACK + UnitOperationMoveModifiers.MOVE_IGNORE_UNEXPLORED_DESTINATION;
			if (UnitManager.CanStartOperation( kUnit, UnitOperationTypes.COASTAL_RAID, nil, tParameters) ) then
				UnitManager.RequestOperation( kUnit, UnitOperationTypes.COASTAL_RAID, tParameters);
			else
				-- Check that unit isn't already in the plot (essentially canceling the move),
				-- otherwise the operation will complete, and while no move is made, the next
				-- unit will auto seltect.
				if plotX ~= kUnit:GetX() or plotY ~= kUnit:GetY() then
					if (UnitManager.CanStartOperation( kUnit, UnitOperationTypes.SWAP_UNITS, nil, tParameters) ) then
						UnitManager.RequestOperation(kUnit, UnitOperationTypes.MOVE_TO, tParameters);
					else
						UnitManager.RequestOperation(kUnit, UnitOperationTypes.MOVE_TO, tParameters);
					end
				end
			end
		end
	end
end

-- ===========================================================================
--	Multiplier value 
--	Return a string with a colorized # and a +/- based on 1.0 based percent.
-- ===========================================================================
function GetColorPercentString( multiplier:number )
	if		multiplier > 1 then return "[COLOR:StatGoodCS]+"..tostring((multiplier-1)*100).."%[ENDCOLOR]";
	elseif	multiplier < 1 then return "[COLOR:StatBadCS]-"..tostring((1-multiplier)*100).."%[ENDCOLOR]";
	else					return "[COLOR:StatNormalCS]100%[ENDCOLOR]";
	end
end

function GetFilteredUnitStatString( statData:table )
	if statData == nil then
		UI.DataError("Invalid stat data passed to GetFilteredUnitStatString");
		return "";
	end

	local statString = "";
	local statStringTooltip = "";
	local newlineCounter = 0;
	for _,statTable in pairs(statData) do
		statString = statString.. statTable.FontIcon.. " ".. statTable.Value.. " ";
		if (newlineCounter == 2) then
			statString = statString.. "[NEWLINE]";
			newlineCounter = 0;
		end
		newlineCounter = newlineCounter + 1;

		statStringTooltip = statStringTooltip.. Locale.Lookup(statTable.Label).. " ".. statTable.Value.. "[NEWLINE]";
	end
	--return statString, statStringTooltip;
	return statString;
end

function FilterUnitStats( hashOrType:number, ignoreStatType:number )
	local unitInfo = GameInfo.Units[hashOrType];

	if(unitInfo == nil) then
		UI.DataError("Invalid unit hash passed to FilterUnitStats");
		return {};
	end


	local data:table = {};

	-- Strength
	if ( unitInfo.Combat > 0 and (ignoreStatType == nil or ignoreStatType ~= CombatTypes.MELEE)) then
		table.insert(data, {Value = unitInfo.Combat, Type = "Combat", Label = "LOC_HUD_UNIT_PANEL_STRENGTH",				FontIcon="[ICON_Strength_Large]",		IconName="ICON_STRENGTH"});
	end
	if ( unitInfo.RangedCombat > 0 and (ignoreStatType == nil or ignoreStatType ~= CombatTypes.RANGED)) then
		table.insert(data, {Value = unitInfo.RangedCombat,		Label = "LOC_HUD_UNIT_PANEL_RANGED_STRENGTH",		FontIcon="[ICON_RangedStrength_Large]",	IconName="ICON_RANGED_STRENGTH"});
	end
	if (unitInfo.Bombard > 0 and (ignoreStatType == nil or ignoreStatType ~= CombatTypes.BOMBARD)) then
		table.insert(data, {Value = unitInfo.Bombard,	Label = "LOC_HUD_UNIT_PANEL_BOMBARD_STRENGTH",		FontIcon="[ICON_Bombard_Large]",		IconName="ICON_BOMBARD"});
	end
	if (unitInfo.ReligiousStrength > 0 and (ignoreStatType == nil or ignoreStatType ~= CombatTypes.RELIGIOUS)) then
		table.insert(data, {Value = unitInfo.ReligiousStrength,	Label = "LOC_HUD_UNIT_PANEL_RELIGIOUS_STRENGTH",	FontIcon="[ICON_ReligionStat_Large]",	IconName="ICON_RELIGION"});
	end
	if (unitInfo.AntiAirCombat > 0 and (ignoreStatType == nil or ignoreStatType ~= CombatTypes.AIR)) then
		table.insert(data, {Value = unitInfo.AntiAirCombat,	Label = "LOC_HUD_UNIT_PANEL_ANTI_AIR_STRENGTH",		FontIcon="[ICON_AntiAir_Large]",		IconName="ICON_STATS_ANTIAIR"});
	end

	-- Movement
	if(unitInfo.BaseMoves > 0) then
		table.insert(data, {Value = unitInfo.BaseMoves, Type = "BaseMoves",		Label = "LOC_HUD_UNIT_PANEL_MOVEMENT",				FontIcon="[ICON_Movement_Large]",		IconName="ICON_MOVES"});
	end

	-- Range
	if (unitInfo.Range > 0) then
		table.insert(data, {Value = unitInfo.Range;			Label = "LOC_HUD_UNIT_PANEL_ATTACK_RANGE",			FontIcon="[ICON_Range_Large]",			IconName="ICON_RANGE"});
	end

	-- Charges
	if (unitInfo.SpreadCharges > 0) then
		table.insert(data, {Value = unitInfo.SpreadCharges,	Type = "SpreadCharges", Label = "LOC_HUD_UNIT_PANEL_SPREADS",				FontIcon="[ICON_ReligionStat_Large]",	IconName="ICON_RELIGION"});
	end
	if (unitInfo.BuildCharges > 0) then
		table.insert(data, {Value = unitInfo.BuildCharges, Type = "BuildCharges",		Label = "LOC_HUD_UNIT_PANEL_BUILDS",				FontIcon="[ICON_Charges_Large]",		IconName="ICON_BUILD_CHARGES"});
	end

	-- If we have more than 4 stats then try to remove melee strength
	if (table.count(data) > 4) then
		for i,stat in ipairs(data) do
			if stat.Type == "Combat" then
				table.remove(data, i);
			end
		end
	end

	-- If we still have more than 4 stats through a data error
	if (table.count(data) > 4) then
		UI.DataError("More than four stats were picked to display for unit ".. unitInfo.UnitType);
	end

	return data;
end

-- ===========================================================================
--	Obtains the texture for a city's current production.
--	pCity				The city
--	productionHash		the production hash (present or past) that you want the info for
--
--	RETURNS	NIL if error, otherwise a table containing:
--			name of production item
--			description
--			icon texture of the produced item
--			u offset of the icon texture
--			v offset of the icon texture
--			(0-1) percent complete
--			(0-1) percent complete after next turn
-- ===========================================================================
function GetProductionInfoOfCity( pCity:table, productionHash:number )
	local pBuildQueue	:table = pCity:GetBuildQueue();
	if pBuildQueue == nil then
		UI.DataError("No production queue in city!");
		return nil;
	end
	
	local hash						= productionHash;
	local progress					:number = 0;
	local cost						:number = 0;
	local percentComplete			:number = 0;
	local percentCompleteNextTurn	:number = 0;
	local productionName			:string;
	local description				:string;
	local statString				:string;		-- stats for unit to display
	local iconName					:string;		-- name of icon to look up
	local texture					:string;		-- texture of icon
	local u							:number = 0;	-- texture horiztonal offset
	local v							:number = 0;	-- texture vertical offset

	-- Nothing being produced.
	if hash == 0 then
		return {
			Name					= Locale.Lookup("LOC_HUD_CITY_NOTHING_PRODUCED"),
			Description				= "", 
			Texture					= "CityPanel_CitizenIcon",	-- Default texture
			u						= 0,
			v						= 0,
			PercentComplete			= 0, 
			PercentCompleteNextTurn	= 0,
			Turns					= 0,
			Progress				= 0,
			Cost					= 0
		};
	end

	-- Find the information
	local buildingDef	:table = GameInfo.Buildings[hash];
	local districtDef	:table = GameInfo.Districts[hash];
	local unitDef		:table = GameInfo.Units[hash];
	local projectDef	:table = GameInfo.Projects[hash];
	local type			:string= "";

	if( buildingDef ~= nil ) then
		prodTurnsLeft = pBuildQueue:GetTurnsLeft(buildingDef.BuildingType);
		productionName	= Locale.Lookup(buildingDef.Name);
		description		= buildingDef.Description;
		progress		= pBuildQueue:GetBuildingProgress(buildingDef.Index);
		percentComplete	= progress / pBuildQueue:GetBuildingCost(buildingDef.Index);
		cost			= pBuildQueue:GetBuildingCost(buildingDef.Index);
		iconName		= "ICON_"..buildingDef.BuildingType;
		type			= ProductionType.BUILDING;

	elseif( districtDef ~= nil ) then
		prodTurnsLeft = pBuildQueue:GetTurnsLeft(districtDef.DistrictType);
		productionName	= Locale.Lookup(districtDef.Name);
		description		= districtDef.Description;
		progress		= pBuildQueue:GetDistrictProgress(districtDef.Index);
		percentComplete	= progress / pBuildQueue:GetDistrictCost(districtDef.Index);
		cost			= pBuildQueue:GetDistrictCost(districtDef.Index);
		iconName		= "ICON_"..districtDef.DistrictType;
		type			= ProductionType.DISTRICT;

	elseif( unitDef ~= nil ) then
		prodTurnsLeft = pBuildQueue:GetTurnsLeft(unitDef.UnitType);
		local eMilitaryFormationType :number = pBuildQueue:GetCurrentProductionTypeModifier();
		productionName	= Locale.Lookup(unitDef.Name);
		description		= unitDef.Description;
		progress		= pBuildQueue:GetUnitProgress(unitDef.Index);
		prodTurnsLeft	= pBuildQueue:GetTurnsLeft(unitDef.UnitType, eMilitaryFormationType);		
		iconName		= "ICON_"..unitDef.UnitType.."_PORTRAIT";
		statString		= GetFilteredUnitStatString(FilterUnitStats(hash));
		type			= ProductionType.UNIT;

		--Units need some additional information to represent the Standard, Corps, and Army versions. This is determined by the MilitaryFormationType
		if (eMilitaryFormationType == MilitaryFormationTypes.STANDARD_FORMATION) then
			percentComplete = progress / pBuildQueue:GetUnitCost(unitDef.Index);	
			cost			= pBuildQueue:GetUnitCost(unitDef.Index);
		elseif (eMilitaryFormationType == MilitaryFormationTypes.CORPS_FORMATION) then
			percentComplete = progress / pBuildQueue:GetUnitCorpsCost(unitDef.Index);
			cost			= pBuildQueue:GetUnitCorpsCost(unitDef.Index);
			if (unitDef.Domain == "DOMAIN_SEA") then
				productionName = productionName .. " " .. Locale.Lookup("LOC_UNITFLAG_FLEET_SUFFIX");
			else
				productionName = productionName .. " " .. Locale.Lookup("LOC_UNITFLAG_CORPS_SUFFIX");
			end
		elseif (eMilitaryFormationType == MilitaryFormationTypes.ARMY_FORMATION) then
			percentComplete = progress / pBuildQueue:GetUnitArmyCost(unitDef.Index);
			cost			= pBuildQueue:GetUnitArmyCost(unitDef.Index);
			if (unitDef.Domain == "DOMAIN_SEA") then
				productionName = productionName .. " " .. Locale.Lookup("LOC_UNITFLAG_ARMADA_SUFFIX");
			else
				productionName = productionName .. " " .. Locale.Lookup("LOC_UNITFLAG_ARMY_SUFFIX");
			end
		end

	elseif (projectDef ~= nil) then
		prodTurnsLeft = pBuildQueue:GetTurnsLeft(projectDef.ProjectType);
		productionName	= Locale.Lookup(projectDef.Name);
		description		= projectDef.Description;
		progress		= pBuildQueue:GetProjectProgress(projectDef.Index);
		cost			= pBuildQueue:GetProjectCost(projectDef.Index);
		percentComplete	= progress / pBuildQueue:GetProjectCost(projectDef.Index);
		iconName		= "ICON_"..projectDef.ProjectType;
		type			= ProductionType.PROJECT;
	else
		for row in GameInfo.Types() do
			if row.Hash == hash then
				UI.DataError("Unknown kind of item being produced in city \""..tostring(row.Kind).."\"");
				return nil;
			end
		end
		UI.DataError("Game database does not contain information that matches what the city "..Locale.Lookup(data.CityName).." is producing!");
		return nil;
	end
	if percentComplete > 1 then
		percentComplete = 1;
	end

	percentCompleteNextTurn = (1-percentComplete)/prodTurnsLeft;
	percentCompleteNextTurn = percentComplete + percentCompleteNextTurn;

	return {
		Name					= productionName,
		Description				= description, 
		Type					= type;
		Icon					= iconName,
		PercentComplete			= percentComplete, 
		PercentCompleteNextTurn	= percentCompleteNextTurn,
		Turns					= prodTurnsLeft,
		StatString				= statString;
		Progress				= progress;
		Cost					= cost;		
	};
end

-- ===========================================================================
--	Obtain the stats for a unit, given it's hash or type string
--	RETURNS: nil if not found, or table o' stats
-- ===========================================================================
function GetUnitStats( hashOrType )
	local info:table= GameInfo.Units[hashOrType];
	if info == nil then
		--error("Was unable to find a Unit to get it's stats with the value \""..tostring(hashOrType).."\"");
		return nil;
	end
	return {
		Bombard		= info.Bombard,
		Combat		= info.Combat,
		Moves		= info.BaseMoves,
		RangedCombat= info.RangedCombat,
		Range		= info.Range
	}
end

-- ===========================================================================
--	Returns the icon info and shadow icon info for the passed in unit or returns default icons if those can't be found
--	RETURN 1: iconInfo - table containing textureSheet, textureOffsetX, and textureOffsetY
--	RETURN 2: iconShadowInfo - table containing textureSheetShadow, textureOffsetShadowX, and textureOffsetShadowY
-- ===========================================================================
function GetUnitIconAndIconShadow( pUnit:table, iconSize:number, isIgnoreShadow:boolean )	
	
	if isIgnoreShadow == nil then isIgnoreShadow = false; end -- Default parameter.
	
	local iconInfo		:table = {};
	local iconShadowInfo:table = {};
	if pUnit then
		local unitInfo:table = GameInfo.Units[pUnit:GetUnitType()];
		if unitInfo then
			-- Get icon info
			local iconName:string = "ICON_" .. unitInfo.UnitType .. "_WHITE";
			iconInfo.textureOffsetX, iconInfo.textureOffsetY, iconInfo.textureSheet = IconManager:FindIconAtlas(iconName, iconSize);
			if (iconInfo.textureSheet == nil) then			--Check to see if the unit has an icon atlas index defined
				print("UIWARNING: Could not find icon for " .. iconName);
				iconInfo.textureOffsetX, iconInfo.textureOffsetY, iconInfo.textureSheet = IconManager:FindIconAtlas("ICON_UNIT_UNKNOWN_WHITE", iconSize);		--If not, resolve the index to be a generic unknown index
			end

			-- Get icon shadow info, use explicit check because IconManager will Assert.
			if (not isIgnoreShadow) then
				local iconNameShadow:string = "ICON_" .. unitInfo.UnitType .. "_BLACK";
				iconShadowInfo.textureOffsetShadowX, iconShadowInfo.textureOffsetShadowY, iconShadowInfo.textureSheetShadow = IconManager:FindIconAtlas(iconNameShadow, iconSize);	
				if iconShadowInfo.textureSheetShadow ~= nil then
					iconShadowInfo.textureOffsetX, iconShadowInfo.textureOffsetY, iconShadowInfo.textureSheet = IconManager:FindIconAtlas("ICON_UNIT_UNKNOWN_BLACK", iconSize);
				end
			end			
		end
	end
	return iconInfo, iconShadowInfo;
end

-- ===========================================================================
--	A helper function to size a GridButton to its string contents
--	ARG 1: gridButton (table) - expects a control of type GridButton to be resized
--	ARG 4: minX (number) - the minimum width of the button
--	ARG 5: minY (number) - the minimum height of the button
--	ARG 2: OPTIONAL padding (number) - the amount of padding which should be inserted around the text string
--	ARG 3: OPTIONAL sizeOption (string) - expects V or H - to specifiy if the button should only be sized vertically or horizontally
-- ===========================================================================
function AutoSizeGridButton(gridButton:table,minX: number, minY: number, padding:number, sizeOption:string)
	if (sizeOption == nil) then
		sizeOption = "1";
	end
	if (padding == nil) then
		padding = 0;
	end
	local labelControl =  gridButton:GetTextControl();
	local labelX = labelControl:GetSizeX() + padding*2;
	local labelY = labelControl:GetSizeY() + padding*2;
	if (minX ~= nil) then
		labelX = math.max(minX, labelX);
	end
	if (minY ~= nil) then
		labelY = math.max(minY, labelY);
	end
	if(sizeOption == "V" or sizeOption == "1") then
		gridButton:SetSizeY(labelY);
	end
	if(sizeOption == "H" or sizeOption == "1") then
		gridButton:SetSizeX(labelX);
	end
end

-- ===========================================================================
function GetLeaderUniqueTraits( leaderType:string )

	-- Gather info.
    local base_leader = GameInfo.Leaders[leaderType];
    if(base_leader == nil) then
        return;
    end           

	function AddInheritedLeaders(leaders, leader)
		local inherit = leader.InheritFrom;
        if(inherit ~= nil) then
            local parent = GameInfo.Leaders[inherit];
            if(parent) then
                table.insert(leaders, parent);
                AddInheritedLeaders(leaders, parent);
            end
        end
    end

	local leaders = {};
    table.insert(leaders, base_leader);
	AddInheritedLeaders(leaders, base_leader);

	-- Enumerate final list and index.
	local has_leader = {};
	for i,leader in ipairs(leaders) do
		has_leader[leader.LeaderType] = true;
	end

	-- Unique Abilities
	-- We're considering a unique ability to be a trait which does 
	-- not have a unique unit, building, district, or improvement associated with it.
	-- While we scrub for unique units and infrastructure, mark traits that match 
	-- so we can filter them later.
    local traits = {};
	local has_trait = {};
	local not_ability = {};
    for row in GameInfo.LeaderTraits() do
        if(has_leader[row.LeaderType] == true) then
			local trait = GameInfo.Traits[row.TraitType];
			if(trait) then
				table.insert(traits, trait);			
			end
			has_trait[row.TraitType] = true;
        end
    end

    -- Unique Units
    local uu = {};
    for row in GameInfo.Units() do
        local trait = row.TraitType;
        if(trait) then
			not_ability[trait] = true;
			if(has_trait[trait] == true) then
				local description :string = Locale.Lookup("LOC_LOADING_"..row.Domain);
				if m_isTraitsFullDescriptions then
					description = Locale.Lookup(GameInfo.Units[row.UnitType].Description);
				end
				table.insert(uu, { Type = row.UnitType, Name = row.Name, Description = description });
			end
        end
    end
    
    -- Unique Buildings/Districts/Improvements
    local ub = {};
    for row in GameInfo.Buildings() do
        local trait = row.TraitType;
        if(trait) then
			not_ability[trait] = true;
			if(has_trait[trait] == true) then
				local districtName:string = Locale.Lookup(GameInfo.Districts[row.PrereqDistrict].Name);
				local description :string = Locale.Lookup("LOC_LOADING_DISTRICT_BUILDING", districtName); 
				if m_isTraitsFullDescriptions then
					description = Locale.Lookup(GameInfo.Buildings[row.BuildingType].Description); 
				end
				table.insert(ub, {Type = row.BuildingType, Name = row.Name, Description = description});
			end
        end
    end

    for row in GameInfo.Districts() do
        local trait = row.TraitType;
        if(trait) then
			not_ability[trait] = true;
			if(has_trait[trait] == true) then
				local description :string = Locale.Lookup("LOC_LOADING_UNIQUE_DISTRICT");
				if m_isTraitsFullDescriptions then
					description = Locale.Lookup(GameInfo.Districts[row.DistrictType].Description); 
				end
				table.insert(ub, {Type = row.DistrictType, Name = row.Name, Description = description});
			end
        end
    end

    for row in GameInfo.Improvements() do
        local trait = row.TraitType;
        if(trait) then
			not_ability[trait] = true;
			if(has_trait[trait] == true) then
				local description :string = Locale.Lookup("LOC_LOADING_UNIQUE_IMPROVEMENT");
				if m_isTraitsFullDescriptions then
					description = Locale.Lookup(GameInfo.Improvements[row.ImprovementType].Description); 
				end
				table.insert(ub, {Type = row.ImprovementType, Name = row.Name, Description = description});
			end
        end
    end

	local unique_abilities = {};
	for i, trait in ipairs(traits) do
		print(trait.InternalOnly);
		if(not_ability[trait.TraitType] ~= true and not trait.InternalOnly) then
			table.insert(unique_abilities, trait);
		end
	end

	return unique_abilities,uu,ub;
end


-- ===========================================================================
function GetCivilizationUniqueTraits( civType:string )
    
	local traits = {};
    for row in GameInfo.CivilizationTraits() do
        if(row.CivilizationType == civType) then
            traits[row.TraitType] = true;
        end
    end

	-- Unique Abilities
	-- We're considering a unique ability to be a trait which does 
	-- not have a unique unit, building, district, or improvement associated with it.
	-- While we scrub for unique units and infrastructure, mark traits that match 
	-- so we can filter them later.
	local not_abilities = {};
	
    -- Unique Units
    local uu = {};
    for row in GameInfo.Units() do
        local trait = row.TraitType;
        if(trait) then
			not_abilities[trait] = true;
			if(traits[trait] == true) then
				local description :string = Locale.Lookup("LOC_LOADING_"..row.Domain);
				if m_isTraitsFullDescriptions then
					description = Locale.Lookup(GameInfo.Units[row.UnitType].Description);
				end
				table.insert(uu, { Type = row.UnitType, Name = row.Name, Description = description });
			end
        end
    end
    
    -- Unique Buildings/Districts/Improvements
    local ub = {};
    for row in GameInfo.Buildings() do
        local trait = row.TraitType;
        if(trait) then
			not_abilities[trait] = true;
			if(traits[trait] == true) then
				local building	  :table  = GameInfo.Buildings[row.BuildingType];
				local description :string = Locale.Lookup("LOC_LOADING_UNIQUE_BUILDING");
				if m_isTraitsFullDescriptions then
					if building == nil then
						UI.DataError("Could not get CIV trait as GameInfo.Buildings["..row.BuildingType.."] does not exist.");
					elseif building.Description == nil then
						UI.DataError("Could not get CIV trait description for GameInfo.Buildings["..row.BuildingType.."].  None supplied.");
					else
						description = Locale.Lookup(building.Description); 
					end				
				end
				table.insert(ub, {Type = row.BuildingType, Name = row.Name, Description = description});
			end
        end
    end

    for row in GameInfo.Districts() do
        local trait = row.TraitType;
        if(trait) then
			not_abilities[trait] = true;
			if(traits[trait] == true) then
				local description :string = Locale.Lookup("LOC_LOADING_UNIQUE_DISTRICT");
				if m_isTraitsFullDescriptions then
					description = Locale.Lookup(GameInfo.Districts[row.DistrictType].Description); 
				end
				table.insert(ub, {Type = row.DistrictType, Name = row.Name, Description = description});
			end
        end
    end

    for row in GameInfo.Improvements() do
        local trait = row.TraitType;
        if(trait) then
			not_abilities[trait] = true;
			if(traits[trait] == true) then
				local description :string = Locale.Lookup("LOC_LOADING_UNIQUE_IMPROVEMENT");
				if m_isTraitsFullDescriptions then
					description = Locale.Lookup(GameInfo.Improvements[row.ImprovementType].Description); 
				end
				table.insert(ub, {Type = row.ImprovementType, Name = row.Name, Description = description});
			end
        end
    end

	local unique_abilities = {};
	for row in GameInfo.CivilizationTraits() do
		if(row.CivilizationType == civType and not_abilities[row.TraitType] ~= true) then
			local trait = GameInfo.Traits[row.TraitType];
			if(trait) then
				table.insert(unique_abilities, trait);
			end			
		end
	end

	return unique_abilities, uu, ub;
end
    

-- ===========================================================================
--	Is the on-rails tutorial active?
-- ===========================================================================
function IsTutorialRunning()
	local mods = Modding.GetActiveMods();
	if mods ~= nil then
		for i,v in ipairs(mods) do
			if v.Id == TUTORIAL_UUID then
				return true;
			end
		end
	else
		UI.DataError("Unable to obtain mods table to determine if tutorial is running.");
	end
	return false;
end
