// $Id: //sw/devrel/SDK/TOOLS/src/Photoshop/scripts/NVIDIA_CubeMapShuffler.jsx#2 $
//
// Copyright NVIDIA Corporation 2005
// TO THE MAXIMUM EXTENT PERMITTED BY APPLICABLE LAW, THIS SOFTWARE IS PROVIDED
// *AS IS* AND NVIDIA AND ITS SUPPLIERS DISCLAIM ALL WARRANTIES, EITHER EXPRESS
// OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY
// AND FITNESS FOR A PARTICULAR PURPOSE.  IN NO EVENT SHALL NVIDIA OR ITS SUPPLIERS
// BE LIABLE FOR ANY SPECIAL, INCIDENTAL, INDIRECT, OR CONSEQUENTIAL DAMAGES
// WHATSOEVER (INCLUDING, WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
// BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR ANY OTHER PECUNIARY LOSS)
// ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE, EVEN IF NVIDIA HAS
// BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
//
// Description:
//	This script works with Adobe Photoshop CS and Photoshop CS2, or with
//		Photoshop 7.0 if you are using the add-on scripting option.
//	The script runs on both Windows or Mac-OS versions of Photoshop.
//	The script should work with all color spaces and pixel depths
//	To use, have an image open and run the script from Photoshop's "File->Automate" menu.
//	It looks at the currently-open document, determines if that document is a cube map and
//		in what layout format, then remaps the contents to a new image in a
//		different cube map format. Great for mixing different
//		sorts of assets and cubemap-related tools that require different
//		formats (such as the NVIDIA and Microsoft DDS importers, or third-party
//		imaging programs like HDRShop). If you've ever tried to do this by hand,
//		you know what a chore un-automated conversion can be.
//	The script will:
//		Copy 6x1 (vertical) and 1x6 (horizontal) strips into Inverse-T
//			("inverse cross") maps
//		Copy Inverse-T maps into 6x1 (horizontal) strips
//		Copy single square maps into 6x1 strips (by duplicating)
//	Newly-created documents will have all the cube faces split into layers
//		with names like "X+" and "Y-" to make editing of individual faces easier.
//	This script is non-destructive -- the original document is unaltered.
//
// Installation:
//	Place this script in your Photoshop Presets\Scripts directory and re-start Photoshop.
//	The script will then appear in your "File->Automate->Scripts..." menu items.
//	For more info on Photoshop scripting, see
//		http://partners.Adobe.com/public/developer/photoshop/devcenter.html
//

var gSquare = new UnitValue(256,"px");
var gZero = new UnitValue(0,"px");

function main_body()
{
    if (!app.documents.length > 0) {    // stop if no document is opened.
	alert("Sorry, No Current Document");
    } else {
	var strtRulerUnits = app.preferences.rulerUnits;
	if (strtRulerUnits != Units.PIXELS) {
	    app.preferences.rulerUnits = Units.PIXELS; // selections are always in pixels
	}
	var origDoc = app.activeDocument;
	var w = origDoc.width;
	var h = origDoc.height;
	var bits = origDoc.bitsPerChannel;
	var layerName = new Array("X+","X-","Y+","Y-","Z+","Z-");
	var orgCorn;
	var orgRot;
	var newCorn;
	var newDoc;
	if (w/3 == (h/4)) {
	    gSquare = w/3;
	    orgCorn = inv_t_corners();
	    orgRot = new Array(-90,-270,-90,-270,0,180);
	    newCorn = horizontal_corners();
	    newDoc = app.documents.add(gSquare*6,gSquare,150,"cubeStrip",NewDocumentMode.RGB,DocumentFill.TRANSPARENT);
	} else if (w == (h*6)) {
	    gSquare = h;
	    newCorn = inv_t_corners();
	    orgRot = new Array(90,270,90,270,0,180);
	    orgCorn = horizontal_corners();
	    newDoc = app.documents.add(gSquare*3,gSquare*4,150,"InverseT",NewDocumentMode.RGB,DocumentFill.TRANSPARENT);
	} else if (h == (w*6)) {
	    gSquare = w;
	    newCorn = inv_t_corners();
	    orgRot = new Array(90,270,90,270,0,180);
	    orgCorn = vertical_corners();
	    newDoc = app.documents.add(gSquare*3,gSquare*4,150,"InverseT",NewDocumentMode.RGB,DocumentFill.TRANSPARENT);
	} else if (h == w) {
	    gSquare = w;
	    newCorn = horizontal_corners();
	    orgRot = new Array(0,0,0,0,0,0);
	    orgCorn = square_corners();
	    newDoc = app.documents.add(gSquare*6,gSquare,150,"cubeStrip",NewDocumentMode.RGB,DocumentFill.TRANSPARENT);
	} else {
	    alert("Sorry: Based on the aspect ratio,\n'"+origDoc.name+"'\ndoesn't look like a cube map");
	    return;
	}
	if (bits == BitsPerChannelType.ONE) { // always upgrade bit maps to 8-bit RGB
	    // alert("Fixing new doc bits to "+BitsPerChannelType.EIGHT);
	    newDoc.bitsPerChannel = BitsPerChannelType.EIGHT;
	} else {
	    // alert("Setting new doc bits to "+bits);
	    newDoc.bitsPerChannel = bits;
	}
	// alert("gSquare is "+gSquare);
	newDoc.info.source = "Generated by NVCubeMapShuffler from '" + origDoc.name + "'";
	newDoc.info.category = "cubemap";
	// duplicate and flatten-mode-adjust as needed
	var tempDoc;
	if (origDoc.artLayers.length > 1) {
	    activeDocument = origDoc;
	    tempDoc = origDoc.duplicate();
	} else {
	    tempDoc = origDoc;
	}
	activeDocument = tempDoc;
	if (tempDoc.mode != DocumentMode.RGB) {
	    tempDoc.changeMode(ChangeMode.RGB);
	}
	tempDoc.flatten();
	// copy panels from 'tempDoc' to 'newDoc'
	var i;
	for (i=0; i<6; i++) {
	    o2n(tempDoc,newDoc,orgCorn[i],newCorn[i],orgRot[i],layerName[i]);
	}
	// if 'tempDoc' was a copy, cose it without saving
	if (origDoc.artLayers.length > 1) {
	    tempDoc.close(SaveOptions.DONOTSAVECHANGES);
	}
	if (strtRulerUnits != Units.PIXELS) {
	    app.preferences.rulerUnits = strtRulerUnits;
	}
    }
}

main_body();

function inv_t_corners()
{
    var pyCorner = new Array(gSquare,gZero);
    var pzCorner = step_down(pyCorner,gSquare);
    var nyCorner = step_down(pzCorner,gSquare);
    var nzCorner = step_down(nyCorner,gSquare);
    var pxCorner = step_right(nyCorner,gSquare);
    var nxCorner = step_right(nyCorner,-gSquare);
    return new Array(pxCorner,nxCorner,pyCorner,nyCorner,pzCorner,nzCorner);
}

function horizontal_corners()
{
    var pzCorner = new Array(gZero,gZero);
    var nzCorner = step_right(pzCorner,gSquare);
    var pyCorner = step_right(nzCorner,gSquare);
    var nyCorner = step_right(pyCorner,gSquare);
    var nxCorner = step_right(nyCorner,gSquare);
    var pxCorner = step_right(nxCorner,gSquare);
    return new Array(pxCorner,nxCorner,pyCorner,nyCorner,pzCorner,nzCorner);
}

function vertical_corners()
{
    var pzCorner = new Array(gZero,gZero);
    var nzCorner = step_down(pzCorner,gSquare);
    var pyCorner = step_down(nzCorner,gSquare);
    var nyCorner = step_down(pyCorner,gSquare);
    var nxCorner = step_down(nyCorner,gSquare);
    var pxCorner = step_down(nxCorner,gSquare);
    return new Array(pxCorner,nxCorner,pyCorner,nyCorner,pzCorner,nzCorner);
}

function square_corners()
{
    var origin = new Array(gZero,gZero);
    return new Array(origin,origin,origin,origin,origin,origin);
}

function step_right(corner,size) { return new Array(corner[0]+size,corner[1]); }

function step_down(corner,size) { return new Array(corner[0],corner[1]+size); }

function o2n(oDoc,nDoc,oCorner,nCorner,angle,newName)
{
    var s = gSquare; // +1;
    var oX  = new Array(oCorner[0]+s, oCorner[1]);
    var oY  = new Array(oCorner[0],   oCorner[1]+s);
    var oXY = new Array(oCorner[0]+s, oCorner[1]+s);
    var nX  = new Array(nCorner[0]+s, nCorner[1]);
    var nY  = new Array(nCorner[0],   nCorner[1]+s);
    var nXY = new Array(nCorner[0]+s, nCorner[1]+s);
    var oBound = new Array(oCorner,oX,oXY,oY);
    var nBound = new Array(nCorner,nX,nXY,nY);
    activeDocument = oDoc;
    oDoc.selection.select(oBound);
    // alert(newName+"\norig bound set to ["+oBound+"],\n new bound to ["+nBound+"]")
    oDoc.selection.copy();
    oDoc.selection.deselect();
    activeDocument = nDoc;
    nDoc.selection.select(nBound);
    var newLayer = nDoc.paste();
    if (nDoc.bitsPerChannel == BitsPerChannelType.THIRTYTWO) {
	nDoc.selection.rotate(angle);
    } else {
	if (angle != 0) {
	    newLayer.rotate(angle);
	}
	newLayer.name = newName;
    }
    nDoc.selection.deselect();
}

// eof

