//=============================================================================
// Balloon_Message.js
//=============================================================================

/*:
 * @plugindesc 吹き出しプラグインです
 * @author liply
 *
 * @param Current X
 * @desc 中央に表示される吹き出しのX座標です
 * @default 416
 *
 * @param Current Y
 * @desc 中央に表示される吹き出しのY座標です
 * @default 312
 *
 * @param Log X
 * @desc 上に流れた吹き出しのX座標です
 * @default 416
 *
 * @param Log Y
 * @desc 上に流れた吹き出しのY座標です
 * @default 100
 *
 * @param Move Frame
 * @desc 移動にかかる時間です(30 = 1秒)
 * @default 30
 *
 * @param Left X
 * @desc 左から出現する場合のX座標です
 * @default 680
 *
 * @param Right X
 * @desc 右から出現する場合のX座標です
 * @default 150
 *
 * @param Shake Frame
 * @desc 振動する時間です(30 = 1秒)
 * @default 30
 *
 * @param Fade Frame
 * @desc フェードエフェクトの持続時間です(30 = 1秒)
 * @default 30
 *
 * @param Font Size
 * @desc フォントの大きさです
 * @default 20
 *
 * @param Text Margin
 * @desc 行間の大きさです
 * @default 5
 *
 * @param Text X
 * @desc テキストのX座標です。1行、2行、3行とカンマで区切ります
 * @default 0, 0, 0
 *
 * @param Text Y
 * @desc テキストのY座標です。1行、2行、3行とカンマで区切ります
 * @default -12, -12, -12
 *
 * @param Avatar Left X
 * @desc 左立ち絵のX座標です
 * @default 120
 *
 * @param Avatar Right X
 * @desc 右立ち絵のX座標です
 * @default 620
 *
 *
 * @help
 * プラグインコマンド：
 *   Avatar 番号 ピクチャ名
 *     1: 左から出現, 2: 右から出現, 3: 左の画像変更, 4: 右の画像変更
 *
 * 吹き出しコマンド:
 * メッセージ先頭に記述お願いします
 *   <吹き出し番号 吹き出しエフェクト X Y>
 *   吹き出し番号:
 *     1: balloon-normal-left-[1,2,3]
 *     2: balloon-normal-right-[1,2,3]
 *     3: balloon-shout-left-[1,2,3]
 *     4: balloon-shout-right-[1,2,3]
 *     5: balloon-think-left-[1,2,3]
 *     6: balloon-think-left-[1,2,3]
 *   吹き出しエフェクト:
 *     1: 左から
 *     2: 右から
 *     3: シェイク
 *     4: ふわっと
 */



(function(){
    var parameters = PluginManager.parameters('Balloon_Message');
    function p(name, def){
        return parseInt(parameters[name]) || def;
    }
    function toIntArray(arr){
        var result = [];
        for(var n = 0; n < arr.length; ++n){
            result[n] = parseInt(arr[n]);
        }
        return result;
    }

    var balloon_CurrentX = p('Current X', 416);
    var balloon_CurrentY = p('Current Y', 312);

    var balloon_LogX = p('Log X', 416);
    var balloon_LogY = p('Log Y', 100);

    var balloon_MoveFrame = p('Move Frame', 30);

    var balloon_LeftX = p('Left X', 600);
    var balloon_RightX = p('Right X', 150);

    var balloon_ShakeFrame = p('Shake Frame', 30);

    var balloon_FadeFrame = p('Fade Frame', 30);

    var balloon_FontSize = p('Font Size', 20);
    var balloon_TextMargin = p('Text Margin', 5);

    var balloon_TextX = toIntArray((parameters['Text X'] || '0, 0, 0').split(','));
    var balloon_TextY = toIntArray((parameters['Text Y'] || '-12, -12, -12').split(','));

    var avatar_LeftX = p('Avatar Left X', 120);
    var avatar_RightX = p('Avatar Right X', 620);


    function createBalloon(opt){
        if(!opt)return null;
        var lineNumber = opt.message.length;

        return {
            x: opt.x || 0,
            y: opt.y || 0,
            dx: opt.dx || 0,
            dy: opt.dy || 0,
            tx: opt.tx || balloon_TextX[lineNumber - 1],
            ty: opt.ty || balloon_TextY[lineNumber - 1],
            textWidth: opt.textWidth || 500,
            type: opt.type || 'normal',
            effectType: opt.effectType || 'left',
            message: opt.message || [''],
            lineNumber: lineNumber,
            opacity: opt.opacity || 255
        };
    }

    function fetchBitmap(balloon){
        return ImageManager.loadPicture('balloon-' + balloon.type + '-' + balloon.lineNumber);
    }

    function Command_Position(x, y, time){
        var ix, iy, t;

        return function(balloon){
            if(t === undefined){
                ix = balloon.x;
                iy = balloon.y;
                t = 0;

                if(time === undefined){
                    balloon.x = x;
                    balloon.y = y;
                    return false;
                }
            }

            if(t++ < time){
                var alpha = t / time;
                balloon.x = x * alpha + ix * (1 - alpha);
                balloon.y = y * alpha + iy * (1 - alpha);
                return "position";
            }else{
                balloon.x = x;
                balloon.y = y;
                return false;
            }
        }
    }

    function Command_Position_Opacity(x, y, opacity, time){
        var ix, iy, io, t;

        return function(balloon){
            if(t === undefined){
                ix = balloon.x;
                iy = balloon.y;
                io = balloon.opacity;
                t = 0;

                if(time === undefined){
                    balloon.x = x;
                    balloon.y = y;
                    balloon.opacity = opacity;
                    return false;
                }
            }

            if(t++ < time){
                var alpha = t / time;
                balloon.x = x * alpha + ix * (1 - alpha);
                balloon.y = y * alpha + iy * (1 - alpha);
                balloon.opacity = opacity * alpha + io * (1 - alpha);
                return "position_opacity";
            }else{
                balloon.x = x;
                balloon.y = y;
                balloon.opacity = opacity;
                return false;
            }
        }
    }

    function Command_Opacity(opacity, time){
        var io, t;

        return function(balloon){
            if(t === undefined){
                io = balloon.opacity;
                t = 0;

                if(time === undefined){
                    balloon.opacity = opacity;
                    return false;
                }
            }

            if(t++ < time){
                var alpha = t / time;
                balloon.opacity = opacity * alpha + io * (1 - alpha);
                return "opacity";
            }else{
                return false;
            }
        }
    }

    function Command_Shake(time){
        var ix, iy, t;

        return function(balloon){
            if(t === undefined){
                ix = balloon.x;
                iy = balloon.y;
                t = 0;
            }

            if(t++ < time){
                var alpha = t / time;
                balloon.x = ix + Math.random() * 20;
                balloon.y = iy + Math.random() * 20;
                return "shake";
            }else{
                balloon.x = ix;
                balloon.y = iy;
                return false;
            }
        }
    }

    function isCommandNext(){
      return Input.isTriggered('ok') || Input.isRepeated('ok') || TouchInput.isTriggered();
    }

    function Command_Wait(){
        return function(balloon){
            if(isCommandNext()){
                return false;
            }else{
                return "wait";
            }
        }
    }

    function Command_Log(){
        return function(balloon, opts){
            if(isCommandNext()){
                if(opts && opts.log)opts.log();
                return false;
            }else{
                return "log";
            }
        }
    }

    function processCommand(balloon, commands, opts){
        var result;
        while(commands.length !== 0){
            result = commands[0](balloon, opts);
            if(result)return result;
            commands.shift();
        }

        return false;
    }

    function createCommandBuilder(){
        var commands = [];

        return {
            position: function(x, y, time){
                commands.push(Command_Position(x, y, time));
            },
            position_opacity: function(x, y, opacity, time){
                commands.push(Command_Position_Opacity(x, y, opacity, time));
            },
            shake: function(time){
                commands.push(Command_Shake(time));
            },
            opacity:function(opacity, time){
                commands.push(Command_Opacity(opacity, time));
            },
            wait: function(){
                commands.push(Command_Wait());
            },
            log: function(){
                commands.push(Command_Log());
            },
            get: function(){
                return commands;
            }
        }
    }

    function createSequence($, balloon){
        switch(balloon.effectType){
        case 'left':
            $.position(balloon_LeftX, balloon_CurrentY);
            $.position(balloon_CurrentX, balloon_CurrentY, balloon_MoveFrame);
            break;

        case 'right':
            $.position(balloon_RightX, balloon_CurrentY);
            $.position(balloon_CurrentX, balloon_CurrentY, balloon_MoveFrame);
            break;

        case 'shake':
            $.position(balloon_CurrentX, balloon_CurrentY);
            $.shake(balloon_ShakeFrame);
            break;

        case 'fade':
            $.opacity(0);
            $.position(balloon_CurrentX, balloon_CurrentY);
            $.opacity(255, balloon_FadeFrame);
            break;
        }

        $.log();
        $.position_opacity(balloon_LogX, balloon_LogY, 0, balloon_MoveFrame);
        $.wait();
        // $.position(balloon_LogX, balloon_LogY, balloon_MoveFrame);
        // $.wait();
        // $.position(balloon_LogX, balloon_LogY - (balloon_CurrentY - balloon_LogY)*2, balloon_MoveFrame);
    }

    Spriteset_Base_prototype_createUpperLayer = Spriteset_Base.prototype.createUpperLayer;
    Spriteset_Base.prototype.createUpperLayer = function() {
        Spriteset_Base_prototype_createUpperLayer.call(this);

        initBalloonSprites(this);
    };

    function initBalloonSprites(sprites){
        sprites._balloons = [
            createBalloonSprite(sprites),
            createBalloonSprite(sprites),
            createBalloonSprite(sprites)
        ];

        sprites._avatars = [
            new Sprite(),
            new Sprite()
        ];

        sprites._avatars[0].anchor.x = sprites._avatars[1].anchor.x = 0.5;
        sprites._avatars[0].anchor.y = sprites._avatars[1].anchor.y = 1;

        sprites.addChild(sprites._balloons[2]);
        sprites.addChild(sprites._balloons[1]);
        sprites.addChild(sprites._avatars[0]);
        sprites.addChild(sprites._avatars[1]);
        sprites.addChild(sprites._balloons[0]);
    }

    function createBalloonSprite(sprites){
        var base = new Sprite();
        base.balloon = new Sprite();
        base.text = new Sprite();
        base.text.bitmap = new Bitmap(500, 500);

        base.addChild(base.balloon);
        base.addChild(base.text);

        return base;
    }


    Spriteset_Base_prototype_update = Spriteset_Base.prototype.update;
    Spriteset_Base.prototype.update = function() {
        Spriteset_Base_prototype_update.call(this);

        updateBalloons(this);
        updateAvatars(this);
    }

    function updateAvatars(sprites){
        var avatars = $gameMessage._avatars;

        for(var n = 0; n < avatars.length; ++n){
            var avatar = avatars[n];

            processCommand(avatar.data, avatar.commands);
        }

        updateAvatarSprites(sprites, avatars);
    }

    function updateAvatarSprites(sprites, avatars){
        var leftSprite = sprites._avatars[0];
        var rightSprite = sprites._avatars[1];

        if(avatars[0]){
            syncAvatarData(avatars[0], avatar_LeftX, leftSprite);
        }else{
            leftSprite.bitmap = null;
        }

        if(avatars[1]){
            syncAvatarData(avatars[1], avatar_RightX, rightSprite);
        }else{
            rightSprite.bitmap = null;
        }
    }

    function syncAvatarData(avatar, x, sprite){
        sprite.bitmap = ImageManager.loadPicture(avatar.name);
        sprite.x = avatar.data.x + x;
        sprite.y = avatar.data.y + 624;
        sprite.opacity = avatar.data.opacity;
    }

    function updateBalloons(sprites){
        var balloons = $gameMessage._balloons;
        for(var n = 0; n < balloons.length; ++n){
            var balloon = balloons[n];

            processCommand(balloon.data, balloon.commands, {
                log: function(){
                    balloons.next = true;
                }
            });
        }

        removeFinishedBalloons(balloons);
        updateBalloonSprites(sprites._balloons, balloons);
    }

    function removeFinishedBalloons(balloons){
        for(var n = 0; n < balloons.length; ++n){
            if(balloons[n].commands.length !== 0){
                balloons.splice(0, n);
                return;
            }
        }
        balloons.splice(0);
    }

    function updateBalloonSprites(balloonSprites, balloons){
        for(var n = 0; n < balloons.length && n < balloonSprites.length; ++n){
            updateBalloonSprite(balloonSprites[n], balloons[n].data);
        }
        for(var m = n; m < balloonSprites.length; ++m){
            clearBalloonSprite(balloonSprites[m]);
        }
    }

    function updateBalloonSprite(sprite, balloon){
        sprite.visible = true;
        var bitmap = fetchBitmap(balloon);
        sprite.x = balloon.x + balloon.dx - bitmap.width / 2;
        sprite.y = balloon.y + balloon.dy - bitmap.height / 2;

        var textBitmap = sprite.text.bitmap;
        sprite.text.x = -textBitmap.width / 2 + bitmap.width / 2 + balloon.tx;
        sprite.text.y = -textBitmap.height / 2 + bitmap.height / 2 + balloon.ty;

        sprite.opacity = balloon.opacity;

        sprite.balloon.bitmap = bitmap;
        if(sprite.textBody !== balloon.message){
            sprite.textBody = balloon.message;

            sprite.text.bitmap.clear();
            sprite.text.bitmap.fontSize = balloon_FontSize;
            for(var n = 0; n < balloon.message.length; ++n){
                sprite.text.bitmap.drawText(
                    balloon.message[n],
                    0, (balloon_FontSize + balloon_TextMargin) * n,
                    balloon.textWidth,
                    500,
                    'center'
                );
            }
        }
    }

    function clearBalloonSprite(sprite){
        sprite.visible = false;
    }


    var Game_Interpreter_prototype_command101 = Game_Interpreter.prototype.command101;
    Game_Interpreter.prototype.command101 = function(){
        if(this.nextEventCode() === 401 && isBalloon(nextCommand(this).parameters[0])){
            var texts = [];
            while (this.nextEventCode() === 401) {  // Text data
                this._index++;
                texts.push(this.currentCommand().parameters[0]);
            }

            addBalloon(texts);
            this.setWaitMode('balloon-message');
            return false;
        }else{
            return Game_Interpreter_prototype_command101.call(this);
        }
    }

    function nextCommand(interpreter){
        var command = interpreter._list[interpreter._index + 1];
        if (command) {
            return command;
        } else {
            return 0;
        }
    }

    function isBalloon(text){
        return /<[0-9 ]*>/.test(text);
    }


    //<type effect [x y]>
    function createBalloonData(texts){
        if(texts[0]){
            var command = texts[0];
            var matches = command.match(/<([0-9 ]*)>/);
            var opt = {};
            if(matches.length === 2){
                var args = matches[1].split(" ");
                var balloonType = ['normal', 'shout', 'think', 'type4', 'type5', 'type6']
                opt.type =
                    balloonType[Math.floor((parseInt(args[0])-1)/2)] + '-' + ['left', 'right'][(parseInt(args[0])+1)%2];
                opt.effectType = ['left', 'right', 'shake', 'fade'][parseInt(args[1])-1];

                texts.shift();
                opt.message = texts;

                if(args.length === 4){
                    opt.dx = parseInt(args[2]);
                    opt.dy = parseInt(args[3]);
                }

                return createBalloon(opt);
            }
        }

        return null;
    }


    function addBalloon(texts){
        var data = createBalloonData(texts);

        var builder = createCommandBuilder();
        var command = createSequence(
            builder,
            data
        );

        $gameMessage._balloons.unshift({data: data, commands: builder.get()});
        $gameMessage._balloons.next = false;
    }

    function clearBalloon(){
        $gameMessage._balloons.splice(0);
    }

    Game_Interpreter_prototype_updateWaitMode = Game_Interpreter.prototype.updateWaitMode;
    Game_Interpreter.prototype.updateWaitMode = function(){
        if(this._waitMode === 'balloon-message'){
            return isBalloonBusy();
        }else{
            return Game_Interpreter_prototype_updateWaitMode.call(this);
        }
    }

    function isBalloonBusy(interpreter){
        if(!ImageManager.isReady()) return true;
        if(!$gameMessage._balloons.next) return true;
    }

    Game_Interpreter_prototype_terminate = Game_Interpreter.prototype.terminate;
    Game_Interpreter.prototype.terminate = function() {
        Game_Interpreter_prototype_terminate.call(this);
        $gameMessage._balloons.splice(0);
        $gameMessage._avatars.splice(0);
    }


    Game_Message_prototype_initialize = Game_Message.prototype.initialize;
    Game_Message.prototype.initialize = function() {
        Game_Message_prototype_initialize.call(this);

        this._balloons = [];
        this._avatars = [];
    }
    var Game_Interpreter_prototype_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        Game_Interpreter_prototype_pluginCommand.call(this, command, args);

        switch(command){
        case 'Avatar':
            createAvatarCommand(args);
            break;

        case 'ClearBalloon':
            clearBalloon();
        }
    };

    function createAvatarCommand(args){
        var position = parseInt(args[0]);
        var before = $gameMessage._avatars[1 - position%2] || {};
        var pictureName = args[1];
        var x = args[3] || 0;
        var y = args[4] || 0;

        var $ = createCommandBuilder();
        switch(position){
        case 1:
            $.opacity(255);
            $.position(-50+x, y);
            $.position(x, y, 15);
            break;
        case 2:
            $.opacity(255);
            $.position(50+x, y);
            $.position(x, y, 15);
            break;

        case 3:
        case 4:
            $.opacity(0);
            $.opacity(255, 10);
            break;

        case 5:
            $.opacity(0, 15);
            break;

        case 6:
            $.opacity(0,15);
            break;
        }

        $gameMessage._avatars[1 - position%2] = {
            data: {
                x: x,
                y: y,
                opacity: before.opacity || 255
            },
            commands: $.get(),
            name: pictureName || before.name
        };
    }

})();
