﻿// ----------------------------------------------------------------------
// These are basic usings. Always let them be here.
// ----------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;

// ----------------------------------------------------------------------
// These are RimWorld-specific usings. Activate/Deactivate what you need:
// ----------------------------------------------------------------------
using UnityEngine;         // Always needed
//using VerseBase;         // Material/Graphics handling functions are found here
using Verse;               // RimWorld universal objects are here (like 'Building')
using Verse.AI;          // Needed when you do something with the AI
using Verse.AI.Group;
using Verse.Sound;       // Needed when you do something with Sound
using Verse.Noise;       // Needed when you do something with Noises
using RimWorld;            // RimWorld specific functions are found here (like 'Building_Battery')
using RimWorld.Planet;   // RimWorld specific functions for world creation
using System.IO;
using System.Text.RegularExpressions;
//using RimWorld.SquadAI;  // RimWorld specific functions for squad brains 

namespace CultOfCthulhu
{
    public static class CultLevel
    {
        public const float PureAntiCultist = 0.1f;
        public const float AntiCultist = 0.3f;
        public const float Middling = 0.5f;
        public const float Cultist = 0.7f;
        public const float PureCultist = 0.9f;
    }
    public class CultUtility
    {
        public static List<TraitDef> immoralistTraits = new List<TraitDef>()
        {
            TraitDefOf.Psychopath,
            TraitDefOf.Bloodlust,
            TraitDefOf.Cannibal,
        };
        //TraitDef.Named("Masochist")
        //TraitDef.Named("PsychicSensitivity"),
        //TraitDef.Named("Nerves"),
        //TraitDefOf.DrugDesire
        public enum CultistType
        {
            None,
            Preacher,
            DarkEmmisary
        }

        #region Pawn
        public static float GetBaseCultistModifier(Pawn pawn)
        {
            float result = 0;
            float bigMod = Rand.Range(0.2f, 0.25f);
            float smallMod = Rand.Range(0.05f, 0.1f);
            if (pawn == null) return result;
            if (pawn.story == null) return result;
            if (pawn.story.adulthood == null) return result;
            if (pawn.story.childhood == null) return result;
            string adultStory = pawn.story.adulthood.FullDescriptionFor(pawn);
            string childStory = pawn.story.childhood.FullDescriptionFor(pawn);
            
            //Immoralist modifiers

            //Immoral Traits:
            //         I do eat human flesh.
            //         I like to kill.
            //         I don't care about others.
            foreach (Trait trait in pawn.story.traits.allTraits)
            {
                foreach (TraitDef def in immoralistTraits)
                {
                    if (trait.def == def) result += bigMod;
                }
            }
            //Cult inclined.
            //          Midworlders are more open to superstition.
            //          Abandoned children, looking for 'family.'

            if (adultStory.Contains("midworld") || adultStory.Contains("Midworld")) result += smallMod;
            if (childStory.Contains("midworld") || childStory.Contains("Midworld")) result += smallMod;
            if (childStory.Contains("abandoned")) result += smallMod;
            
            //Moralist modifiers

            //Moral: I am not a violent person.
            if (pawn.story.adulthood.workDisables == WorkTags.Violent ||
                pawn.story.childhood.workDisables == WorkTags.Violent)
            {
                result -= bigMod;
            }
            //Cult disinclined.
            //          Glitterworlders. Morality is paramount.
            if (adultStory.Contains("glitterworld") || adultStory.Contains("Glitterworld")) result -= smallMod;
            if (childStory.Contains("glitterworld") || childStory.Contains("Glitterworld")) result -= smallMod;
            
            //Randomness
            //          Evangelists can be cultist or moralists.
            if (pawn.story.adulthood.Title.Contains("Evangelist"))
            {
                if (Rand.Range(0,100) > 50)
                {
                    result += bigMod;
                }
                result -= bigMod;
            }
            if (Rand.Range(0,100) > 50)
            {
                result += smallMod;
            }
            else
            {
                result -= smallMod;
            }

            return Mathf.Clamp(result, -0.5f, 0.2f);
            
        }
        public static bool TrySpawnWalkInCultist(Map map, CultistType type = CultistType.None, bool showMessage = true)
        {
            IntVec3 loc;
            if (!CellFinder.TryFindRandomEdgeCellWith((IntVec3 c) => map.reachability.CanReachColony(c), map, out loc))
            {
                return false;
            }
            Pawn p = null;
            if (!TryGenerateCultist(out p, map, type))
            {
                Log.Message("Unable to generate cultist");
                return false;
            }
            if (p == null)
            {
                Log.Message("Pawn is null");
                return false;
            }
            GenSpawn.Spawn(p, loc, map);
            string text = "CultistJoin".Translate(new object[]
            {
                p.kindDef.label,
                p.story.adulthood.Title.ToLower()
            });
            text = text.AdjustedFor(p);
            string label = "LetterLabelCultistJoin".Translate();
            if (showMessage) Find.LetterStack.ReceiveLetter(label, text, LetterType.Good);
            PawnRelationUtility.TryAppendRelationsWithColonistsInfo(ref text, ref label, p);
            return true;
        }
        public static bool TryGenerateCultist(out Pawn cultist, Map map, CultistType type = CultistType.None)
        {
            PawnKindDef pawnKindDef = new List<PawnKindDef>
            {
                PawnKindDefOf.Villager
            }.RandomElement<PawnKindDef>();


            Pawn p = null;
            PawnGenerationRequest request;

            //Resolve the type of cultist.
            //If it's a preacher, we need a high speaking skill.
            if (type == CultistType.Preacher)
            {
                for (int i = 0; i < 999; i++)
                {
                    p = null;
                    request = new PawnGenerationRequest(pawnKindDef, Faction.OfPlayer, PawnGenerationContext.NonPlayer, map, false, false, false, false, true, true, 20f, false, true, true, null, null, null, null, null, null);
                    p = PawnGenerator.GeneratePawn(request);

                    if (p.skills.GetSkill(SkillDefOf.Social).TotallyDisabled) continue;
                    if (p.skills.GetSkill(SkillDefOf.Social).Level >= 5)
                    {
                        foreach (WorkTags tag in p.story.DisabledWorkTags)
                        {
                            if (tag == WorkTags.Social) continue;
                        }
                        break;
                    }
                }
            }
            //If it's a dark emissary of Nyarlathotep, we need to add clothing.
            else if (type == CultistType.DarkEmmisary)
            {
                request = new PawnGenerationRequest(pawnKindDef, Faction.OfPlayer, PawnGenerationContext.NonPlayer, map, false, false, false, false, true, true, 20f, false, true, true, null, null, null, null, null, null);
                p = PawnGenerator.GeneratePawn(request);
                Thing tHood = ThingMaker.MakeThing(ThingDef.Named("Apparel_NyarlathotepHood"), ThingDef.Named("DevilstrandCloth"));
                Thing tRobe = ThingMaker.MakeThing(ThingDef.Named("Apparel_CultistRobes"), ThingDef.Named("DevilstrandCloth"));
                Apparel Hood = tHood as Apparel;
                Apparel Robe = tRobe as Apparel;
                p.apparel.Wear(Hood, false);
                p.apparel.Wear(Robe,false);
            }
            else
            {
                request = new PawnGenerationRequest(pawnKindDef, Faction.OfPlayer, PawnGenerationContext.NonPlayer, map, false, false, false, false, true, true, 20f, false, true, true, null, null, null, null, null, null);
                p = PawnGenerator.GeneratePawn(request);
            }
            //We need psychopathic cannibals
            //GenSpawn.Spawn(p, loc);
            if (p == null) { cultist = p; return false; }

            //Add cultist traits.
            TraitDef traitToAdd = TraitDefOf.Psychopath;
            if (!p.story.traits.HasTrait(TraitDefOf.Cannibal)) traitToAdd = TraitDefOf.Cannibal;
            if (!p.story.traits.HasTrait(TraitDefOf.Psychopath)) traitToAdd = TraitDefOf.Psychopath;
            if (p.story.traits.allTraits.Count < 3) p.story.traits.GainTrait(new Trait(traitToAdd));
            else
            {
                foreach (Trait t in p.story.traits.allTraits)
                {
                    if (t.def != TraitDefOf.Cannibal && t.def != TraitDefOf.Psychopath)
                    {
                        p.story.traits.allTraits.Remove(t);
                        break; //Remove 1 trait and get out
                    }
                }
                p.story.traits.GainTrait(new Trait(traitToAdd));
            }

            //Add cult-mindedness.
            AffectCultMindedness(p, 0.8f);

            cultist = p;
            return true;
        }
        #endregion Pawn

        public static int remainingDuration = 1100; // 15 second timer
        public static int ritualDuration = 1100; // 15 seconds max
        public static int reflectDuration = 600; // 10 seconds max

        #region GetResults
        public enum SacrificeResult
        {
            none = 0,
            criticalfailure,
            failure,
            mixedsuccess,
            success
        }
        public enum SacrificeType
        {
            none,
            meat,
            plants,
            meals,
            animal,
            human
        }
        public enum OfferingSize : int
        {
            none = 0,
            meagre = 5,
            decent = 10,
            sizable = 20,
            worthy = 50,
            impressive = 100
        }
        public static SacrificeResult GetSacrificeResult(Map map)
        {
            //Temporary
            //return SacrificeResult.success;

            bool Success = false;
            bool TableOfFun = false;

            int randSuccess = Rand.Range(1, 10);
            int modifier = 0;
            int successCheck = 2;
            Cthulhu.Utility.DebugReport("Initial Failure rate: " + successCheck.ToString() + "0%");
            Cthulhu.Utility.DebugReport("Rolling d10. " + randSuccess.ToString() + " result.");
            TryGetModifierForSpell(map.GetComponent<MapComponent_SacrificeTracker>().lastUsedAltar, out modifier);
            successCheck += modifier;
            Cthulhu.Utility.DebugReport("Adjusted Failure rate: " + successCheck.ToString() + "0%");
            if (randSuccess >= successCheck) Success = true;  //80% chance

            int randFun = Rand.Range(1, 10);
            if (randFun >= 6) TableOfFun = true;   //40% chance

            if (Success && TableOfFun) return SacrificeResult.mixedsuccess;
            if ((!Success) && TableOfFun) return SacrificeResult.failure;
            if (Success && (!TableOfFun)) return SacrificeResult.success;
            return SacrificeResult.criticalfailure;
        }

        public static void TryGetModifierForSpell(Building_SacrificialAltar altar, out int modifier)
        {
            modifier = 0;
            if (altar == null) return;
            if (altar.currentSacrificeDeity == null) return;
            if (altar.currentSpell == null) return;
            if (altar.Map.GetComponent<MapComponent_SacrificeTracker>().lastSacrificeCongregation == null) return;

            CosmicEntity deity = altar.currentSacrificeDeity;
            IncidentDef spell = altar.currentSpell;

            //Is tier 1?
            foreach (IncidentDef current in deity.tier1Spells)
            {
                if (current == spell) { Cthulhu.Utility.DebugReport(current.defName + " is a tier 1 spell. No difficulty modifier added."); goto
            CongregationBonus; }
            }

            //Is tier 2? +10% difficulty
            foreach (IncidentDef current in deity.tier2Spells)
            {
                if (current == spell) { Cthulhu.Utility.DebugReport(current.defName + " is a tier 2 spell. +10% sacrifice failure rate."); modifier = 1; goto
            CongregationBonus; }
            }

            //Is tier 3? +20% difficulty
            foreach (IncidentDef current in deity.tier3Spells)
            {
                if (current == spell) { Cthulhu.Utility.DebugReport(current.defName + " is a tier 3 spell. +20% sacrifice failure rate."); modifier = 2; goto
            CongregationBonus; }
            }

            //Is final spell? +50% difficulty
            if (spell == deity.finalSpell) { Cthulhu.Utility.DebugReport(spell.defName + " is a final spell. +50% sacrifice failure rate."); modifier = 5; }

            CongregationBonus:

            bool perfect = false;
            float value = CongregationBonus(altar.Map.GetComponent<MapComponent_SacrificeTracker>().lastSacrificeCongregation, deity, out perfect);
            if (perfect) modifier -= 1;
        }

        public static float CongregationBonus(List<Pawn> congregationIn, CosmicEntity entity, out bool perfect)
        {
            StringBuilder s = new StringBuilder();
            s.Append("Congregation Bonus Report");
            s.AppendLine();
            s.AppendLine();
            List<Pawn> congregation = new List<Pawn>(congregationIn);
            perfect = false;
            float result = 0f;
            int count = 0;
            if (congregation == null) return result;
            if (congregation.Count == 0) return result;
            if (entity == null) return result;
            //Are they wearing the right outfits?
            foreach (Pawn member in congregation)
            {
                bool wearingHood = false;
                bool wearingRobes = false;
                if (member == null) { count++; continue; }
                if (member.Dead) { count++; continue; }
                if (!member.IsColonist) { count++; continue; }
                if (member.apparel == null) continue;
                if (member.apparel.WornApparel == null) continue;
                if (member.apparel.WornApparelCount == 0) continue;
                foreach (Apparel clothing in member.apparel.WornApparel)
                {
                    if (clothing == null) continue;
                    if (clothing.def == null) continue;
                    if (clothing.def.defName == "Apparel_CultistRobes") { wearingRobes = true; result += 0.005f; }
                    if (clothing.def.defName == "Apparel_CultistHood" ||
                        clothing.def.defName == "Apparel_StandardHood" ||
                        clothing.def.defName == "Apparel_CthulhuMaskHood" ||
                        clothing.def.defName == "Apparel_NyarlathotepHood" ||
                        clothing.def.defName == "Apparel_DagonMitre" ||
                        clothing.def.defName == "Apparel_ShubMask") { wearingHood = true; result += 0.005f; }

                    if (entity.favoredApparel == null) continue;
                    if (entity.favoredApparel.Count == 0) continue;
                    foreach (ThingDef def in entity.favoredApparel)
                    {
                        if (def == null) continue;
                        if (clothing.def == def) { result += 0.025f; }
                    }
                }
                if (wearingHood && wearingRobes) { count++; s.Append(member.LabelShort + " is perfectly attired for the congregation."); s.AppendLine(); }
                else { s.Append(member.LabelShort + " is not perfectly attired for the congregation."); s.AppendLine(); }
            }
            if (result == 0) CultUtility.RemindPlayerAboutCongregationBonuses();
            if (count >= congregation.Count) { perfect = true; s.Append("Perfect Bonus: +0.05"); s.AppendLine(); result += 0.05f; }
            s.Append("Congregation Bonus: " + result.ToString("F"));
            s.AppendLine();
            s.Append("=========================");
            Cthulhu.Utility.DebugReport(s.ToString());
            return result;
        }


        /// <summary>
        /// When an execution completes, this method should trigger.
        /// </summary>
        /// <param name="sacrifice"></param>
        /// <param name="executioner"></param>
        /// <param name="altar"></param>
        /// <param name="deity"></param>
        /// <param name="spell"></param>
        public static void SacrificeExecutionComplete(Pawn sacrifice, Pawn executioner, Building_SacrificialAltar altar, CosmicEntity deity, IncidentDef spell)
        {
            altar.ChangeState(Building_SacrificialAltar.State.sacrificing, Building_SacrificialAltar.SacrificeState.finishing);

            deity.ReceiveSacrifice(sacrifice, altar.Map);

            altar.sacrifice = null;

            altar.Map.GetComponent<MapComponent_SacrificeTracker>().lastUsedAltar = altar;
            if (altar.Map.GetComponent<MapComponent_SacrificeTracker>().lastSacrificeType == SacrificeType.human)
            {

                altar.Map.GetComponent<MapComponent_SacrificeTracker>().lastResult = GetSacrificeResult(altar.Map);

                CultTableOfFun funTable = new CultTableOfFun();

                var result = altar.Map.GetComponent<MapComponent_SacrificeTracker>().lastResult;
                switch (result)
                {
                    case SacrificeResult.success:
                        Cthulhu.Utility.DebugReport("Sacrifice: Success");
                        CastSpell(spell, altar.Map, true);
                        break;
                    case SacrificeResult.mixedsuccess:
                        Cthulhu.Utility.DebugReport("Sacrifice: Mixed Success");
                        CastSpell(spell, altar.Map, true);
                        funTable.RollTableOfFun(altar.Map);
                        break;
                    case SacrificeResult.failure:
                        Cthulhu.Utility.DebugReport("Sacrifice: Failure");
                        funTable.RollTableOfFun(altar.Map);
                        SacrificeSpellComplete(executioner, altar);
                        break;
                    case SacrificeResult.criticalfailure:
                        Cthulhu.Utility.DebugReport("Sacrifice: Critical failure");
                        SacrificeSpellComplete(executioner, altar);
                        break;

                }
            }
            //Tell the player!
            altar.Map.GetComponent<MapComponent_SacrificeTracker>().GenerateSacrificeMessage();
        }
        public static void WorshipComplete(Pawn preacher, Building_SacrificialAltar altar, CosmicEntity deity)
        {
            altar.ChangeState(Building_SacrificialAltar.State.worshipping, Building_SacrificialAltar.WorshipState.finishing);

            deity.ReceiveWorship(preacher);
            
            altar.ChangeState(Building_SacrificialAltar.State.worshipping, Building_SacrificialAltar.SacrificeState.finished);
            //altar.currentState = Building_SacrificialAltar.State.finished;

            Messages.Message("Finished worshipping.", MessageSound.Benefit);
        }
        public static void OfferingComplete(Pawn offerer, Building_SacrificialAltar altar, CosmicEntity deity)
        {
            //altar.ChangeState(Building_SacrificialAltar.State.worshipping, Building_SacrificialAltar.WorshipState.finishing);

            altar.ChangeState(Building_SacrificialAltar.State.offering, Building_SacrificialAltar.OfferingState.finished);
            deity.ReceiveOffering(offerer, altar);

            if (IsActorAvailable(offerer))
            {
                Job job = new Job(CultDefOfs.ReflectOnOffering);
                job.targetA = altar;
                offerer.QueueJob(job);
                offerer.jobs.EndCurrentJob(JobCondition.InterruptForced);
            }
            Messages.Message("Finished worshipping.", MessageSound.Benefit);
        }
        #endregion GetResults

        #region Bools

        public static bool IsSomeoneInvestigating(Map map)
        {
            if (map.GetComponent<MapComponent_LocalCultTracker>() != null)
            {
                if (map.GetComponent<MapComponent_LocalCultTracker>().CurrentSeedState == CultSeedState.NeedWriting)
                    return true;
            }
            if (map.mapPawns.FreeColonists != null)
            {
                foreach (Pawn colonist in map.mapPawns.FreeColonists)
                {
                    if (colonist.CurJob != null)
                    {
                        if (colonist.CurJob.def.defName == "Investigate" ||
                            colonist.CurJob.def.defName == "WriteTheBook")
                        {
                            return true;
                        }
                    }
                }
            }
            return false;
        }

        public static bool AreCultObjectsAvailable(Map map)
        {
            Cthulhu.Utility.DebugReport("Cult Objects Check");
            //Do we have a forbidden knowledge center?
            if (AreForbiddenKnowledgeCentersAvailable(map)) {
                Cthulhu.Utility.DebugReport("Passed");
                return true; }
            //Do we have a book available?
            if (AreOccultGrimoiresAvailable(map)) {
                Cthulhu.Utility.DebugReport("Failed");
                return true; }
            return false;
        }

        public static bool AreOccultGrimoiresAvailable(Map map)
        {
            if (map.listerThings.AllThings != null)
            {
                foreach (Thing thing in map.listerThings.ThingsOfDef(ThingDef.Named("Cults_Grimoire")))
                {
                    return true;
                }
            }
            return false;
        }

        public static bool AreForbiddenKnowledgeCentersAvailable(Map map)
        {
            if (map.listerBuildings.AllBuildingsColonistOfClass<Building_ForbiddenReserachCenter>() != null)
            {
                foreach (Building_ForbiddenReserachCenter frc in map.listerBuildings.AllBuildingsColonistOfClass<Building_ForbiddenReserachCenter>())
                {
                    return true;
                }
            }
            return false;
        }

        public static bool CheckValidCultName(string str)
        {
            if (str.Length > 40)
            {
                return false;
            }
            string str2 = new string(Path.GetInvalidFileNameChars());
            Regex regex = new Regex("[" + Regex.Escape(str2) + "]");
            return !regex.IsMatch(str);
        }
        public static bool IsPreacher(Pawn p)
        {
            List<Thing> list = p.Map.listerThings.AllThings.FindAll(s => s.GetType() == typeof(Building_SacrificialAltar));
            foreach (Building_SacrificialAltar b in list)
            {
                if (b.preacher == p) return true;
            }
            return false;
        }
        public static bool IsExecutioner(Pawn p)
        {
            List<Thing> list = p.Map.listerThings.AllThings.FindAll(s => s.GetType() == typeof(Building_SacrificialAltar));
            foreach (Building_SacrificialAltar b in list)
            {
                if (b.executioner == p) return true;
            }
            return false;
        }
        public static bool IsSacrifice(Pawn p)
        {
            List<Thing> list = p.Map.listerThings.AllThings.FindAll(s => s.GetType() == typeof(Building_SacrificialAltar));
            foreach (Building_SacrificialAltar b in list)
            {
                if (b.sacrifice == p) return true;
            }
            return false;
        }
        public static bool ResultFalseWithReport(StringBuilder s)
        {
            s.Append("ActorAvailble: Result = Unavailable");
            Cthulhu.Utility.DebugReport(s.ToString());
            return false;
        }
        public static bool IsActorAvailable(Pawn preacher, bool downedAllowed=false)
        {
            StringBuilder s = new StringBuilder();
            s.Append("ActorAvailble Checks Initiated");
            s.AppendLine();
            if (preacher == null)
                return ResultFalseWithReport(s);
            s.Append("ActorAvailble: Passed null Check");
            s.AppendLine();
            //if (!preacher.Spawned)
            //    return ResultFalseWithReport(s);
            //s.Append("ActorAvailble: Passed not-spawned check");
            //s.AppendLine();
            if (preacher.Dead)
                return ResultFalseWithReport(s);
            s.Append("ActorAvailble: Passed not-dead");
            s.AppendLine();
            if (preacher.Downed && !downedAllowed)
                return ResultFalseWithReport(s);
            s.Append("ActorAvailble: Passed downed check & downedAllowed = " + downedAllowed.ToString());
            s.AppendLine();
            if (preacher.Drafted)
                return ResultFalseWithReport(s);
            s.Append("ActorAvailble: Passed drafted check");
            s.AppendLine();
            if (preacher.InAggroMentalState)
                return ResultFalseWithReport(s);
            s.Append("ActorAvailble: Passed drafted check");
            s.AppendLine();
            if (preacher.InMentalState)
                return ResultFalseWithReport(s);
            s.Append("ActorAvailble: Passed InMentalState check");
            s.AppendLine();
            s.Append("ActorAvailble Checks Passed");
            return true;
        }
        public static bool IsCultMinded(Pawn pawn)
        {
            if (pawn.needs.TryGetNeed<Need_CultMindedness>().CurLevel > Need_CultMindedness.ThreshHigh)
                return true;
            return false;
        }
        public static bool ShouldAttendSacrifice(Pawn p, Building_SacrificialAltar altar)
        {
            if (!IsActorAvailable(altar.executioner))
            {
                AbortCongregation(altar);
                return false;
            }
            //Everyone get over here!
            if (p != altar.executioner && p != altar.sacrifice)
            {
                return true;
            }

            return false;
        }
        public static bool ShouldAttendWorship(Pawn p, Building_SacrificialAltar altar)
        {
            if (!IsActorAvailable(altar.preacher))
            {
                AbortCongregation(altar);
                return false;
            }
            //Everyone get over here!
            if (p != altar.preacher)
            {
                return true;
            }

            return false;
        }
        #endregion Bools

        public static void RemindPlayerAboutCongregationBonuses()
        {
            if (Rand.Range(0, 100) < 20)
            {
                Messages.Message("Tip: Wear cultist apparel for a worship bonus.", MessageSound.Silent);
            }
        }

        #region ThoughtGivers
        public static void AffectCultMindedness(Pawn pawn, float amount=0f, float max=0.99f)
        {
            float trueMax = max;
            if (pawn == null) return;
            float result = pawn.needs.TryGetNeed<Need_CultMindedness>().CurLevel;
            if (result > trueMax) trueMax = result;
            result += amount;
            result = Mathf.Clamp(result, 0.01f, trueMax);
            pawn.needs.TryGetNeed<Need_CultMindedness>().CurLevel = result;
        }
        public static void InvestigatedCultSeed(Pawn pawn, Thing investigatee)
        {
            //It's a day to remember
            TaleDef taleToAdd = TaleDef.Named("ObservedNightmareMonolith");
            if (investigatee is Building_TreeOfMadness) taleToAdd = TaleDef.Named("ObservedNightmareTree");
            if ((pawn.IsColonist || pawn.HostFaction == Faction.OfPlayer) && taleToAdd != null)
            {
                TaleRecorder.RecordTale(taleToAdd, new object[]
                {
                    pawn,
                });
            }
            //Internal memory
            pawn.needs.mood.thoughts.memories.TryGainMemoryThought(DefDatabase<ThoughtDef>.GetNamed("MadeInvestigation"));

            Cthulhu.Utility.ApplySanityLoss(pawn);
            AffectCultMindedness(pawn, 0.10f);
            pawn.Map.GetComponent<MapComponent_LocalCultTracker>().CurrentSeedState = CultSeedState.NeedWriting;
            pawn.Map.GetComponent<MapComponent_LocalCultTracker>().CurrentSeedPawn = pawn;
        }
        public static void FinishedTheBook(Pawn pawn)
        {
            pawn.needs.mood.thoughts.memories.TryGainMemoryThought(DefDatabase<ThoughtDef>.GetNamed("BlackoutBook"));
            Cthulhu.Utility.ApplySanityLoss(pawn);
            pawn.Map.GetComponent<MapComponent_LocalCultTracker>().CurrentSeedState = CultSeedState.NeedTable;
            pawn.Map.GetComponent<MapComponent_LocalCultTracker>().CurrentSeedPawn = pawn;

            //Spawn in the book.
            IntVec3 spawnLoc = pawn.Position.RandomAdjacentCell8Way();
            ThingWithComps thing = (ThingWithComps)ThingMaker.MakeThing(CultDefOfs.Cults_Grimoire, null);
            //thing.SetFaction(Faction.OfPlayer);
            GenPlace.TryPlaceThing(thing, spawnLoc, pawn.Map, ThingPlaceMode.Near);
            Find.WindowStack.Add(new Dialog_MessageBox("CultBookSummary".Translate(new object[]
            {
                pawn.Name.ToStringShort
            }), "CultBookLabel".Translate()));

            //string textLabel = "CultBookLabel".Translate();
            //string textSummary = "CultBookSummary".Translate(new object[]
            //    {
            //        pawn.Name.ToStringShort
            //    });
            //Find.LetterStack.ReceiveLetter(textLabel, textSummary, LetterType.Good, spawnLoc);
            
        }
        public static void AttendSacrificeTickCheckEnd(Pawn pawn)
        {
            if (pawn == null) return;
            TryGainTempleRoomThought(pawn);
            if (pawn.RaceProps.Animal) return;
            ThoughtDef newThought = CultUtility.GetSacrificeThoughts(pawn);
            if (newThought != null)
            {
                pawn.needs.mood.thoughts.memories.TryGainMemoryThought(newThought);
            }
        }
        public static void AttendWorshipTickCheckEnd(Pawn preacher, Pawn pawn)
        {
            if (preacher == null) return;
            if (pawn == null) return;
            TryGainTempleRoomThought(pawn);
            ThoughtDef newThought = CultUtility.GetAttendWorshipThoughts(preacher, pawn);
            if (newThought != null)
            {
                pawn.needs.mood.thoughts.memories.TryGainMemoryThought(newThought);
            }
        }
        public static void HoldWorshipTickCheckEnd(Pawn preacher)
        {
            if (preacher == null) return;
            TryGainTempleRoomThought(preacher);
            ThoughtDef newThought = DefDatabase<ThoughtDef>.GetNamed("HeldSermon");
            if (newThought != null)
            {
                preacher.needs.mood.thoughts.memories.TryGainMemoryThought(newThought);
            }
        }

        public static ThoughtDef GetSacrificeThoughts(Pawn attendee)
        {
            if (attendee != null && attendee.Map.GetComponent<MapComponent_SacrificeTracker>().lastSacrificeType == SacrificeType.human)
            {
                switch (attendee.Map.GetComponent<MapComponent_SacrificeTracker>().lastResult)
                {
                    case SacrificeResult.mixedsuccess:
                    case SacrificeResult.success:
                        if (IsCultMinded(attendee))
                        {
                            return CultDefOfs.AttendedSuccessfulSacrifice;
                        }
                        return CultDefOfs.InnocentAttendedSuccessfulSacrifice;

                    case SacrificeResult.failure:
                    case SacrificeResult.criticalfailure:
                        if (IsCultMinded(attendee))
                        {
                            return CultDefOfs.AttendedFailedSacrifice;
                        }

                        return CultDefOfs.InnocentAttendedFailedSacrifice;
                    case SacrificeResult.none:
                        return null;
                }
            }
            return null;
        }
        public static ThoughtDef GetAttendWorshipThoughts(Pawn preacher, Pawn attendee)
        {
            //The grades of a sermon are categorized like this internally.
            const float S_Effect = 0.3f;
            const float A_Effect = 0.25f;
            const float B_Effect = 0.1f;
            const float C_Effect = 0.05f;
            const float F_Effect = 0.01f;

            float CultistMod = Rand.Range(0.01f, 0.02f);
            float InnocentMod = Rand.Range(-0.005f, 0.1f);

            if (attendee != null)
            {
                int num = preacher.skills.GetSkill(SkillDefOf.Social).Level;
                num += Rand.Range(-6, 6); //Randomness

                
                //S-Ranked Sermon: WOW!
                if (num > 20)
                {
                    if (IsCultMinded(attendee))
                    {
                        CultUtility.AffectCultMindedness(attendee, S_Effect + CultistMod);
                        return CultDefOfs.AttendedIncredibleSermonAsCultist;
                    }
                    CultUtility.AffectCultMindedness(attendee, S_Effect + InnocentMod);
                    return CultDefOfs.AttendedIncredibleSermonAsInnocent;
                }
                //A-Ranked Sermon: Fantastic
                if (num <= 20 && num > 15)
                {
                    if (IsCultMinded(attendee))
                    {
                        CultUtility.AffectCultMindedness(attendee, A_Effect + CultistMod);
                        return CultDefOfs.AttendedGreatSermonAsCultist;
                    }
                    CultUtility.AffectCultMindedness(attendee, A_Effect + InnocentMod);
                    return CultDefOfs.AttendedGreatSermonAsInnocent;
                }
                //B-Ranked Sermon: Alright
                if (num <= 15 && num > 10)
                {
                    if (IsCultMinded(attendee))
                    {
                        CultUtility.AffectCultMindedness(attendee, B_Effect + CultistMod);
                        return CultDefOfs.AttendedGoodSermonAsCultist;
                    }
                    CultUtility.AffectCultMindedness(attendee, B_Effect + InnocentMod);
                    return CultDefOfs.AttendedGoodSermonAsInnocent;
                }
                //C-Ranked Sermon: Average
                if (num <= 10 && num > 5)
                {
                    if (IsCultMinded(attendee))
                    {
                        CultUtility.AffectCultMindedness(attendee, C_Effect + CultistMod);
                        return CultDefOfs.AttendedDecentSermonAsCultist;
                    }
                    CultUtility.AffectCultMindedness(attendee, C_Effect + InnocentMod);
                    return CultDefOfs.AttendedDecentSermonAsInnocent;
                }
                //F-Ranked Sermon: Garbage
                else
                {
                    if (IsCultMinded(attendee))
                    {
                        CultUtility.AffectCultMindedness(attendee, F_Effect + CultistMod);
                        return CultDefOfs.AttendedAwfulSermonAsCultist;
                    }
                    CultUtility.AffectCultMindedness(attendee, F_Effect + InnocentMod);
                    return CultDefOfs.AttendedAwfulSermonAsInnocent;
                }
            }
            return null;
        }
        // RimWorld.JoyUtility
        public static void TryGainTempleRoomThought(Pawn pawn)
        {
            Room room = pawn.GetRoom();
            ThoughtDef def = ThoughtDef.Named("PrayedInImpressiveTemple");
            if (pawn == null) return;
            if (room == null) return;
            if (room.Role == null) return;
            if (def == null) return;
            if (room.Role.defName == "Temple")
            {
                int scoreStageIndex = RoomStatDefOf.Impressiveness.GetScoreStageIndex(room.GetStat(RoomStatDefOf.Impressiveness));
                if (def.stages[scoreStageIndex] == null) return;
                pawn.needs.mood.thoughts.memories.TryGainMemoryThought(ThoughtMaker.MakeThought(def, scoreStageIndex), null);
            }
        }


        #endregion ThoughtGivers

        #region AltarJobs
        #region Offering

        #endregion Offering
        #region Sacrifice
        public static void GiveAttendSacrificeJob(Building_SacrificialAltar altar, Pawn attendee)
        {
            if (IsExecutioner(attendee)) return;
            if (IsSacrifice(attendee)) return;
            if (!IsActorAvailable(attendee)) return;
            if (attendee.jobs.curJob.def.defName == "ReflectOnResult") return;
            if (attendee.jobs.curJob.def.defName == "AttendSacrifice") return;
            IntVec3 result;
            Building chair;
            if (!WatchBuildingUtility.TryFindBestWatchCell(altar, attendee, true, out result, out chair))
            {

                if (!WatchBuildingUtility.TryFindBestWatchCell(altar as Thing, attendee, false, out result, out chair))
                {
                    return;
                }
            }


            if (chair != null)
            {
                Job J = new Job(CultDefOfs.AttendSacrifice, altar.executioner, altar, chair);
                //if (altar.executioner.CurJob.def == CultDefOfs.ReflectOnResult)
                //    J = new Job(CultDefOfs.ReflectOnResult, altar);
                attendee.QueueJob(J);
                attendee.jobs.EndCurrentJob(JobCondition.InterruptForced);
            }
            else
            {
                Job J = new Job(CultDefOfs.AttendSacrifice, altar.executioner, altar, result);
                //if (altar.executioner.CurJob.def == CultDefOfs.ReflectOnResult)
                //    J = new Job(CultDefOfs.ReflectOnResult, altar);
                attendee.QueueJob(J);
                attendee.jobs.EndCurrentJob(JobCondition.InterruptForced);
            }
        }
        #endregion Sacrifice
        #region Worship
        public static Pawn DetermineBestResearcher(Map map)
        {
            Pawn result = null;
            foreach (Pawn p in map.mapPawns.FreeColonistsSpawned)
            {
                if (result == null) result = p;
                if (Cthulhu.Utility.GetResearchSkill(result) < Cthulhu.Utility.GetResearchSkill(p))
                {
                    result = p;
                }
            }
            return result;
        }
        public static Pawn DetermineBestPreacher(Map map)
        {
            Pawn result = null;
            foreach (Pawn p in map.mapPawns.FreeColonistsSpawned)
            {
                if (result == null) result = p;
                if (CultUtility.IsCultMinded(p) && Cthulhu.Utility.GetSocialSkill(result) < Cthulhu.Utility.GetSocialSkill(p))
                {
                    result = p;
                }
            }
            if (!CultUtility.IsCultMinded(result)) result = null;
            return result;
        }
        public static void GiveAttendWorshipJob(Building_SacrificialAltar altar, Pawn attendee)
        {
            if (IsPreacher(attendee)) return;
            if (attendee.Drafted) return;
            if (attendee.jobs.curJob.def.defName == "ReflectOnWorship") return;
            if (attendee.jobs.curJob.def.defName == "AttendWorship") return;
            IntVec3 result;
            Building chair;
            if (!WatchBuildingUtility.TryFindBestWatchCell(altar, attendee, true, out result, out chair))
            {

                if (!WatchBuildingUtility.TryFindBestWatchCell(altar as Thing, attendee, false, out result, out chair))
                {
                    return;
                }
            }


            if (chair != null)
            {
                Job J = new Job(CultDefOfs.AttendWorship, altar.preacher, altar, chair);
                if (altar.preacher.CurJob.def == CultDefOfs.ReflectOnWorship)
                    J = new Job(CultDefOfs.ReflectOnWorship, altar);
                attendee.QueueJob(J);
                attendee.jobs.EndCurrentJob(JobCondition.InterruptForced);
            }
            else
            {
                Job J = new Job(CultDefOfs.AttendWorship, altar.preacher, altar, result);
                if (altar.preacher.CurJob.def == CultDefOfs.ReflectOnWorship)
                    J = new Job(CultDefOfs.ReflectOnWorship, altar);
                attendee.QueueJob(J);
                attendee.jobs.EndCurrentJob(JobCondition.InterruptForced);
            }
        }
        public static void AbortCongregation(Building_SacrificialAltar altar)
        {
            if (altar != null) altar.ChangeState(Building_SacrificialAltar.State.notinuse);
        }
        public static void AbortCongregation(Building_SacrificialAltar altar, String reason)
        {
            if (altar != null) altar.ChangeState(Building_SacrificialAltar.State.notinuse);
            Messages.Message(reason + " Aborting congregation.", MessageSound.Negative);
        }
        #endregion Worship
        #endregion AltarJobs

        #region Spells
        public static void CastSpell(IncidentDef spell, Map map, bool fromAltar = false)
        {
            if(spell != null)
            {
                IncidentParms parms = StorytellerUtility.DefaultParmsNow(Find.Storyteller.def, spell.category, map as IIncidentTarget);
                spell.Worker.TryExecute(parms);
                Cthulhu.Utility.DebugReport("Spell cast: " + spell.ToString());
            }
            if (fromAltar)
            {
                Building_SacrificialAltar lastAltar = map.GetComponent<MapComponent_SacrificeTracker>().lastUsedAltar;
                SacrificeSpellComplete(lastAltar.executioner, lastAltar);
            }
        }
        public static void SacrificeSpellComplete(Pawn executioner, Building_SacrificialAltar altar)
        {
            if(IsActorAvailable(executioner))
            {
                Job job = new Job(CultDefOfs.ReflectOnResult);
                job.targetA = altar;
                executioner.QueueJob(job);
                executioner.jobs.EndCurrentJob(JobCondition.InterruptForced);
            }
            if (altar == null) return;
            altar.ChangeState(Building_SacrificialAltar.State.sacrificing, Building_SacrificialAltar.SacrificeState.finished);
            //altar.currentState = Building_SacrificialAltar.State.finished;
            //Map.GetComponent<MapComponent_SacrificeTracker>().lastUsedAltar.ChangeState(Building_SacrificialAltar.State.sacrificing, Building_SacrificialAltar.SacrificeState.finished);
        }
        #endregion Spells

    }
}