﻿// ----------------------------------------------------------------------
// These are basic usings. Always let them be here.
// ----------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;

// ----------------------------------------------------------------------
// These are RimWorld-specific usings. Activate/Deactivate what you need:
// ----------------------------------------------------------------------
using UnityEngine;         // Always needed
//using VerseBase;         // Material/Graphics handling functions are found here
using Verse;               // RimWorld universal objects are here (like 'Building')
using Verse.AI;          // Needed when you do something with the AI
using Verse.AI.Group;
using Verse.Sound;       // Needed when you do something with Sound
using Verse.Noise;       // Needed when you do something with Noises
using RimWorld;            // RimWorld specific functions are found here (like 'Building_Battery')
using RimWorld.Planet;   // RimWorld specific functions for world creation
using System.Reflection;
using System.Text.RegularExpressions;
//using RimWorld.SquadAI;  // RimWorld specific functions for squad brains 

namespace CultOfCthulhu
{
    class MapComponent_TransmogrifyTracker : MapComponent
    {
        public MapComponent_TransmogrifyTracker(Map map) : base (map)
        {
            this.map = map;
        }

        public Dictionary<int, ThingDef> newDefList = new Dictionary<int, ThingDef>();
        public bool postSave = false;
        public bool postLoad = false;

        public List<Pawn> toBeRemoved = new List<Pawn>();
        public Dictionary<int,ThingDef> toBeRefreshed = new Dictionary<int,ThingDef>();

        public MapComponent_TransmogrifyTracker Get
        {
            get
            {
                MapComponent_TransmogrifyTracker MapComponent_TransmogrifyTracker = map.components.OfType<MapComponent_TransmogrifyTracker>().FirstOrDefault<MapComponent_TransmogrifyTracker>();
                bool flag = MapComponent_TransmogrifyTracker == null;
                if (flag)
                {
                    MapComponent_TransmogrifyTracker = new MapComponent_TransmogrifyTracker(map);
                    map.components.Add(MapComponent_TransmogrifyTracker);
                }
                return MapComponent_TransmogrifyTracker;
            }
        }

        public IEnumerable<Pawn> PetsToTransmogrify
        {
            get {
                   //Get a pet with a master.
                   IEnumerable<Pawn> one =  from Pawn pets in map.mapPawns.AllPawnsSpawned
                   where pets.RaceProps.Animal && pets.Faction == Faction.OfPlayer && !pets.Dead && !pets.Downed && pets.RaceProps.petness > 0f && pets.playerSettings.master != null
                   select pets;
                    //No master? Okay, still search for pets.
                    if (one.Count<Pawn>() == 0)
                {
                    one = from Pawn pets in map.mapPawns.AllPawnsSpawned
                          where pets.RaceProps.Animal && pets.Faction == Faction.OfPlayer && !pets.Dead && !pets.Downed && pets.RaceProps.petness > 0f
                          select pets;
                }
                    //No pets? Okay, search for player animals.
                    if (one.Count<Pawn>() == 0)
                {
                    one = from Pawn pets in map.mapPawns.AllPawnsSpawned
                          where pets.RaceProps.Animal && pets.Faction == Faction.OfPlayer && !pets.Dead && !pets.Downed
                          select pets;
                }   
                    //Return anything if we find anything, or return a null, it's all good.
                return one;

            }
        }
        
        public IEnumerable<ThingDef_Transmogrified> TransmogrifyDefs
        {
            get
            {
                return from ThingDef_Transmogrified def in DefDatabase<ThingDef_Transmogrified>.AllDefs
                       where def.InUse == false
                       select def;

            }
        }

        public Pawn Transmogrify(Pawn pawn=null, bool loadMode=false)
        {
            //No pawn? Okay, find one.
            if (pawn == null)
                pawn = PetsToTransmogrify.RandomElement<Pawn>();
   
            if (loadMode)
            {
                ThingDef prevDef;
                if (!newDefList.TryGetValue(pawn.thingIDNumber, out prevDef))
                {
                    Log.Error("Can't get value from dictionary");
                }
            }

            //Old Pawn
            //Keeps a duplicate inside the loop to prevent problems.
            IntVec3 oldLoc = pawn.Position;
            ThingDef oldDef = pawn.def;
            string oldString = pawn.ToString();
            //Pawn oldPawn = GenerateNewPawnFromSource(pawn.def, pawn);
            pawn.DeSpawn();
            Cthulhu.Utility.DebugReport("DeSpawned " + oldString);

            //New def
            ThingDef newDef = GenerateMonstrousDef(oldDef);
            TransmogrifyRacialValues(oldDef, newDef);

            //New Pawn with new def
            Pawn newPawn = GenerateNewPawnFromSource(newDef, pawn);
            UpgradeBody(newPawn);
            GenSpawn.Spawn(newPawn, oldLoc, map);
           
            //Add new pawn and old def to the list
            newDefList.Add(newPawn.thingIDNumber, oldDef);

            pawn.Destroy(0);
            //pawn.Discard();

            if (!loadMode) Messages.Message(newPawn.LabelShort + "'s form has been enhanced.", MessageSound.Benefit);
            
            return newPawn;
        }

        public ThingDef GenerateMonstrousDef(ThingDef oldDef)
        {
            ThingDef newDef = new ThingDef();
            try
            {
                #region baseproperties
                //Copy every base thing
                newDef.altitudeLayer = oldDef.altitudeLayer;
                newDef.category = oldDef.category;
                newDef.thingClass = oldDef.thingClass;
                newDef.selectable = oldDef.selectable;
                newDef.tickerType = TickerType.Normal;
                newDef.useHitPoints = oldDef.useHitPoints;
                newDef.hasTooltip = oldDef.hasTooltip;
                newDef.alwaysHaulable = oldDef.alwaysHaulable;
                newDef.socialPropernessMatters = oldDef.socialPropernessMatters;
                newDef.pathCost = oldDef.pathCost;
                newDef.tradeability = oldDef.tradeability;
                newDef.soundImpactDefault = oldDef.soundImpactDefault;
                newDef.inspectorTabs = new List<Type>();
                foreach (Type tab in oldDef.inspectorTabs)
                {
                    newDef.inspectorTabs.Add(tab);
                    //Cthulhu.Utility.DebugReport("Added " + tab.ToString());
                }
                newDef.inspectorTabsResolved = new List<InspectTabBase>();
                foreach (InspectTabBase tab in oldDef.inspectorTabsResolved)
                {
                    newDef.inspectorTabsResolved.Add(tab);
                }
                newDef.comps = new List<CompProperties>();
                foreach (CompProperties comp in oldDef.comps)
                {
                    newDef.comps.Add(comp);
                }
                newDef.drawGUIOverlay = oldDef.drawGUIOverlay;
                #endregion baseproperties


                //Copy every animal base thing
                newDef.statBases = new List<StatModifier>();
                foreach (StatModifier mod in oldDef.statBases)
                {
                    newDef.statBases.Add(mod);
                }
                newDef.race = oldDef.race;

                newDef.recipes = new List<RecipeDef>();
                foreach (RecipeDef recipe in oldDef.recipes)
                {
                    newDef.recipes.Add(recipe);
                }


                string oldName = oldDef.defName;
                string newName = Regex.Replace(oldName, "[0-9]", "");
                if (!newName.Contains("Monstrous")) newName = newName + "_Monstrous";
                Cthulhu.Utility.DebugReport(oldName);
                Cthulhu.Utility.DebugReport(newName);
                newDef.defName = newName;
                newDef.label = "Monstrous " + oldDef.label;
                newDef.description = oldDef.description;
                //newDef.Verbs = oldDef.Verbs;
                //Pawn_MeleeVerbs tempVerbs = oldDef.Verbs;
                newDef.tradeTags = new List<string>();
                //Cthulhu.Utility.DebugReport("trade tags");
                foreach (string s in oldDef.tradeTags)
                {
                    newDef.tradeTags.Add(s);
                }
                if (newDef.thingCategories == null)
                {
                    newDef.thingCategories = new List<ThingCategoryDef>();
                }
                CrossRefLoader.RegisterListWantsCrossRef<ThingCategoryDef>(newDef.thingCategories, "Animal");


            }
            catch (Exception e)
            {
                Log.Error(e.ToString());
            }
            return newDef;
        }

        public void TransmogrifyRacialValues(ThingDef oldDef, ThingDef newDef)
        {
            newDef.race.body = oldDef.race.body;

            newDef.race.trainableIntelligence = TrainableIntelligence.Advanced;
            newDef.race.foodType = FoodTypeFlags.CarnivoreAnimalStrict;
            newDef.race.predator = true;

            newDef.race.lifeExpectancy += 50;
            newDef.race.baseBodySize += 1.0f;
            newDef.race.baseHealthScale += 0.5f;
            newDef.race.baseHungerRate += 0.1f;

        }

        public void CopyPawnRecords(Pawn pawn, Pawn newPawn)
        {
            //Who has a relationship with this pet?
            Pawn pawnMaster = null;
            foreach (Pawn current in map.mapPawns.AllPawns)
            {
                if (current.relations.DirectRelationExists(PawnRelationDefOf.Bond, pawn))
                {
                    pawnMaster = current;
                }
            }

            //Fix the relations
            if (pawnMaster != null)
            {
                pawnMaster.relations.TryRemoveDirectRelation(PawnRelationDefOf.Bond, pawn);
                pawnMaster.relations.AddDirectRelation(PawnRelationDefOf.Bond, newPawn);
                //Train that stuff!

                DefMap<TrainableDef, int> oldMap = (DefMap<TrainableDef, int>)typeof(Pawn_TrainingTracker).GetField("steps", BindingFlags.Instance | BindingFlags.NonPublic).GetValue(pawn.training);
                DefMap<TrainableDef, int> newMap = (DefMap<TrainableDef, int>)typeof(Pawn_TrainingTracker).GetField("steps", BindingFlags.Instance | BindingFlags.NonPublic).GetValue(newPawn.training);

                foreach (TrainableDef def in DefDatabase<TrainableDef>.AllDefs)
                {
                    newMap[def] = oldMap[def];
                }
            }


            foreach (Hediff hediff in pawn.health.hediffSet.hediffs)
            {
                newPawn.health.AddHediff(hediff);
            }

        }

        public void UpgradeBody(Pawn newPawn)
        {

            //Where is your head, sir?
            BodyPartRecord tempRecord = null;
            foreach (BodyPartRecord current in newPawn.RaceProps.body.AllParts)
            {
                if (current.def == BodyPartDefOf.Brain)
                {
                    tempRecord = current;
                }
            }

            //Where is your body, sir?
            BodyPartRecord tempRecord2 = null;
            foreach (BodyPartRecord current in newPawn.RaceProps.body.AllParts)
            {
                if (current.def == BodyPartDefOf.Heart)
                {
                    tempRecord2 = current;
                }
            }

            //Error catch: Missing head!
            if (tempRecord == null)
            {
                Log.Error("Couldn't find brain of the pawn to upgrade.");
                return;
            }

            //Error catch: Missing body!
            if (tempRecord2 == null)
            {
                Log.Error("Couldn't find heart to upgrade.");
                return;
            }

            //Check if they are already upgraded.
            foreach (Hediff current in newPawn.health.hediffSet.hediffs)
            {
                if (current.def == CultDefOfs.Cults_MonstrousBody)
                {
                    Messages.Message(newPawn.LabelShort + " already posesses " + CultDefOfs.Cults_MonstrousBody.label, MessageSound.Negative);
                    return;
                }
                if (current.def == CultDefOfs.Cults_MonstrousBrain)
                {
                    Messages.Message(newPawn.LabelShort + " already posesses " + CultDefOfs.Cults_MonstrousBrain.label, MessageSound.Negative);
                    return;
                }
            }
            
            newPawn.health.AddHediff(CultDefOfs.Cults_MonstrousBody, tempRecord2, null);
            newPawn.health.AddHediff(CultDefOfs.Cults_MonstrousBrain, tempRecord, null);

        }

        public Pawn GenerateNewPawnFromSource(ThingDef newDef, Pawn sourcePawn)
        {
            Pawn pawn = (Pawn)ThingMaker.MakeThing(newDef);
            //Cthulhu.Utility.DebugReport("Declare a new thing");
            pawn.Name = sourcePawn.Name;
            //Cthulhu.Utility.DebugReport("The name!");
            pawn.SetFactionDirect(Faction.OfPlayer);
            pawn.kindDef = sourcePawn.kindDef;
            //Cthulhu.Utility.DebugReport("The def!");
            pawn.pather = new Pawn_PathFollower(pawn);
            //Cthulhu.Utility.DebugReport("The pather!");
            pawn.ageTracker = new Pawn_AgeTracker(pawn);
            pawn.health = new Pawn_HealthTracker(pawn);
            pawn.jobs = new Pawn_JobTracker(pawn);
            pawn.mindState = new Pawn_MindState(pawn);
            pawn.filth = new Pawn_FilthTracker(pawn);
            pawn.needs = new Pawn_NeedsTracker(pawn);
            pawn.stances = new Pawn_StanceTracker(pawn);
            pawn.natives = new Pawn_NativeVerbs(pawn);
            pawn.relations = sourcePawn.relations;
            PawnComponentsUtility.CreateInitialComponents(pawn);

            if (pawn.RaceProps.ToolUser)
            {
                pawn.equipment = new Pawn_EquipmentTracker(pawn);
                pawn.carryTracker = new Pawn_CarryTracker(pawn);
                pawn.apparel = new Pawn_ApparelTracker(pawn);
                pawn.inventory = new Pawn_InventoryTracker(pawn);
            }
            if (pawn.RaceProps.intelligence <= Intelligence.ToolUser)
            {
                pawn.caller = new Pawn_CallTracker(pawn);
            }
            pawn.gender = sourcePawn.gender;
            pawn.needs.SetInitialLevels();
            GenerateRandomAge(pawn);
            CopyPawnRecords(sourcePawn, pawn);
            //Cthulhu.Utility.DebugReport("We got so far.");
            return pawn;
        }

        public void DebugDef(ThingDef d)
        {
            string s =
            "ThingClass: " + d.thingClass.ToString() + "@" +
            "Category: " + d.category.ToString() + "@" +
            "Selectable: " + d.selectable.ToString() + "@" +
            "TickerType: " + d.tickerType.ToString() + "@" +
            "altitudeLayer: " + d.altitudeLayer.ToString() + "@" +
            "useHitPoints: " + d.useHitPoints.ToString() + "@" +
            "hasTooltip: " + d.hasTooltip.ToString() + "@" +
            "alwaysHaulable: " + d.alwaysHaulable.ToString() + "@" +
            "socialPropernessMatters " + d.socialPropernessMatters.ToString() + "@" +
            "pathCost: " + d.pathCost.ToString() + "@" +
            "tradeability: " + d.tradeability.ToString() + "@" +
            "soundImpactDefault: " + d.soundImpactDefault.ToString() + "@" +
            "comps: " + d.comps.ToString() + "@" +
            "drawGUIOnMap: " + d.drawGUIOverlay.ToString() + "@" +
            "defName: " + d.defName.ToString() + "@" +
            "label: " + d.label.ToString() + "@" +
            "description: " + d.description.ToString();

            //"inspectorTabs: " + d.inspectorTabs.ToString() + "@" +
            //"tradeTags: " + d.tradeTags.ToString() + "@" +
            //"statBases: " + d.statBases.ToString() + "@" +
            //"race: " + d.race.ToString() + "@" +
            //"recipes: " + d.recipes.ToString() + "@

            s = s.Replace("@", System.Environment.NewLine);

            Cthulhu.Utility.DebugReport
            (
                s
            );
        }

        public void GenerateRandomAge(Pawn pawn)
        {
            int num = 0;
            int num2;
            do
            {
                if (pawn.RaceProps.ageGenerationCurve != null)
                {
                    num2 = Mathf.RoundToInt(Rand.ByCurve(pawn.RaceProps.ageGenerationCurve, 200));
                }
                else if (pawn.RaceProps.IsMechanoid)
                {
                    num2 = Rand.Range(0, 2500);
                }
                else
                {
                    if (!pawn.RaceProps.Animal)
                    {
                        goto IL_84;
                    }
                    num2 = Rand.Range(1, 10);
                }
                num++;
                if (num > 100)
                {
                    goto IL_95;
                }
            }
            while (num2 > pawn.kindDef.maxGenerationAge || num2 < pawn.kindDef.minGenerationAge);
            goto IL_A5;
        IL_84:
            Log.Warning("Didn't get age for " + pawn);
            return;
        IL_95:
            Log.Error("Tried 100 times to generate age for " + pawn);
        IL_A5:
            pawn.ageTracker.AgeBiologicalTicks = ((long)((float)num2 * 3600000f) + (long)Rand.Range(0, 3600000));
            int num3;
            if (Rand.Value < pawn.kindDef.backstoryCryptosleepCommonality)
            {
                float value = Rand.Value;
                if (value < 0.7f)
                {
                    num3 = Rand.Range(0, 100);
                }
                else if (value < 0.95f)
                {
                    num3 = Rand.Range(100, 1000);
                }
                else
                {
                    int num4 = GenLocalDate.Year(map) - 2026 - pawn.ageTracker.AgeBiologicalYears;
                    num3 = Rand.Range(1000, num4);
                }
            }
            else
            {
                num3 = 0;
            }
            long num5 = (long)GenTicks.TicksAbs - pawn.ageTracker.AgeBiologicalTicks;
            num5 -= (long)num3 * 3600000L;
            pawn.ageTracker.BirthAbsTicks = num5;
            if (pawn.ageTracker.AgeBiologicalTicks > pawn.ageTracker.AgeChronologicalTicks)
            {
                pawn.ageTracker.AgeChronologicalTicks = (pawn.ageTracker.AgeBiologicalTicks);
            }
        }

        public void LoadTransmogrification()
        {
            Cthulhu.Utility.DebugReport("Started Transmogrification");
            if (this.newDefList == null)
            {
                Log.Error("Missing New Def List");
                return;
            }
            try
            {
                int y = newDefList.Count;
                Cthulhu.Utility.DebugReport(y.ToString());
                for (int i = 0; i < y; i++)
                {
                    foreach (KeyValuePair<int, ThingDef> x in this.newDefList)
                    {
                        //this.newDefList.RemoveAll((KeyValuePair<Pawn, ThingDef> x) => x.Key.Destroyed);
                        Pawn p = map.mapPawns.AllPawnsSpawned.Find((Pawn s) => s.thingIDNumber == x.Key);
                        if (!IsMonstrous(p))
                        {
                            Transmogrify(p, true);
                            break;
                        }
                    }
                }
            }
            catch (Exception e) { Cthulhu.Utility.DebugReport(e.ToString()); }
        }

        public bool IsMonstrous(Pawn p)
        {
            if (p == null) return false;
            if (p.def == null) return false;
            if (p.def.defName == null) return false;
            if (p.def.defName.Contains("Monstrous"))
            {
                return true;
            }
            return false;
        }

        public void RefreshMonstrous(Pawn pawn)
        {
            if (!IsMonstrous(pawn))
            {
                pawn.def = toBeRefreshed[pawn.thingIDNumber];
            }
        }

        public override void MapComponentTick()
        {
            base.MapComponentTick();
            if (this.postSave)
            {
                this.postSave = false;

                foreach (int i in toBeRefreshed.Keys)
                {
                    Pawn p = map.mapPawns.AllPawnsSpawned.Find((Pawn x) => x.thingIDNumber == i);
                    RefreshMonstrous(p);
                }
                toBeRefreshed = new Dictionary<int,ThingDef>();
            }

            if (this.postLoad)
            {
                this.postLoad = false;
                
                //this.newDefList.RemoveAll((KeyValuePair<string, ThingDef> x) => x.Key.Destroyed);
                if (this.newDefList.Count > 0) Cthulhu.Utility.DebugReport(this.newDefList.ToStringFullContents<int, ThingDef>());

                LoadTransmogrification();
            }

        }

        public override void ExposeData()
        {
            if (Scribe.mode == LoadSaveMode.Saving)
            {
                //this.newDefList.RemoveAll((KeyValuePair<string, ThingDef> x) => x.Key.Destroyed);
                if (this.newDefList.Count > 0) Cthulhu.Utility.DebugReport(this.newDefList.ToStringFullContents<int, ThingDef>());

                for (int i = 0; i < newDefList.Count; i++)
                {
                    foreach (KeyValuePair<int, ThingDef> x in newDefList)
                    {
                        Pawn p = map.mapPawns.AllPawnsSpawned.Find((Pawn q) => q.thingIDNumber == x.Key);
                        if (IsMonstrous(p))
                        {
                            toBeRefreshed.Add(x.Key, p.def);
                            p.def = x.Value;
                            Cthulhu.Utility.DebugReport(x.Key.ToString() + " " + x.Value.ToString());
                            
                            break;
                        }
                    }
                }

                postSave = true;
            }

            base.ExposeData();
            
            Scribe_Collections.LookDictionary<int, ThingDef>(ref this.newDefList, "newDefList", LookMode.Value, LookMode.Def);

            if (Scribe.mode == LoadSaveMode.PostLoadInit)
            {
                if (this.newDefList == null)
                {
                    this.newDefList = new Dictionary<int, ThingDef>();
                }
                postLoad = true;
            }

        }

    }
}
