-- ===========================================================================
--	Single Player Create Game w/ Advanced Options
-- ===========================================================================
include("InstanceManager");
include("PlayerSetupLogic");
include("Civ6Common");


-- ===========================================================================
-- ===========================================================================

local m_NonLocalPlayerSlotManager	:table = InstanceManager:new("NonLocalPlayerSlotInstance", "Root", Controls.NonLocalPlayersSlotStack);
local m_singlePlayerID				:number = 0;			-- The player ID of the human player in singleplayer.
local m_AdvancedMode				:boolean = false;



-- ===========================================================================
-- Input Handler
-- ===========================================================================
function OnInputHandler( pInputStruct:table )
	local uiMsg = pInputStruct:GetMessageType();
	if uiMsg == KeyEvents.KeyUp then
		local key:number = pInputStruct:GetKey();
		if key == Keys.VK_ESCAPE then
			OnBackButton();
		end
	end
	return true;
end

-- ===========================================================================
function CreatePulldownDriver(o, parameter, c)
	local driver = {
		Control = c,
		UpdateValue = function(value)
			local button = c:GetButton();
			button:SetText( value and value.Name or nil);
		end,
		UpdateValues = function(values)
			c:ClearEntries();

			for i,v in ipairs(values) do
				local entry = {};
				c:BuildEntry( "InstanceOne", entry );
				entry.Button:SetText(v.Name);
				entry.Button:SetToolTipString(v.Description);

				entry.Button:RegisterCallback(Mouse.eLClick, function()
					o:SetParameterValue(parameter, v);
					Network.BroadcastGameConfig();
				end);
			end
			c:CalculateInternals();
		end,
		SetEnabled = function(enabled)
			c:SetDisabled(not enabled);
		end,
		SetVisible = nil,	-- Never hide the basic pulldown.
		Destroy = nil,		-- It's a fixed control, no need to delete.
	};
	
	return driver;	
end

-- ===========================================================================
-- Override parameter behavior for basic setup screen.
g_ParameterFactories["GameDifficulty"] = function(o, parameter)

	local drivers = {};
	-- Basic setup version.
	-- Use an explicit table.
	table.insert(drivers, CreatePulldownDriver(o, parameter, Controls.CreateGame_GameDifficulty));

	-- Advanced setup version.
	-- Create the parameter dynamically like we normally would...
	table.insert(drivers, GameParameters_UI_DefaultCreateParameterDriver(o, parameter));

	return drivers;
end

-- ===========================================================================
g_ParameterFactories["GameSpeeds"] = function(o, parameter)

	local drivers = {};
	-- Basic setup version.
	-- Use an explicit table.
	table.insert(drivers, CreatePulldownDriver(o, parameter, Controls.CreateGame_SpeedPulldown));

	-- Advanced setup version.
	-- Create the parameter dynamically like we normally would...
	table.insert(drivers, GameParameters_UI_DefaultCreateParameterDriver(o, parameter));

	return drivers;
end

-- ===========================================================================
g_ParameterFactories["Map"] = function(o, parameter)

	local drivers = {};
	-- Basic setup version.
	-- Use an explicit table.
	table.insert(drivers, CreatePulldownDriver(o, parameter, Controls.CreateGame_MapType));

	-- Advanced setup version.
	-- Create the parameter dynamically like we normally would...
	table.insert(drivers, GameParameters_UI_DefaultCreateParameterDriver(o, parameter));

	return drivers;
end

-- ===========================================================================
g_ParameterFactories["MapSize"] = function(o, parameter)

	local drivers = {};
	-- Basic setup version.
	-- Use an explicit table.
	table.insert(drivers, CreatePulldownDriver(o, parameter, Controls.CreateGame_MapSize));

	-- Advanced setup version.
	-- Create the parameter dynamically like we normally would...
	table.insert(drivers, GameParameters_UI_DefaultCreateParameterDriver(o, parameter));

	return drivers;
end

-- ===========================================================================
-- Remove player handler.
function RemovePlayer(voidValue1, voidValue2, control)
	print("Removing Player " .. tonumber(voidValue1));
	local playerConfig = PlayerConfigurations[voidValue1];
	playerConfig:SetLeaderTypeName(nil);
	
	GameConfiguration.RemovePlayer(voidValue1);

	LuaEvents.GameSetup_PlayerCountChanged();
	GameSetup_RefreshParameters();
end

-- ===========================================================================
-- Add UI entries for all the players.  This does not set the
-- UI values of the player.
-- ===========================================================================
function RefreshPlayerSlots()

	RebuildPlayerParameters();
	m_NonLocalPlayerSlotManager:ResetInstances();

	local player_ids = GameConfiguration.GetParticipatingPlayerIDs();

	local minPlayers = MapConfiguration.GetMinMajorPlayers() or 2;
	local maxPlayers = MapConfiguration.GetMaxMajorPlayers() or 2;
	local can_remove = #player_ids > minPlayers;
	local can_add = #player_ids < maxPlayers;

	Controls.AddAIButton:SetHide(not can_add);

	print("There are " .. #player_ids .. " participating players.");
	for i, player_id in ipairs(player_ids) do	
		if(m_singlePlayerID == player_id) then
			SetupLeaderPulldown(player_id, Controls, "Basic_LocalPlayerPulldown", "Basic_LocalPlayerCivIcon", "Basic_LocalPlayerLeaderIcon");
			SetupLeaderPulldown(player_id, Controls, "Advanced_LocalPlayerPulldown", "Advanced_LocalPlayerCivIcon", "Advanced_LocalPlayerLeaderIcon");
		else
			local ui_instance = m_NonLocalPlayerSlotManager:GetInstance();
			
			-- Assign the Remove handler
			if(can_remove) then
				ui_instance.RemoveButton:SetVoid1(player_id);
				ui_instance.RemoveButton:RegisterCallback(Mouse.eLClick, RemovePlayer);
			end
			ui_instance.RemoveButton:SetHide(not can_remove);
			
			SetupLeaderPulldown(player_id, ui_instance,"PlayerPullDown");
		end
	end

	Controls.NonLocalPlayersSlotStack:CalculateSize();
	Controls.NonLocalPlayersSlotStack:ReprocessAnchoring();
	Controls.NonLocalPlayersStack:CalculateSize();
	Controls.NonLocalPlayersStack:ReprocessAnchoring();
	
	Controls.NonLocalPlayersPanel:CalculateInternalSize();
	GameSetup_RefreshParameters();
end

-- ===========================================================================
function OnPlayerCountChanged()
	print("Player Count Changed");
	RefreshPlayerSlots();
end

-- ===========================================================================
function OnShow()
	BuildGameSetup();
	RefreshPlayerSlots();
	GameSetup_RefreshParameters();
	AutoSizeGridButton(Controls.DefaultButton,50,22,10,"H");
	AutoSizeGridButton(Controls.CloseButton,133,36,10,"H");
	LuaEvents.GameDebug_GetValues( "RestartGame" );
end

-- ===========================================================================
function OnHide()
	HideGameSetup();
	ReleasePlayerParameters();
end


-- ===========================================================================
-- Button Handlers
-- ===========================================================================

-- ===========================================================================
function OnAddAIButton()
	-- Search for an empty slot number and mark the slot as computer.
	-- Then dispatch the player count changed event.
	local iPlayer = 0;
	while(true) do
		local playerConfig = PlayerConfigurations[iPlayer];
		
		-- If we've reached the end of the line, exit.
		if(playerConfig == nil) then
			break;
		end

		-- Find a suitable slot to add the AI.
		if (playerConfig:GetSlotStatus() == SlotStatus.SS_CLOSED) then
			playerConfig:SetSlotStatus(SlotStatus.SS_COMPUTER);
			playerConfig:SetMajorCiv();

			LuaEvents.GameSetup_PlayerCountChanged();
			GameSetup_RefreshParameters();
			Controls.NonLocalPlayersPanel:CalculateSize();
			break;
		end

		-- Increment the AI, this assumes that either player config will hit nil 
		-- or we'll reach a suitable slot.
		iPlayer = iPlayer + 1;
	end
end

-- ===========================================================================
function OnAdvancedSetup()
	Controls.CreateGameWindow:SetHide(true);
	Controls.AdvancedOptionsWindow:SetHide(false);
	m_AdvancedMode = true;
end

-- ===========================================================================
function OnDefaultButton()
	print("Reseting Setup Parameters");
	g_Refreshing = true;
	g_GameParameters:ResetDefaults();
	GameConfiguration.RegenerateSeeds();
    g_GameParameters:FullRefresh();
	ResetPlayerParameters();
	g_Refreshing = false;
	if(g_NeedsAdditionalRefresh) then
		g_NeedsAdditionalRefresh = false;
		print("Refreshing again, to be sure.")
		return GameSetup_RefreshParameters();
	end

end

-- ===========================================================================
function OnStartButton()
	-- Is WorldBuilder active?
	if (GameConfiguration.IsWorldBuilder()) then
		UI.SetWorldRenderView( WorldRenderView.VIEW_2D );
		UI.PlaySound("Set_View_2D");
		playerParams = deepcopy(g_PlayerParameters);
		LuaEvents.GameDebug_AddValue("RestartGameConfiguration","PlayerParameters", playerParams);
		gameParams = deepcopy(g_GameParameters.Parameters);
		LuaEvents.GameDebug_AddValue("RestartGameConfiguration","GameParameters", gameParams);
		Network.HostGame(ServerType.SERVER_TYPE_NONE);
		
	else
		-- No, start a normal game
		UI.PlaySound("Set_View_3D");
		playerParams = deepcopy(g_PlayerParameters);
		LuaEvents.GameDebug_AddValue("RestartGameConfiguration","PlayerParameters", playerParams);
		gameParams = deepcopy(g_GameParameters.Parameters);
		LuaEvents.GameDebug_AddValue("RestartGameConfiguration","GameParameters", gameParams);
		Network.HostGame(ServerType.SERVER_TYPE_NONE);
	end
end



----------------------------------------------------------------    
function OnBackButton()
	if(m_AdvancedMode) then
		Controls.CreateGameWindow:SetHide(false);
		Controls.AdvancedOptionsWindow:SetHide(true);
		m_AdvancedMode = false;		
	else
		UIManager:DequeuePopup( ContextPtr );
	end
end

----------------------------------------------------------------    
-- ===========================================================================
--	Handle Window Sizing
-- ===========================================================================

function Resize()
	local screenX, screenY:number = UIManager:GetScreenSizeVal();
	local hideLogo = true;
	if(screenY >= Controls.MainWindow:GetSizeY() + (Controls.LogoContainer:GetSizeY()+ Controls.LogoContainer:GetOffsetY())*2) then
		hideLogo = false;
	end
	Controls.LogoContainer:SetHide(hideLogo);
	Controls.MainGrid:ReprocessAnchoring();
end

-- ===========================================================================
function OnUpdateUI( type:number, tag:string, iData1:number, iData2:number, strData1:string )   
  if type == SystemUpdateUI.ScreenResize then
    Resize();
  end
end

-- ===========================================================================
function OnBeforeMultiplayerInviteProcessing()
	-- We're about to process a game invite.  Get off the popup stack before we accidently break the invite!
	UIManager:DequeuePopup( ContextPtr );
end

function deepcopy(orig)
    local orig_type = type(orig)
    local copy
    if orig_type == 'table' then
        copy = {}
        for orig_key, orig_value in next, orig, nil do
            copy[deepcopy(orig_key)] = deepcopy(orig_value)
        end
        setmetatable(copy, deepcopy(getmetatable(orig)))
    else -- number, string, boolean, etc
        copy = orig
    end
    return copy
end

-- ===========================================================================
function OnGameDebugReturn( context, contextTable )
	if context=="RestartGame" then
		if (contextTable["Stage2"]) then
			LuaEvents.GameDebug_AddValue("RestartGame","Stage2", nil);
			LuaEvents.GameDebug_GetValues( "RestartGameConfiguration" );
		end	
	elseif context=="RestartGameConfiguration" then
		local previousGame = contextTable["GameParameters"];
		local previousPlayers = contextTable["PlayerParameters"];
		if previousGame ~= nil then
		
			for k in pairs(previousGame) do
				g_GameParameters:SetParameterValue(previousGame[k], previousGame[k].Value );
			end
			GameConfiguration.RegenerateSeeds();
			GameSetup_RefreshParameters();
			
			for i = 1 , GameConfiguration.GetParticipatingPlayerCount() do
				PlayerConfigurations[i-1]:SetLeaderName(previousPlayers[i][2].Parameters.PlayerLeader.Value.Name);
				PlayerConfigurations[i-1]:SetLeaderTypeName(previousPlayers[i][2].Parameters.PlayerLeader.Value.Value);
			end
			GameSetup_RefreshParameters();
			RefreshPlayerParameters();
			Network.HostGame(ServerType.SERVER_TYPE_NONE);
		end
	end
	
end

-- ===========================================================================
--
-- ===========================================================================
function Initialize()

	ContextPtr:SetInputHandler( OnInputHandler, true );
	ContextPtr:SetShowHandler( OnShow );
	ContextPtr:SetHideHandler( OnHide );

	Controls.AddAIButton:RegisterCallback( Mouse.eLClick, OnAddAIButton );
	Controls.AddAIButton:RegisterCallback( Mouse.eMouseEnter, function() UI.PlaySound("Main_Menu_Mouse_Over"); end);
	Controls.AdvancedSetupButton:RegisterCallback( Mouse.eLClick, OnAdvancedSetup );
	Controls.AdvancedSetupButton:RegisterCallback( Mouse.eMouseEnter, function() UI.PlaySound("Main_Menu_Mouse_Over"); end);
	Controls.DefaultButton:RegisterCallback( Mouse.eLClick, OnDefaultButton);
	Controls.DefaultButton:RegisterCallback( Mouse.eMouseEnter, function() UI.PlaySound("Main_Menu_Mouse_Over"); end);
	Controls.StartButton:RegisterCallback( Mouse.eLClick, OnStartButton );
	Controls.StartButton:RegisterCallback( Mouse.eMouseEnter, function() UI.PlaySound("Main_Menu_Mouse_Over"); end);
	Controls.CloseButton:RegisterCallback( Mouse.eLClick, OnBackButton );
	Controls.CloseButton:RegisterCallback( Mouse.eMouseEnter, function() UI.PlaySound("Main_Menu_Mouse_Over"); end);

	LuaEvents.GameSetup_PlayerCountChanged.Add(OnPlayerCountChanged);
	LuaEvents.GameDebug_Return.Add( OnGameDebugReturn );
	Events.SystemUpdateUI.Add( OnUpdateUI );
	Events.BeforeMultiplayerInviteProcessing.Add( OnBeforeMultiplayerInviteProcessing );
	Resize();
end
Initialize();