/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedList;
import java.util.Objects;

/**
 * An inputstream that holds a second inputstream to which it compares every
 * byte that is read.
 *
 * Used for validating classes that read and write structured data.
 *
 * @author Mark Fairchild
 * @version 2016/06/20
 */
final public class ComparatorInputStream extends InputStream implements Closeable {

    /**
     * Creates a new <code>ComparatorInputStream</code> that compares output data to
     * the specified <code>InputStream</code>.
     *
     * @param current The <code>InputStream</code> from which to read.
     * @param original The <code>InputStream</code> to compare to .
     */
    public ComparatorInputStream(InputStream current, InputStream original) {
        this(current, original, 0);
    }

    /**
     * Creates a new <code>ComparatorInputStream</code> that compares output data to
     * the specified <code>InputStream</code>.
     *
     * @param current The <code>InputStream</code> from which to read.
     * @param original The <code>InputStream</code> to compare to .
     * @param skips The number of mismatched bytes to ignore before raising an
     * exception.
     */
    public ComparatorInputStream(InputStream current, InputStream original, int skips) {
        this.CURRENT = Objects.requireNonNull(current);
        this.ORIGINAL = Objects.requireNonNull(original);
        this.position = 0;
        this.CONTEXT = new LinkedList<>();
        this.setSkips(skips);
    }

    /**
     * @see InputStream#read()
     * @return
     * @throws IOException
     */
    @Override
    public int read() throws IOException {
        int currentByte = this.CURRENT.read();
        int originalByte = this.ORIGINAL.read();
        
        if ((byte) currentByte != (byte) originalByte) {
            if (this.skips > 0) {
                this.skips--;
            } else {
                StringBuilder context = new StringBuilder();
                this.CONTEXT.forEach(b -> context.append(String.format("%02x ", b)));
                throw new Mismatch(this.position, originalByte, currentByte, context.toString());
            }
        }

        this.position++;
        this.CONTEXT.addLast((byte) originalByte);

        if (this.CONTEXT.size() > 32) {
            this.CONTEXT.removeFirst();
        }
        
        return currentByte;
    }

    /**
     * @see InputStream#available() 
     * @return
     * @throws IOException 
     */
    @Override
    public int available() throws IOException {
        return this.CURRENT.available();
    }

    /**
     * Sets the number of mismatched bytes to skip.
     *
     * @param newSkips The new value for the skips field. Must be nonnegative.
     */
    public void setSkips(int newSkips) {
        if (newSkips < 0) {
            throw new IllegalArgumentException("skips must be nonnegative.");
        }
        this.skips = newSkips;
    }

    final private InputStream CURRENT;
    final private InputStream ORIGINAL;
    final private LinkedList<Byte> CONTEXT;
    private long position;
    private int skips;

    /**
     * Exception that represent a mismatch between the original file and the
     * data being written out.
     */
    static final public class Mismatch extends IOException {

        private Mismatch(long position, int originalByte, int writeByte, String context) {
            super(String.format(MSG, position, originalByte, writeByte, context));
            this.POSITION = position;
            this.ORIGINALBYTE = originalByte;
            this.WRITEBYTE = writeByte;
            this.CONTEXT = context;
        }

        /**
         * @return Returns the position in the stream where the mismatch
         * occurred.
         */
        public long getPosition() {
            return this.POSITION;
        }

        /**
         * @return Returns the byte that appeared in the original stream at the
         * site of the mismatch.
         */
        public int getOriginal() {
            return this.ORIGINALBYTE;
        }

        /**
         * @return Returns the byte that appeared in the output stream at the
         * site of the mismatch.
         */
        public int getWritten() {
            return this.WRITEBYTE;
        }

        /**
         * @return Returns the previous 32 bytes (at most) that were read before
         * the mismatch.
         */
        public String getContext() {
            return this.CONTEXT;
        }

        final private long POSITION;
        final private int ORIGINALBYTE;
        final private int WRITEBYTE;
        final private String CONTEXT;
        static final private String MSG = "Mismatch at %d: expected %02x but found %02x. Context:\n%s";
    }
}
