/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * An outputstream that holds an inputstream to which it compares every byte
 * that is written.
 *
 * Used for validating classes that read and write structured data.
 *
 * @author Mark Fairchild
 * @version 2016/07/04
 */
final public class ComparatorOutputStream extends OutputStream implements Closeable {

    /**
     * Creates a new <code>ComparatorStream</code> that compares output data to
     * the specified <code>InputStream</code>. Data will be discarded rather
     * than written out.
     *
     * @param original The <code>InputStream</code> to compare to .
     */
    public ComparatorOutputStream(InputStream original) {
        this(null, original, 0);
    }

    /**
     * Creates a new <code>ComparatorStream</code> that compares output data to
     * the specified <code>InputStream</code>.
     *
     * @param current The <code>OutputStream</code> to receive the new data.
     * @param original The <code>InputStream</code> to compare to .
     */
    public ComparatorOutputStream(OutputStream current, InputStream original) {
        this(current, original, 0);
    }

    /**
     * Creates a new <code>ComparatorStream</code> that compares output data to
     * the specified <code>InputStream</code>.
     *
     * @param current The <code>OutputStream</code> to receive the new data.
     * @param original The <code>InputStream</code> to compare to .
     * @param skips The number of mismatched bytes to ignore before raising an
     * exception.
     */
    public ComparatorOutputStream(OutputStream current, InputStream original, int skips) {
        this.CURRENT = current;
        this.ORIGINAL = Objects.requireNonNull(original);
        this.position = 0;

        this.CURR_CTX = new LinkedList<>();
        this.ORIG_CTX = new LinkedList<>();

        this.setSkips(skips);
    }

    /**
     * @see OutputStream#write(int)
     * @param b
     * @throws IOException
     */
    @Override
    public void write(int b) throws IOException {
        int writeByte = (0xFF) & b;
        int originalByte = this.ORIGINAL.read();
        this.updateContext(originalByte, writeByte);

        if (null != this.CURRENT) {
            this.CURRENT.write(writeByte);
        }

        if ((byte) writeByte != (byte) originalByte) {
            if (this.skips > 0) {
                this.skips--;
            } else {
                String ctxMsg = this.generateContextString();
                throw new Mismatch(this.position, originalByte, writeByte, ctxMsg);
            }
        }

        this.position++;
    }

    /**
     * Sets the number of mismatched bytes to skip.
     *
     * @param newSkips The new value for the skips field. Must be nonnegative.
     */
    public void setSkips(int newSkips) {
        if (newSkips < 0) {
            throw new IllegalArgumentException("skips must be nonnegative.");
        }
        this.skips = newSkips;
    }

    /**
     * Produces the context message for mismatches.
     *
     * @return
     */
    private String generateContextString() {
        assert this.CURR_CTX.size() > 0;
        assert this.CURR_CTX.size() == ORIG_CTX.size();

        final int POS = this.CURR_CTX.size() - 1;

        try {
            for (int i = 0; i < 8 && this.ORIGINAL.available() > 0; i++) {
                int b = this.ORIGINAL.read();
                this.ORIG_CTX.add(b);
            }
        } catch (IOException ex) {

        }

        List<Integer> origLeft = this.ORIG_CTX.subList(0, POS);
        List<Integer> origRight = this.ORIG_CTX.subList(POS + 1, this.ORIG_CTX.size());
        List<Integer> currLeft = this.CURR_CTX.subList(0, POS);
        int origByte = this.ORIG_CTX.get(POS);
        int currByte = this.CURR_CTX.get(POS);

        String origLeftStr = origLeft.stream()
                .map(i -> String.format("%02x", i))
                .collect(Collectors.joining(" "));

        String origRightStr = origRight.stream()
                .map(i -> String.format("%02x", i))
                .collect(Collectors.joining(" "));

        String currLeftStr = currLeft.stream()
                .map(i -> String.format("%02x", i))
                .collect(Collectors.joining(" "));

        String origByteStr = String.format("%02x", origByte);
        String currByteStr = String.format("%02x", currByte);

        StringBuilder BUF = new StringBuilder();
        BUF
                .append("Original: ")
                .append(origLeftStr)
                .append("[").append(origByteStr).append("]")
                .append(origRightStr).append("\n")
                .append("Current : ")
                .append(currLeftStr)
                .append("[").append(currByteStr).append("]");

        return BUF.toString();
    }

    /**
     * Updates the context lists.
     *
     * @param originalByte The byte from the original file.
     * @param writeByte The byte being written.
     */
    private void updateContext(int originalByte, int writeByte) {
        this.ORIG_CTX.addLast(originalByte);
        this.CURR_CTX.addLast(writeByte);

        while (this.CURR_CTX.size() > 16) {
            this.CURR_CTX.removeFirst();
        }
        while (this.ORIG_CTX.size() > 16) {
            this.ORIG_CTX.removeFirst();
        }
    }

    final private OutputStream CURRENT;
    final private InputStream ORIGINAL;
    final private LinkedList<Integer> CURR_CTX;
    final private LinkedList<Integer> ORIG_CTX;
    private long position;
    private int skips;

    /**
     * Exception that represent a mismatch between the original file and the
     * data being written out.
     */
    static final public class Mismatch extends IOException {

        private Mismatch(long position, int originalByte, int writeByte, String context) {
            super(String.format(MSG, position, (byte) originalByte, (byte) writeByte, context));
            this.POSITION = position;
            this.ORIGINALBYTE = originalByte;
            this.WRITEBYTE = writeByte;
            this.CONTEXT = context;
        }

        /**
         * @return Returns the position in the stream where the mismatch
         * occurred.
         */
        public long getPosition() {
            return this.POSITION;
        }

        /**
         * @return Returns the byte that appeared in the original stream at the
         * site of the mismatch.
         */
        public int getOriginal() {
            return this.ORIGINALBYTE;
        }

        /**
         * @return Returns the byte that appeared in the output stream at the
         * site of the mismatch.
         */
        public int getWritten() {
            return this.WRITEBYTE;
        }

        /**
         * @return Returns the previous 32 bytes (at most) that were read before
         * the mismatch.
         */
        public String getContext() {
            return this.CONTEXT;
        }

        final private long POSITION;
        final private int ORIGINALBYTE;
        final private int WRITEBYTE;
        final private String CONTEXT;
        static final private String MSG = "Mismatch at %d: expected %02x but found %02x. Context:\n%s";
    }
}
