/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer;

import java.io.File;
import java.util.List;
import java.util.regex.Pattern;
import javafx.stage.FileChooser;
import javax.swing.filechooser.FileNameExtensionFilter;

/**
 *
 * @author Mark Fairchild
 */
public enum Game {
    SKYRIM_LE("Skyrim Legendary Edition", "Skyrim Savefile", "ess", "skse", "skyrim", "tesv.exe"),
    SKYRIM_SE("Skyrim Special Edition", "Skyrim SE Savefile", "ess", "ssse", "skyrim special edition", "skyrimse.exe"),
    FALLOUT4("Fallout 4", "Fallout 4 Savefile", "fos", "f4se", "fallout 4", "fallout4.exe");

    final public String NAME;
    final public String EXT;
    final public Pattern REGEX;
    final public String SCRIPTEXT;
    final public String DIR;
    final public String EXE;
    final public FileNameExtensionFilter SWING_FILTER;
    final public FileChooser.ExtensionFilter FX_FILTER;
    static public List<Game> VALUES = java.util.Collections.unmodifiableList(java.util.Arrays.asList(Game.values()));

    /**
     * Finds the <code>Game</code> that matches a filename, by extension.
     *
     * @param filename
     * @return
     */
    static public Game matchFilename(String filename) {
        for (Game game : VALUES) {
            if (game.REGEX.asPredicate().test(filename)) {
                return game;
            }
        }
        return null;
    }

    /**
     * Test if a savefile matches.
     *
     * @param f
     * @return
     */
    public boolean testFilename(File f) {
        return REGEX.asPredicate().test(f.getName());
    }

    /**
     * @return Flag indicating whether the game has a 32bit string model.
     */
    public boolean isStr32() {
        return this.isSSE() || this.isFO4();
    }

    /**
     * @return Flag indicating whether the game has a 64bit IDs.
     */
    public boolean isID64() {
        return this.isSSE() || this.isFO4();
    }
    
    /**
     * @return Flag indicating whether the game is Fallout 4.
     */
    public boolean isFO4() {
        return this == FALLOUT4;
    }

    /**
     * @return Flag indicating whether the game is Skyrim Legendary Edition or
     * Skyrim Special Edition.
     *
     */
    public boolean isSkyrim() {
        return this.isSSE() || this.isSLE();
    }

    /**
     * @return Flag indicating whether the game is Skyrim Special Edition.
     */
    public boolean isSSE() {
        return this == SKYRIM_SE;
    }

    /**
     * @return Flag indicating whether the game is Skyrim Legendary Edition.
     */
    public boolean isSLE() {
        return this == SKYRIM_LE;
    }

    /**
     * Creates a new <code>Game</code> for the specified extension.
     *
     * @param name The game's name.
     * @param save The name for savefiles.
     * @param extension The game's savefile extension.
     * @param scriptExt The game's script extender co-save extension.
     * @param dir The default name of the game's directory.
     * @param exe The filename of the game's executable.
     */
    private Game(String name, String save, String extension, String scriptExt, String dir, String exe) {
        this.NAME = name;
        this.EXT = extension;
        this.SCRIPTEXT = scriptExt;
        this.DIR = dir;
        this.EXE = exe;
        this.REGEX = Pattern.compile(String.format(".+\\.(%s)", extension), Pattern.CASE_INSENSITIVE);
        this.SWING_FILTER = new FileNameExtensionFilter(save, extension);
        this.FX_FILTER = new FileChooser.ExtensionFilter(save, "*." + extension);
    }

};
