/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer;

import java.io.BufferedOutputStream;
import java.io.Closeable;
import java.io.DataOutput;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Objects;

/**
 * A rough variant of <code>java.io.DataOutputStream</code> that is
 * little-endian.
 *
 * @author Mark Fairchild
 * @version 2016/04/21
 */
public class LittleEndianDataOutput extends OutputStream implements DataOutput, Closeable {

    /**
     * Creates a new <code>LittleEndianDataOutput</code> that opens a
     * <code>File</code>.
     *
     * @param file The <code>File</code> to open.
     * @return The new <code>LittleEndianDataOutput</code>.
     * @throws FileNotFoundException
     *
     */
    static public LittleEndianDataOutput open(File file) throws FileNotFoundException {
        return new LittleEndianDataOutput(new BufferedOutputStream(new FileOutputStream(file)));
    }

    /**
     * Creates a new <code>LittleEndianDataOutput</code> that wraps an
     * <code>InputStream</code>.
     *
     * @param orig The <code>OutputStream</code> to wrap.
     * @return The new <code>LittleEndianDataOutput</code>.
     *
     */
    static public LittleEndianDataOutput wrap(OutputStream orig) {
        Objects.requireNonNull(orig);
        return new LittleEndianDataOutput(orig);
    }

    /**
     * Creates a new <code>LittleEndianDataOutput</code> that opens a
     * <code>File</code>.
     *
     * @param file The <code>File</code> to open.
     * @return The new <code>LittleEndianDataOutput</code>.
     * @throws FileNotFoundException
     *
     */
    static public LittleEndianDataOutput test(File file) throws FileNotFoundException {
        return new LittleEndianDataOutput(new BufferedOutputStream(new FileOutputStream(file)));
    }

    /**
     * Creates a new <code>LittleEndianDataOutput</code> that wraps around a
     * supplied <code>OutputStream</code>.
     *
     * @param bigEnd The <code>OutputStream</code> to wrap.
     */
    public LittleEndianDataOutput(OutputStream bigEnd) {
        if (null == bigEnd) {
            throw new NullPointerException();
        }

        this.BIGEND = bigEnd;
        this.BUFFER = new byte[8];
    }

    /**
     * @see java.io.OutputStream#close()
     * @throws IOException
     */
    @Override
    public void close() throws IOException {
        this.BIGEND.close();
    }

    /**
     * @see java.io.OutputStream#write(int)
     * @throws IOException
     */
    @Override
    public void write(int b) throws IOException {
        this.BIGEND.write(b);
        this.position++;
    }

    /**
     * @see java.io.OutputStream#write(byte[])
     * @throws IOException
     */
    @Override
    public void write(byte[] b) throws IOException {
        this.BIGEND.write(b);
        this.position += b.length;
    }

    /**
     * @see java.io.OutputStream#write(byte[], int, int)
     * @throws IOException
     */
    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        this.BIGEND.write(b, off, len);
        this.position += len;
    }

    /**
     * @see java.io.DataOutput#writeBoolean(boolean)
     * @param v The boolean to write.
     * @throws IOException
     */
    @Override
    public void writeBoolean(boolean v) throws IOException {
        this.write(v ? 1 : 0);
    }

    /**
     * @see java.io.DataOutput#writeByte(int)
     * @param v The byte to write.
     * @throws IOException
     */
    @Override
    public void writeByte(int v) throws IOException {
        this.write(v);
    }

    /**
     * @see java.io.DataOutput#writeShort(int)
     * @param v The short to write.
     * @throws IOException
     */
    @Override
    public void writeShort(int v) throws IOException {
        BUFFER[0] = (byte) v;
        BUFFER[1] = (byte) (v >> 8);
        this.write(BUFFER, 0, 2);
    }

    /**
     * @see java.io.DataOutput#writeInt(int)
     * @param v The int to write.
     * @throws IOException
     */
    @Override
    public void writeInt(int v) throws IOException {
        BUFFER[0] = (byte) v;
        BUFFER[1] = (byte) (v >> 8);
        BUFFER[2] = (byte) (v >> 16);
        BUFFER[3] = (byte) (v >> 24);
        this.write(BUFFER, 0, 4);
    }

    /**
     * @see java.io.DataOutput#writeLong(long)
     * @param v The long to write.
     * @throws IOException
     */
    @Override
    public void writeLong(long v) throws IOException {
        BUFFER[0] = (byte) v;
        BUFFER[1] = (byte) (v >> 8);
        BUFFER[2] = (byte) (v >> 16);
        BUFFER[3] = (byte) (v >> 24);
        BUFFER[4] = (byte) (v >> 32);
        BUFFER[5] = (byte) (v >> 40);
        BUFFER[6] = (byte) (v >> 48);
        BUFFER[7] = (byte) (v >> 56);
        this.write(BUFFER, 0, 8);
    }

    /**
     * @see java.io.DataOutput#writeChar(char)
     * @param v The char to write.
     * @throws IOException
     */
    @Override
    public void writeChar(int v) throws IOException {
        this.writeShort((short) v);
    }

    /**
     * @see java.io.DataOutput#writeFloat(float)
     * @param v The float to write.
     * @throws IOException
     */
    @Override
    public void writeFloat(float v) throws IOException {
        this.writeInt(Float.floatToIntBits(v));
    }

    /**
     * @see java.io.DataOutput#writeDouble(double)
     * @param v The double to write.
     * @throws IOException
     */
    @Override
    public void writeDouble(double v) throws IOException {
        this.writeLong(Double.doubleToLongBits(v));
    }

    /**
     * Writes out a <code>String</code> in little-endian UTF8 format.
     *
     * @see java.io.DataOutput#writeUTF(String)
     * @param s The String to write.
     * @throws IOException
     */
    @Override
    public void writeUTF(String s) throws IOException {
        byte[] bytes = s.getBytes(StandardCharsets.UTF_8);
        this.writeShort(bytes.length);
        this.write(bytes);
    }

    /**
     * Writes out a <code>String</code> in little-endian 2-byte length-prefixed
     * format.
     *
     * @param s The String to write.
     * @throws IOException
     */
    public void writeWString(String s) throws IOException {
        byte[] bytes = s.getBytes(StandardCharsets.UTF_8);
        this.writeShort(bytes.length);
        this.write(bytes);
    }

    /**
     * Writes out a <code>String</code> in little-endian 4-byte length-prefixed
     * format.
     *
     * @param s The String to write.
     * @throws IOException
     */
    public void writeLString(String s) throws IOException {
        byte[] bytes = s.getBytes(StandardCharsets.UTF_8);
        this.writeInt(bytes.length);
        this.write(bytes);
    }

    /**
     * Writes out a <code>String</code> in little-endian 0-terminated format.
     *
     * @param s The String to write.
     * @throws IOException
     */
    public void writeZString(String s) throws IOException {
        this.write(s.getBytes(StandardCharsets.UTF_8));
        this.writeByte(0);
    }

    /**
     * Write out an <code>Element</code> from an <code>ESS</code>.
     *
     * @param e The <code>Element</code> to write.
     * @throws IOException
     */
    public void writeESSElement(restringer.ess.Element e) throws IOException {
        e.write(this);
    }

    @Override
    public String toString() {
        return String.format("%s (pos=%d)", super.toString(), this.position);
    }

    @Override
    public void writeBytes(String s) throws IOException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void writeChars(String s) throws IOException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     * Accessor for the underlying big-endian <code>OutputStream</code>.
     *
     * @return The underlying stream.
     */
    protected OutputStream getBigEndian() {
        return this.BIGEND;
    }

    /**
     * @return The position field.
     */
    public int getPosition() {
        return this.position;
    }

    final private OutputStream BIGEND;
    final private byte[] BUFFER;
    private int position;

}
