/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer;

import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.RandomAccessFile;
import java.util.Objects;

/**
 * Mirrors the behaviour of <code>RandomAccessFile</code> but with LittleEndianInput
 behaviour.
 *
 * @author Mark Fairchild
 * @version 2016/06/25
 */
public class LittleEndianRAF extends LittleEndianInput {

    /**
     * Creates a new <code>LittleEndianInputStream</code> that opens a
     * <code>File</code>.
     *
     * @param file The <code>File</code> to open.
     * @return The new <code>LittleEndianInputStream</code>.
     * @throws FileNotFoundException
     *
     */
    static public LittleEndianRAF open(File file) throws FileNotFoundException {
        Objects.requireNonNull(file);
        assert file.isFile();
        assert file.canRead();
        assert file.exists();

        final LittleEndianRAF RAF = new LittleEndianRAF(file, "r");
        return RAF;
    }

    /**
     * @see RandomAccessFile#RandomAccessFile(java.io.File, java.lang.String)
     * @param file
     * @param mode
     * @throws FileNotFoundException
     */
    public LittleEndianRAF(File file, String mode) throws FileNotFoundException {
        this.RAF = new RandomAccessFile(file, mode);
    }

    /**
     * Accessor for the underlying big-endian <code>RandomAccessFile</code>.
     *
     * @return The underlying stream.
     */
    public RandomAccessFile getBigEndian() {
        return this.RAF;
    }

    /**
     * @see java.io.InputStream#available()
     * @return The number of bytes available to be read.
     * @throws IOException
     */
    @Override
    public int available() throws IOException {
        return (int)(this.length() - this.getFilePointer());
    }

    /**
     * @see java.io.RandomAccessFile#seek(long)
     * @param pos
     * @throws IOException
     */
    public void seek(long pos) throws IOException {
        this.RAF.seek(pos);
    }

    /**
     * @see java.io.RandomAccessFile#getFilePointer()
     * @return
     * @throws IOException
     */
    public long getFilePointer() throws IOException {
        return this.RAF.getFilePointer();
    }
    
    /**
     * @see java.io.Closeable#close()
     * @throws IOException
     */
    @Override
    public void close() throws IOException {
        this.RAF.close();
    }

    /**
     * @see java.io.RandomAccessFile#skipBytes(int)
     * @param n The number of bytes to skip.
     * @return The number of bytes that were skipped.
     * @throws IOException
     */
    @Override
    public int skipBytes(int n) throws IOException {
        return this.RAF.skipBytes(n);
    }

    /**
     * @see java.io.RandomAccessFile#setLength(long)
     * @param newLength
     * @throws IOException
     */
    public void setLength(long newLength) throws IOException {
        this.RAF.setLength(newLength);
    }

    /**
     * @see java.io.RandomAccessFile#length()
     * @return
     * @throws IOException
     */
    public long length() throws IOException {
        return this.RAF.length();
    }

    /**
     * @see java.io.InputInput#read()
     * @return The next byte in the stream.
     * @throws IOException
     */
    @Override
    public int read() throws IOException {
        return this.RAF.read();
    }

    /**
     * @see LittleEndianInput#read(byte[]) 
     * @param b
     * @return 
     * @throws IOException
     */
    @Override
    public int read(byte[] b) throws IOException {
        int bytesRead = this.RAF.read(b);
        return bytesRead;
    }

    /**
     * @see LittleEndianInput#read(byte[], int, int) 
     * @param b
     * @param off
     * @param len
     * @return 
     * @throws IOException
     */
    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        int bytesRead = this.RAF.read(b, off, len);        
        return bytesRead;
    }

    @Override
    public String toString() {
        return this.RAF.toString();
    }

    final private RandomAccessFile RAF;

}
