/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer;

import it.unimi.dsi.fastutil.objects.ObjectLinkedOpenHashSet;
import java.io.*;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Predicate;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.swing.JOptionPane;
import org.apache.commons.compress.archivers.sevenz.*;
import restringer.pex.PexFile;
import restringer.pex.Pex;
import restringer.pex.RemappingStats;
import restringer.pex.TokenGenerator;
import restringer.gui.ProgressModel;

/**
 * Describes a set of mods that can be restrung.
 *
 * @author Mark Fairchild
 * @version 2016/06/16
 */
public class Profile {

    /**
     * Creates a new profile named "Default".
     */
    private Profile() {
        this("Default");
    }

    /**
     * Creates a new <code>Profile</code> with a specified name.
     *
     * @param name The name of the profile, which must be valid as a Windows
     * filename.
     */
    public Profile(String name) {
        Objects.requireNonNull(name);
        if (name.isEmpty()) {
            throw new IllegalArgumentException("Profile name must not be empty.");
        } else if (!validateName(name)) {
            throw new IllegalArgumentException("Profile name must be a valid filename.");
        }

        this.NAME = name;
        this.buildTime = -1;
        this.TIMER = new Timer("Timer");
        this.MODLIST = new ArrayList<>();

        final String HOME = System.getProperty("user.home");
        final File HOME_DIRECTORY = new File(HOME);
        final String PROFILE_PATH = "Restringer-" + this.NAME;
        this.outputDirectory = new File(HOME_DIRECTORY, PROFILE_PATH);
        this.outputDirectory = null;
    }

    /**
     * @return The name of the <code>Profile</code>.
     */
    public String getName() {
        return this.NAME;
    }

    /**
     * @return The list of mods.
     */
    public List<Mod> getMods() {
        return Collections.unmodifiableList(this.MODLIST);
    }

    /**
     * @return The list of mods.
     */
    public List<Mod> getCheckmarkedMods() {
        Predicate<Mod> isChecked = (mod -> mod.getStatus() == Mod.Status.CHECKED && !mod.isEmpty());
        Stream<Mod> stream = this.MODLIST.stream().filter(isChecked);
        return stream.collect(Collectors.toList());
    }

    /**
     * @see List#contains(java.lang.Object)
     * @param mod
     * @return
     */
    public boolean contains(Mod mod) {
        return this.MODLIST.contains(mod);
    }

    /**
     * @return The number of mods in the <code>Profile</code>.
     */
    public int size() {
        return this.MODLIST.size();
    }

    /**
     * @see List#add(java.lang.Object)
     * @param index The index at which to add the <code>Mod</code>.
     * @param mod The <code>Mod</code> to add.
     * @return The index of the <code>Mod</code>.
     */
    public int addMod(int index, Mod mod) {
        Objects.requireNonNull(mod);
        index = Math.min(index, this.MODLIST.size());

        if (this.MODLIST.contains(mod)) {
            this.MODLIST.remove(mod);
            this.MODLIST.add(index, mod);
        } else {
            this.MODLIST.add(index, mod);
        }

        return index;
    }

    /**
     * @see List#add(java.lang.Object)
     * @param mod The <code>Mod</code> to add.
     * @return The index of the <code>Mod</code>.
     */
    public int addMod(Mod mod) {
        Objects.requireNonNull(mod);

        if (!this.MODLIST.contains(mod)) {
            this.MODLIST.add(mod);
            return this.MODLIST.indexOf(mod);
        }

        return this.getMods().indexOf(mod);
    }

    /**
     * @see List#remove(java.lang.Object)
     * @param mod The <code>Mod</code> to remove.
     * @return The former index of the <code>Mod</code>, or <code>-1</code> if
     * the <code>Mod</code> wasn't found.
     */
    public int removeMod(Mod mod) {
        Objects.requireNonNull(mod);

        int index = this.MODLIST.indexOf(mod);
        this.MODLIST.remove(mod);
        return index;
    }

    /**
     * @see List#indexOf(java.lang.Object)
     * @param mod
     * @return
     */
    public int indexOf(Mod mod) {
        return this.MODLIST.indexOf(mod);
    }

    /**
     * @see List#clear()
     */
    public void clear() {
        this.MODLIST.clear();
    }

    /**
     * Retrieves the fast-analysis of the whole <code>Profile</code>. The
     * analysis will only include those mods that have already been analyzed.
     *
     * @return The (potentially partial) fast analysis of the profile.
     */
    public Mod.Analysis getFastAnalysis() {
        Mod.Analysis total = new Mod.Analysis();

        for (Mod mod : this.getCheckmarkedMods()) {
            Mod.Analysis subAnalysis = mod.fastAnalysis();
            if (null != subAnalysis) {
                total = Mod.Analysis.combine(total, subAnalysis);
            }
        }

        return total;
    }

    /**
     * Updates the last-built field to the current time.
     */
    public void updateBuildTime() {
        this.buildTime = new Date().getTime();
    }

    /**
     * @return The last time at which the profile was restrung or null if the
     * profile has never been successfully built.
     */
    public Date getBuildTime() {
        if (this.buildTime < 0) {
            return null;
        } else {
            return new Date(this.buildTime);
        }
    }

    /**
     * @return The output directory.
     */
    public File getOutputDirectory() {
        return this.outputDirectory;
    }

    /**
     * Sets the output directory.
     *
     * @param dir The new path for the output directory.
     * @throws IllegalArgumentException If the new directory already exists and
     * is actually a file, an exception will be thrown.
     */
    public void setOutputDirectory(File dir) {
        if (dir.exists() && !dir.isDirectory()) {
            throw new IllegalArgumentException();
        }

        this.outputDirectory = dir;
    }

    /**
     * @return String representation of the <code>Profile</code>.
     */
    @Override
    public String toString() {
        return this.NAME;
    }

    /**
     * Generates a patch. Skips any script that is detected to be the parent of
     * another script.
     *
     * @param model The progress model.
     * @param compress A flag indicating whether the compress the scripts.
     * @return The remapping stats.
     * @throws RestringingError
     */
    public RemappingStats execute2(ProgressModel model, boolean compress) throws RestringingError {
        /*Objects.requireNonNull(model);

        //-------------------------------------------------------------
        // Make the patch directory and scripts subdirectory if they don't 
        // already exist.
        final File PATCH_DIR = this.getOutputDirectory();
        if (!PATCH_DIR.exists()) {
            if (!PATCH_DIR.mkdir()) {
                JOptionPane.showMessageDialog(null, "Couldn't create output directory:\n" + PATCH_DIR, "Error", JOptionPane.ERROR_MESSAGE);
                System.exit(-1);
            }
        }

        final File PATCH_SCRIPTS_DIR = new File(PATCH_DIR, SCRIPTS_PATH);
        if (!compress && !PATCH_SCRIPTS_DIR.exists()) {
            if (!PATCH_SCRIPTS_DIR.mkdir()) {
                JOptionPane.showMessageDialog(null, "Couldn't create output directory:\n" + PATCH_SCRIPTS_DIR, "Error", JOptionPane.ERROR_MESSAGE);
                System.exit(-1);
            }
        }

        //-------------------------------------------------------------
        // Retrieve the full list of mods.
        final List<Mod> MODS = this.getMods();

        // Retrieve the list of mods to be included in the patch.
        final List<Mod> INMODS = this.getCheckmarkedMods();

        // Retrieve the list of mods to be excluded.
        final List<Mod> EXMODS = new LinkedList<>(this.getMods());
        EXMODS.removeAll(INMODS);

        //-------------------------------------------------------------
        // Use this for displaying progress data.       
        final double WT_PREFACE = 6.20;
        final double WT_READ = 6.978922817940983 / 1048576.0; // per total MB
        final double WT_SORT1 = 0.0681829618272595; // per script
        final double WT_SORT2 = 0.01710972205841884; // per script
        final double WT_REMAP = 0.9040582615874513; // per script
        final double WT_REBUILD = 0.08480677752879663; // per script
        final double WT_WRITE = 4.914670755199119; // per script
        final double WT_ZIP = 8.778613043991037; // per script

        Mod.Analysis analysis = new Mod.Analysis();
        for (Mod mod : MODS) {
            analysis = Mod.Analysis.combine(analysis, mod.analysis());
        }

        double max = WT_PREFACE;
        max += analysis.NUMBYTES * WT_READ;
        max += analysis.NUMSCRIPTS * (WT_SORT1 + WT_SORT2 + WT_REBUILD + WT_REMAP);
        max += analysis.NUMSCRIPTS * WT_WRITE;
        //max += analysis.NUMSCRIPTS * WT_ZIP;

        model.setValueIsAdjusting(true);
        model.setMaximum((int) max);
        model.setMinimum(0);
        model.setValue(0);
        model.setValueIsAdjusting(false);

        LOG.info(String.format("Total bytes %d, total scripts %d.", analysis.NUMBYTES, analysis.NUMSCRIPTS));

        model.modifyValue(WT_PREFACE);

        //-------------------------------------------------------------
        // Collect all the script files from BSAs into their respective mods' 
        // script lists.
        LOG.info("Reading Mod scripts.");
        TIMER.start();

        final List<Mod.ScriptReadError> READERRORS = new LinkedList<>();

        MODS.parallelStream().forEach(mod -> {
            try {
                // Read the mod's scripts.
                mod.readScripts(false);

                // Update the progress model.
                model.modifyValue(mod.analysis().NUMBYTES * WT_READ);

            } catch (Mod.ScriptReadError ex) {
                READERRORS.add(ex);
            }
        });

        if (!READERRORS.isEmpty()) {
            throw new RestringingReadError(READERRORS);
        }

        TIMER.stop();
        LOG.info(String.format("Required %s to read %d mods.", TIMER.getFormattedTime(), this.MODLIST.size()));
        TIMER.reset();

        System.out.println(analysis);

        //-------------------------------------------------------------
        LOG.info("Remapping strings.");
        TIMER.start();

        // Inheritance sort. :-)
        // Create a name->identity map to resolve conflicts by compilation date.
        final Map<IString, PexFile> IDENTITY = new HashMap<>(analysis.NUMSCRIPTS);
        final Map<IString, PexFile> EXCLUSION = new HashMap<>(analysis.NUMSCRIPTS);

        EXMODS.forEach(mod -> {
            mod.getScripts().values()
                    .stream()
                    .forEach(v -> EXCLUSION.put(v.getFilename(), v));
        });

        INMODS.forEach(mod -> {
            mod.getScripts().values().forEach(script -> {
                // We'll need the script name for matching.
                final IString SCRIPTNAME = script.getFilename();
                
                // Check if this script is overwriting a script from the exclusion list.
                if (EXCLUSION.containsKey(SCRIPTNAME)) {
                    // Do nothing. Just skip it completely.
                    LOG.info(String.format("Skipping %s, overwrites a script from an excluded mod.", SCRIPTNAME));
                }

                // Check for overwrite.
                if (IDENTITY.containsKey(SCRIPTNAME)) {
                    LOG.info(String.format("Overwriting %s with a higher priority version from %s.", SCRIPTNAME, mod.getName()));
                }

                IDENTITY.put(SCRIPTNAME, script);
            });
        });

        final List<PexFile> SCRIPTS = new ArrayList<>(IDENTITY.values());

        LOG.info("Remapping strings: made identity map.");

        // Update the progress model.
        model.modifyValue(analysis.NUMSCRIPTS * WT_SORT1);

        // Put all the scripts into a pool.
        final Map<IString, Pex> POOL = new LinkedHashMap<>();
        SCRIPTS.stream().forEach(v -> POOL.put(v.getFilename(), v));
        LOG.info("Remapping strings: made script pool.");

        // If any scripts in the pool are ancestors of scripts in the 
        // exclusion list or the pool, move them to the exclusion list.
        Set<Pex> ANCESTORS;

        do {
            ANCESTORS = EXCLUSION
                    .values()
                    .parallelStream()
                    .filter(v -> POOL.containsKey(v.PARENTNAME))
                    .map(v -> POOL.get(v.PARENTNAME))
                    .collect(Collectors.toSet());
            ANCESTORS.forEach(v -> EXCLUSION.put(v.NAME, POOL.remove(v.NAME)));
            ANCESTORS.forEach(v -> LOG.info(String.format("Skipping %s, parent of a script from an excluded mod.", v.NAME)));
        } while (!ANCESTORS.isEmpty());

        do {
            ANCESTORS = POOL
                    .values()
                    .parallelStream()
                    .filter(v -> POOL.containsKey(v.PARENTNAME))
                    .map(v -> POOL.get(v.PARENTNAME))
                    .collect(Collectors.toSet());
            ANCESTORS.forEach(v -> EXCLUSION.put(v.NAME, POOL.remove(v.NAME)));
            ANCESTORS.forEach(v -> LOG.info(String.format("Skipping %s, parent of a script in the patching pool.", v.NAME)));
        } while (!ANCESTORS.isEmpty());

        // AKA Kahn's algorithm. Simple, brute force, does the job.           
        // This guarantees that no object in ANCESTRAL_ORDER precedes its
        // parent class, vastly simplifying what is to come.
        final List<Pex> KAHN_TABLE = new java.util.LinkedList<>();

        while (!POOL.isEmpty()) {
            // Get some arbitrary object from the pool.
            Pex pexObj = POOL.values().stream().findAny().get();
            IString objectName = pexObj.NAME;
            IString parentName = pexObj.PARENTNAME;

            // Follow its ancestry to find an object whose parent is 
            // not in POOL.
            while (POOL.containsKey(parentName)) {
                objectName = parentName;
                pexObj = POOL.get(parentName);
                parentName = pexObj.PARENTNAME;
            }

            // Add the parentless object to the ordering.
            KAHN_TABLE.add(pexObj);
            POOL.remove(objectName);
        }

        TIMER.stop();
        LOG.info(String.format("Required %s to perform Kahn's algorithm on %d scripts.", TIMER.getFormattedTime(), SCRIPTS.size()));
        TIMER.reset();

        // Update the progress model.
        model.modifyValue(WT_SORT2 * analysis.NUMSCRIPTS);

        //-------------------------------------------------------------
        // Do the remapping.
        // Autovar renaming scheme and name generator.
        TIMER.start();
        final Map<IString, Scheme> AUTOVAR_SCHEMES = new HashMap<>();
        final Map<IString, TokenGenerator> AUTOVAR_GENS = new HashMap<>();
        final RemappingStats STATS = new RemappingStats();

        // No parallelism here; this has to be done in order. Sort of.
        // Skip anything on the exlusion list. 
        KAHN_TABLE
                .stream()
                .filter(script -> !EXCLUSION.containsKey(script.NAME))
                .forEach(script -> {
                    // Remap the variables.
                    script.restring2(true, true, false, false, false);
                    LOG.fine(String.format("Remapping strings: remapped %d of %d (%s).", STATS.getScripts(), KAHN_TABLE.size(), script.NAME));
                });

        // Filter for determining which scripts have objects in the Kahn table.
        Predicate<PexFile> KAHNFILTER = o -> KAHN_TABLE.contains(o.OBJECT) && !EXCLUSION.containsKey(o.OBJECT.NAME);

        TIMER.stop();
        LOG.info(String.format("Remapping strings: required %s to finish remapping %d scripts.", TIMER.getFormattedTime(), KAHN_TABLE.size()));
        TIMER.reset();

        // Update the progress model.
        model.modifyValue(WT_REMAP * analysis.NUMSCRIPTS);

        //-------------------------------------------------------------
        // Rebuild function and property calls and the string table.
        LOG.info("Rebuilding string tables.");
        TIMER.start();

        // Rebuild the string tables for any scripts who Pex is in the
        // Kahn table.
        SCRIPTS.parallelStream().filter(KAHNFILTER).forEach(script -> {
            script.rebuildStringTable();
        });
        LOG.info("Remapping strings: rebuilt the string table.");

        TIMER.stop();
        LOG.info(String.format("Required %s to finish rebuilding %d string tables.", TIMER.getFormattedTime(), SCRIPTS.size()));
        TIMER.reset();

        // Update the progress model.
        model.modifyValue(WT_REBUILD * analysis.NUMSCRIPTS);

        //-------------------------------------------------------------
        // Write the scripts to the patch directory.  
        LOG.info(String.format("Writing %d scripts.", SCRIPTS.size()));
        TIMER.start();

        final List<IOException> WRITEERRORS = new LinkedList<>();

        if (compress) {
            final String ARCHIVE_NAME = String.format("ReString - %s.7z", this.NAME);
            final File ARCHIVE = new File(outputDirectory, ARCHIVE_NAME);

            try (final SevenZOutputFile SEVENZ = new SevenZOutputFile(ARCHIVE);
                    final OutputStream SEVENZOUT = new SevenZOutputStream(SEVENZ);
                    final DataOutputStream DATAOUT = new DataOutputStream(SEVENZOUT)) {
                SEVENZ.setContentCompression(SevenZMethod.DEFLATE);

                SCRIPTS.stream().filter(KAHNFILTER).forEach(script -> {
                    String filename = script.getFilename().toString();
                    File file = new File(PATCH_SCRIPTS_DIR, filename);

                    try {
                        final String FILEPATH = SCRIPTS_PATH + file.getName();
                        final SevenZArchiveEntry ENTRY = SEVENZ.createArchiveEntry(file, FILEPATH);
                        SEVENZ.putArchiveEntry(ENTRY);

                        script.write(DATAOUT);
                        DATAOUT.flush();
                        SEVENZ.closeArchiveEntry();

                        // Update the progress model.
                        model.modifyValue(WT_WRITE);

                    } catch (IOException ex) {
                        WRITEERRORS.add(ex);
                    }
                });

                SEVENZ.finish();

            } catch (IOException ex) {
                throw new RestringingWriteError(Collections.emptyList());
            }

        } else {
            SCRIPTS.parallelStream().filter(KAHNFILTER).forEach(script -> {
                String filename = script.getFilename().toString();
                File outputFile = new File(PATCH_SCRIPTS_DIR, filename);

                try {
                    PexFile.writeScript(script, outputFile);

                    // Update the progress model.
                    model.modifyValue(WT_WRITE);

                } catch (IOException ex) {
                    WRITEERRORS.add(ex);
                }
            });
        }

        if (!WRITEERRORS.isEmpty()) {
            throw new RestringingWriteError(WRITEERRORS);
        }

        final File DIGEST = new File(outputDirectory, "Summary.txt");
        try (PrintStream out = new PrintStream(DIGEST)) {
            out.printf("Patch name: %s\n", this.NAME);
            out.printf("Patch built on %s\n", new Date(this.buildTime).toString());
            out.println("The following mods were included in the patch:");
            out.println();
            INMODS.forEach(v -> out.println(v));

            out.println("---------------------------------\n");
            out.println("The following scripts were included in the patch:");
            out.println();
            SCRIPTS.stream()
                    .filter(KAHNFILTER)
                    .sorted((a, b) -> a.getFilename().compareTo(b.getFilename()))
                    .forEach(script -> out.println(script.getFilename()));
        } catch (IOException ex) {
            LOG.warning("Couldn't write the summary file.");
        }

        TIMER.stop();
        LOG.info(String.format("Required %s to write %d scripts.", TIMER.getFormattedTime(), SCRIPTS.size()));
        TIMER.reset();

        this.updateBuildTime();

        LOG.info(String.join("==STATS==\n\n", STATS.toString()));
        return STATS;
         */
        return null;
    }

    /**
     *
     * @param mod
     * @return
     */
    static public Analysis analyzeMod(Mod mod) {
        // Read scripts.
        if (mod.getScripts().isEmpty()) {
            try {
                mod.readScripts(true);
            } catch (Mod.ScriptReadError ex) {

            }
        }

        final Analysis ANALYSIS = new Analysis();
        final String MODNAME = mod.getName();
        ANALYSIS.MODS.add(mod);
        ANALYSIS.ESPS.put(MODNAME, new ObjectLinkedOpenHashSet<>(mod.getESPNames()));

        mod.getScripts().forEach((file, pexFile) -> {
            pexFile.STRINGS.parallelStream().forEach(string -> {
                Map<IString, Map<String, Integer>> stringOrigins = ANALYSIS.STRING_ORIGINS;
                Map<String, Integer> modCounts = stringOrigins.computeIfAbsent(string, s -> new it.unimi.dsi.fastutil.objects.Object2IntLinkedOpenHashMap<>(6));
                modCounts.merge(MODNAME, 1, Integer::sum);
            });

            pexFile.SCRIPTS.forEach(pex -> {
                ANALYSIS.SCRIPTS.put(pex.NAME, file);
                Map<IString, SortedSet<String>> scriptOrigins = ANALYSIS.SCRIPT_ORIGINS;
                SortedSet<String> scriptMods = scriptOrigins.computeIfAbsent(pex.NAME, s -> new ObjectLinkedOpenHashSet<>());
                scriptMods.add(MODNAME);

                pex.getFunctionNames().forEach(name -> {
                    Map<IString, SortedSet<String>> functionOrigins = ANALYSIS.FUNCTION_ORIGINS;
                    SortedSet<String> funcMods = functionOrigins.computeIfAbsent(name, s -> new ObjectLinkedOpenHashSet<>());
                    funcMods.add(MODNAME);
                });
            });
        });

        return ANALYSIS;
    }

    /**
     * Generates a patch.
     *
     * @param model The progress model.
     * @return A map of stuff to mod it probably came from.
     * @throws RestringingError
     */
    public Analysis createAnalysisMap(ProgressModel model) throws RestringingError {
        Objects.requireNonNull(model);

        //-------------------------------------------------------------
        final List<Mod> MODS = this.getCheckmarkedMods();

        //-------------------------------------------------------------
        // Use this for displaying progress data.       
        final double WT_PREFACE = 6.20;
        final double WT_READ = 6.978922817940983 / 1048576.0; // per total MB
        final double WT_SORT1 = 0.0681829618272595; // per script

        Mod.Analysis analysis = new Mod.Analysis();
        for (Mod mod : MODS) {
            analysis = Mod.Analysis.combine(analysis, mod.analysis());
        }

        double max = WT_PREFACE;
        max += analysis.NUMBYTES * WT_READ;
        max += analysis.NUMSCRIPTS * WT_SORT1;

        model.setValueIsAdjusting(true);
        model.setMaximum((int) max);
        model.setMinimum(0);
        model.setValue(0);
        model.setValueIsAdjusting(false);

        LOG.info(String.format("Total bytes %d, total scripts %d.", analysis.NUMBYTES, analysis.NUMSCRIPTS));

        model.modifyValue(WT_PREFACE);

        //-------------------------------------------------------------
        // Collect all the script files from BSAs into their respective mods' 
        // script lists.
        LOG.info("Reading Mod scripts.");
        TIMER.start();

        final List<Mod.ScriptReadError> READERRORS = new LinkedList<>();

        MODS.parallelStream().forEach(mod -> {
            try {
                // Read the mod's scripts.
                mod.readScripts(false);

                // Update the progress model.
                model.modifyValue(mod.analysis().NUMBYTES * WT_READ);

            } catch (Mod.ScriptReadError ex) {
                READERRORS.add(ex);
            }
        });

        if (!READERRORS.isEmpty()) {
            throw new RestringingReadError(READERRORS);
        }

        TIMER.stop();
        LOG.info(String.format("Required %s to read %d mods.", TIMER.getFormattedTime(), this.MODLIST.size()));
        TIMER.reset();

        System.out.println(analysis);

        //-------------------------------------------------------------
        LOG.info("Sorting strings.");
        TIMER.start();

        // Inheritance sort. :-)
        // Create a name->identity map. Resolve conflicts by compilation date.
        final Map<IString, PexFile> SCRIPTS = new LinkedHashMap<>();
        MODS.forEach(mod -> mod.getScripts().values().forEach(script -> {
            IString filename = script.getFilename();

            // Check if this script is overwriting another one.
            if (SCRIPTS.containsKey(filename)) {
                long originalDate = SCRIPTS.get(filename).getDate();
                long newDate = script.getDate();

                if (newDate > originalDate) {
                    SCRIPTS.put(filename, script);
                }
            } else {
                SCRIPTS.put(filename, script);
            }

        }));
        LOG.info("Remapping strings: made identity map.");

        // Update the progress model.
        model.modifyValue(analysis.NUMSCRIPTS * WT_SORT1);

        TIMER.start();
        final Analysis ANALYSIS = new Analysis();

        MODS.forEach(mod -> {
            final String MODNAME = mod.getName();
            ANALYSIS.MODS.add(mod);
            ANALYSIS.ESPS.put(MODNAME, new ObjectLinkedOpenHashSet<>(mod.getESPNames()));

            mod.getScripts().forEach((file, pexFile) -> {
                pexFile.STRINGS.parallelStream().forEach(string -> {
                    Map<IString, Map<String, Integer>> stringOrigins = ANALYSIS.STRING_ORIGINS;
                    Map<String, Integer> modCounts = stringOrigins.computeIfAbsent(string, s -> new it.unimi.dsi.fastutil.objects.Object2IntLinkedOpenHashMap<>(6));
                    modCounts.merge(MODNAME, 1, Integer::sum);
                });

                pexFile.SCRIPTS.forEach(pex -> {
                    ANALYSIS.SCRIPTS.put(pex.NAME, file);
                    Map<IString, SortedSet<String>> scriptOrigins = ANALYSIS.SCRIPT_ORIGINS;
                    SortedSet<String> scriptMods = scriptOrigins.computeIfAbsent(pex.NAME, s -> new ObjectLinkedOpenHashSet<>());
                    scriptMods.add(MODNAME);

                    pex.getFunctionNames().forEach(name -> {
                        Map<IString, SortedSet<String>> functionOrigins = ANALYSIS.FUNCTION_ORIGINS;
                        SortedSet<String> funcMods = functionOrigins.computeIfAbsent(name, s -> new ObjectLinkedOpenHashSet<>());
                        funcMods.add(MODNAME);
                    });
                });
            });
        });

        TIMER.stop();
        LOG.info(String.format("Required %s to analyse %d scripts over %d mods.", TIMER.getFormattedTime(), SCRIPTS.size(), MODS.size()));
        TIMER.reset();

        return ANALYSIS;
    }

    final private String NAME;
    final private ArrayList<Mod> MODLIST;
    private File outputDirectory;
    private long buildTime;
    transient final private Timer TIMER;

    static final private Logger LOG = Logger.getLogger(Profile.class.getCanonicalName());
    static final private String SCRIPTS_PATH = "scripts\\";
    static final private Pattern FILENAME_PATTERN = Pattern.compile(
            "# Match a valid Windows filename (unspecified file system).          \n"
            + "^                                # Anchor to start of string.        \n"
            + "(?!                              # Assert filename is not: CON, PRN, \n"
            + "  (?:                            # AUX, NUL, COM1, COM2, COM3, COM4, \n"
            + "    CON|PRN|AUX|NUL|             # COM5, COM6, COM7, COM8, COM9,     \n"
            + "    COM[1-9]|LPT[1-9]            # LPT1, LPT2, LPT3, LPT4, LPT5,     \n"
            + "  )                              # LPT6, LPT7, LPT8, and LPT9...     \n"
            + "  (?:\\.[^.]*)?                  # followed by optional extension    \n"
            + "  $                              # and end of string                 \n"
            + ")                                # End negative lookahead assertion. \n"
            + "[^<>:\"/\\\\|?*\\x00-\\x1F]*     # Zero or more valid filename chars.\n"
            + "[^<>:\"/\\\\|?*\\x00-\\x1F\\ .]  # Last char is not a space or dot.  \n"
            + "$                                # Anchor to end of string.            ",
            Pattern.CASE_INSENSITIVE | Pattern.UNICODE_CASE | Pattern.COMMENTS);

    /**
     * Compares two files, for testing purposes.
     *
     * Exceptions are not handled. At all. Not even a little bit.
     *
     * @param f1 The first file.
     * @param f2 The second file.
     *
     * @throws FileNotFoundException
     * @throws IOException
     *
     */
    static public void compareFiles(File f1, File f2) throws FileNotFoundException, IOException {
        assert f1.isFile();
        assert f1.canRead();
        assert f1.exists();
        assert f2.isFile();
        assert f2.canRead();
        assert f2.exists();

        try (
                BufferedInputStream bis1 = new BufferedInputStream(new FileInputStream(f1));
                BufferedInputStream bis2 = new BufferedInputStream(new FileInputStream(f2))) {

            final byte[] BUF1 = new byte[1024];
            final byte[] BUF2 = new byte[1024];
            int position = 0;
            int mismatches = 0;

            System.out.println();
            while (0 < bis1.available() && 0 < bis2.available() && mismatches < 10) {
                bis1.read(BUF1);
                bis2.read(BUF2);

                if (!Arrays.equals(BUF1, BUF2)) {
                    for (int offset = 0; offset < BUF1.length; offset++) {
                        if (BUF1[offset] != BUF2[offset]) {
                            int index = position + offset;
                            System.out.printf("Mismatch: %06x : %02x -> %02x\n", index, BUF1[offset], BUF2[offset]);
                            mismatches++;
                        }
                    }
                }

                position += BUF1.length;
            }

            if (mismatches == 0) {
                System.out.printf("\nPerfect match.\n");
            } else {
                System.out.printf("\nThere were %d mismatches.\n", mismatches);
            }
        }
    }

    /**
     * Validates a profile name.
     *
     * @param name The name.
     * @return True if the string is a valid profile name, false otherwise.
     */
    static public boolean validateName(String name) {
        return Profile.FILENAME_PATTERN.matcher(name).matches();
    }

    /**
     * Stores data relating strings, scripts, and functions to their
     *
     * @author Mark Fairchild
     * @version 2016/07/07
     */
    static public class Analysis implements java.io.Serializable {

        /**
         * List: (Mod name)
         */
        final public SortedSet<Mod> MODS = new ObjectLinkedOpenHashSet<>();

        /**
         * Map: (IString) -> File
         */
        final public Map<IString, File> SCRIPTS = new ConcurrentHashMap<>();

        /**
         * Map: (Mod name) -> (Lisp[ESP name])
         */
        final public Map<String, SortedSet<String>> ESPS = new ConcurrentHashMap<>();

        /**
         * Map: (IString) -> (Map: (Mod name) -> (# of ocurrences))
         */
        final public Map<IString, Map<String, Integer>> STRING_ORIGINS = new ConcurrentHashMap<>();

        /**
         * Map: (IString) -> (List: (Mod name))
         */
        final public Map<IString, SortedSet<String>> SCRIPT_ORIGINS = new ConcurrentHashMap<>();

        /**
         * Map: (IString) -> (List: (Mod name))
         */
        final public Map<IString, SortedSet<String>> STRUCT_ORIGINS = new ConcurrentHashMap<>();

        /**
         * Map: (IString) -> (List: (Mod name))
         */
        final public Map<IString, SortedSet<String>> FUNCTION_ORIGINS = new ConcurrentHashMap<>();

        /**
         * Merges analyses.
         *
         * @param sub
         * @return
         */
        public Analysis merge(Analysis sub) {
            this.MODS.addAll(sub.MODS);
            this.SCRIPTS.putAll(sub.SCRIPTS);

            sub.ESPS.forEach((name, list) -> {
                this.ESPS.merge(name, list, (l1, l2) -> {
                    l1.addAll(l2);
                    return l1;
                });
            });

            sub.STRING_ORIGINS.forEach((name, list) -> {
                this.STRING_ORIGINS.merge(name, list, (l1, l2) -> {
                    l1.putAll(l2);
                    return l1;
                });
            });

            sub.SCRIPT_ORIGINS.forEach((name, list) -> {
                this.SCRIPT_ORIGINS.merge(name, list, (l1, l2) -> {
                    l1.addAll(l2);
                    return l1;
                });
            });

            sub.STRUCT_ORIGINS.forEach((name, list) -> {
                this.STRUCT_ORIGINS.merge(name, list, (l1, l2) -> {
                    l1.addAll(l2);
                    return l1;
                });
            });

            sub.FUNCTION_ORIGINS.forEach((name, list) -> {
                this.FUNCTION_ORIGINS.merge(name, list, (l1, l2) -> {
                    l1.addAll(l2);
                    return l1;
                });
            });

            return this;
        }
    }

    /**
     * An exception that indicates an error while restringing.
     */
    abstract public class RestringingError extends IOException {

        private RestringingError(String msg) {
            super(msg);
        }
    }

    /**
     * An exception that indicates an error while reading scripts and BSAs.
     */
    public class RestringingReadError extends RestringingError {

        /**
         * Creates a new <code>RestringingReadError</code> for script read
         * errors.
         *
         * @param causes A list of script read errors.
         *
         * @see Exception#Exception()
         */
        private RestringingReadError(Collection<Mod.ScriptReadError> causes) {
            super("Error while reading scripts.");
            Objects.requireNonNull(causes);
            this.CAUSES = new ArrayList<>(causes);
        }

        /**
         * Returns the collection of causes of the error.
         *
         * @return A read-only collection of <code>Mod.ScriptReadError</code>.
         */
        public Collection<Mod.ScriptReadError> getCauses() {
            return Collections.unmodifiableCollection(this.CAUSES);
        }

        final private Collection<Mod.ScriptReadError> CAUSES;
    }

    /**
     * An exception that indicates an error while writing scripts.
     */
    public class RestringingWriteError extends RestringingError {

        /**
         * Creates a new <code>RestringingWriteError</code> for script write
         * errors.
         *
         * @param causes A list of IOExceptions.
         *
         * @see Exception#Exception()
         */
        private RestringingWriteError(Collection<IOException> causes) {
            super("Error while writing patched scripts.");
            Objects.requireNonNull(causes);
            this.CAUSES = new ArrayList<>(causes);
        }

        /**
         * Returns the collection of causes of the error.
         *
         * @return A read-only collection of <code>IOException</code>.
         */
        public Collection<IOException> getCauses() {
            return Collections.unmodifiableCollection(this.CAUSES);
        }

        final private Collection<IOException> CAUSES;
    }

}
