/*
 * Copyright 2016 Mark.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer;

import java.awt.BorderLayout;
import java.io.*;
import java.util.LinkedList;
import java.util.logging.*;
import restringer.pex.*;
import org.apache.commons.cli.*;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.JOptionPane;

/**
 * Entry class for ReAssembler.
 *
 * @author Mark Fairchild
 * @version 2016/09/05
 */
public class ReAssembler {

    /**
     * @param line The parsed command line.
     */
    public static void execute(CommandLine line) {
        // Set up the logger.
        try {
            LOG.getParent().addHandler(new FileHandler("ReAssembler.%g.log.xml", 1 << 20, 1));
        } catch (IOException ex) {
            LOG.warning("Couldn't create log file.");
            LOG.warning(ex.toString());
        }

        LOG.getParent().getHandlers()[0].setFormatter(new java.util.logging.Formatter() {
            @Override
            public String format(LogRecord record) {
                final java.util.logging.Level LEVEL = record.getLevel();
                final String MSG = record.getMessage();
                final String SRC = record.getSourceClassName() + "." + record.getSourceMethodName();
                return String.format("%s: %s (in %s)\n", LEVEL, MSG, SRC);
            }
        });

        LOG.setLevel(Level.FINE);
        LOG.getParent().getHandlers()[0].setLevel(Level.FINE);

        // Check the command line options to decide what we need to do.
        final AssemblyLevel LEVEL;
        if (line.hasOption('c')) {
            LEVEL = AssemblyLevel.STRIPPED;
        } else if (line.hasOption('b')) {
            LEVEL = AssemblyLevel.BYTECODE;
        } else {
            LEVEL = AssemblyLevel.FULL;
        }

        // The console flag indicates that we should output the script to 
        // the console (the standard output) instead of to a file.
        final boolean CONSOLE = line.hasOption('s');

        // Everything else should be a filename.
        final List<String> FILES = line.getArgList();

        // If there's no console and there IS a windowing system of some kind, 
        // operate graphically.
        if (null == System.console() && !java.awt.GraphicsEnvironment.isHeadless()) {
            if (FILES.isEmpty()) {
                JOptionPane.showMessageDialog(null, "No script files!", "Error", JOptionPane.ERROR_MESSAGE);
                return;
            }

            final String DOC1 = makeDoc(FILES, AssemblyLevel.FULL);
            final String DOC2 = makeDoc(FILES, AssemblyLevel.STRIPPED);
            final String DOC3 = makeDoc(FILES, AssemblyLevel.BYTECODE);            
            final DisasmPane PANE1 = new DisasmPane(DOC1);
            final DisasmPane PANE2 = new DisasmPane(DOC2);
            final DisasmPane PANE3 = new DisasmPane(DOC3);
            final javax.swing.JTabbedPane PANE = new javax.swing.JTabbedPane();
            PANE.add("Source", new javax.swing.JScrollPane(PANE1));
            PANE.add("Stripped", new javax.swing.JScrollPane(PANE2));
            PANE.add("ByteCode", new javax.swing.JScrollPane(PANE3));            
            PANE.setPreferredSize(new java.awt.Dimension(850, 700));
            //javax.swing.JPanel P = new javax.swing.JPanel();
            //P.add(BorderLayout.NORTH, PANE);
            JOptionPane.showMessageDialog(null, PANE, "Disassembly", JOptionPane.INFORMATION_MESSAGE);

        } // Run in the console.
        else {

            // Go through the list of files one by one.
            for (String filename : FILES) {
                // Check the file extension, see if it's actually a PEX file.
                // Skip anything that isn't, and output an error message.
                final File SCRIPTFILE = new File(filename);
                final Matcher REGEX = FILENAMEPATTERN.matcher(SCRIPTFILE.getName());
                if (!REGEX.matches()) {
                    System.err.println("Skipping invalid filename: " + filename);
                    continue;
                }

                // Read the script.
                final PexFile SCRIPT;
                try {
                    SCRIPT = PexFile.readScript(SCRIPTFILE);
                } catch (IOException ex) {
                    System.err.println("Error reading script: " + ex.getMessage());
                    continue;
                }

                // The disassembled code will go into this array.
                final List<String> CODE = new LinkedList<>();
                try {
                    SCRIPT.disassemble(CODE, LEVEL);
                } catch (RuntimeException ex) {
                    System.err.println("Error disassembling script: " + ex.getMessage());
                    continue;
                }

                // In console mode, output to the console. Duh.
                if (CONSOLE) {
                    CODE.forEach(s -> System.out.println(s));

                } // In file mode (the default) output to a new file.
                else {

                    // Choose a name for the output file.
                    final File OUTFILE;
                    if (LEVEL != AssemblyLevel.FULL) {
                        OUTFILE = new File(filename.replaceFirst("\\.pex$", ".txt"));
                    } else {
                        OUTFILE = new File(filename.replaceFirst("\\.pex$", ".psc"));
                    }

                    // Make sure that we're not overwriting anything.
                    if (OUTFILE.exists()) {
                        System.err.printf("Output file already exists: %s\n", OUTFILE);
                    }

                    // Write the file.
                    try (Writer OUT = new BufferedWriter(new FileWriter(OUTFILE))) {
                        for (String s : CODE) {
                            OUT.write(s);
                            OUT.write('\n');
                        }
                    } catch (IOException ex) {
                        System.err.println("Error disassembling script: " + ex.getMessage());
                        continue;
                    }

                    // Output a message when everything is finished.
                    if (!CONSOLE) {
                        switch (LEVEL) {
                            case FULL:
                                System.out.println("Wrote disassembled script for " + filename + " to " + OUTFILE);
                                break;
                            case STRIPPED:
                                System.out.println("Wrote cleaned-bytecode script for " + filename + " to " + OUTFILE);
                                break;
                            case BYTECODE:
                                System.out.println("Wrote bytecode-level script for " + filename + " to " + OUTFILE);
                                break;
                            default:
                        }
                    }
                }
            }
        }
    }

    /**
     * Turns a list of PEX files into an HTML document.
     * 
     * @param files
     * @param level
     * @return 
     */
    static private String makeDoc(List<String> files, AssemblyLevel level) {
        // Print everything into a PrintWriter wrapped around a 
        // StringWriter. 
        final StringWriter BUF = new StringWriter(20480);
        final PrintWriter OUT = new PrintWriter(BUF);
        OUT.println("<html>");

        // Go through the list of files one by one.
        for (String filename : files) {
            // Check the file extension, see if it's actually a PEX file.
            // Skip anything that isn't, and output an error message.
            final File SCRIPTFILE = new File(filename);
            final Matcher REGEX = FILENAMEPATTERN.matcher(SCRIPTFILE.getName());
            if (!REGEX.matches()) {
                OUT.printf("<p>Skipped invalid filename: %s\n</p>", filename);
                continue;
            }

            // Read the script.
            final PexFile SCRIPT;
            try {
                SCRIPT = PexFile.readScript(SCRIPTFILE);
            } catch (IOException ex) {
                OUT.printf("<p>Error reading script: %s\n</p>", filename);
                continue;
            }

            // The disassembled code will go into this array.
            final List<String> CODE = new LinkedList<>();
            try {
                SCRIPT.disassemble(CODE, level);
            } catch (Exception | Error ex) {
                ex.printStackTrace(System.err);
                OUT.printf("<p>Error disassembling script: %s\n</p>", filename);
            }

            // Output the code to the PrintWriter.
            OUT.println("<code><pre>");
            CODE.forEach(s -> {
                String s2 = s.replace("<", "%lt;");
                OUT.println(s2);
            });
            OUT.println("</pre></code>");
            OUT.println("<hr/>");
        }
        OUT.println("</html>");

        return BUF.toString();
    }

    static private class DisasmPane extends javax.swing.JTextPane {
        public DisasmPane(String doc) {
            super.setFont(super.getFont().deriveFont(12.0f));
            super.setEditable(true);
            super.setContentType("text/html");
            super.setPreferredSize(new java.awt.Dimension(850, 700));
            super.setText(doc);
            
        }
    }
    static final Logger LOG = Logger.getLogger(ReAssembler.class.getCanonicalName());
    static final Pattern FILENAMEPATTERN = Pattern.compile("^.*\\.(pex)$", Pattern.CASE_INSENSITIVE);

}
