/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Objects;
import org.apache.commons.compress.archivers.sevenz.SevenZOutputFile;

/**
 * An <code>OutputStream</code> that wraps a
 * <code>SevenZOutputFile</code>.
 *
 * @author Mark Fairchild
 * @version 2016/07/07
 */
public class SevenZOutputStream extends OutputStream {

    /**
     * Creates a new <code>SevenZOutputStream</code> that wraps around a supplied
     * <code>SevenZOutputFile</code>.
     *
     * @param sevenz The <code>SevenZOutputFile</code> to wrap.
     */
    public SevenZOutputStream(SevenZOutputFile sevenz) {
        Objects.requireNonNull(sevenz);
        this.SEVENZ = sevenz;
    }

    /**
     * Accessor for the underlying <code>SevenZOutputFile</code>.
     *
     * @return The underlying file.
     */
    public SevenZOutputFile getSevenZ() {
        return this.SEVENZ;
    }

    /**
     * @see java.io.OutputStream#close()
     * @throws IOException
     */
    @Override
    public void close() throws IOException {
        this.SEVENZ.close();
    }

    /**
     * @see java.io.OutputStream#write(int)
     * @see SevenZOutputFile#write(int)
     * 
     * @throws IOException
     */
    @Override
    public void write(int b) throws IOException {
        this.SEVENZ.write(b);
        this.position++;
    }

    /**
     * @see java.io.OutputStream#write(byte[])
     * @see SevenZOutputFile#write(byte[])
     * @throws IOException
     */
    @Override
    public void write(byte[] b) throws IOException {
        this.SEVENZ.write(b);
        this.position += b.length;
    }

    /**
     * @see java.io.OutputStream#write(byte[], int, int)
     * @see SevenZOutputFile#write(byte[], int, int)
     * 
     * @throws IOException
     */
    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        this.SEVENZ.write(b, off, len);
        this.position += len;
    }

    final private SevenZOutputFile SEVENZ;
    private int position;
}
