/*
 * Copyright 2016 Mark Fairchild
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.bsa;

import restringer.LittleEndianRAF;
import java.io.IOException;

/**
 * Describes a BSA header.
 *
 * @author Mark Fairchild
 * @version 2016/04/26
 */
final class BSAHeader {

    /**
     * Creates a new <code>BSAHeader</code> by reading from a
     * <code>LittleEndianDataInput</code>.
     *
     * @param input The file from which to readFully.
     * @throws IOException
     */
    public BSAHeader(LittleEndianRAF input) throws IOException {
        this.FILEID = new byte[4];
        input.readFully(this.FILEID);
        this.VERSION = input.readInt();
        this.FOLDER_OFFSET = input.readInt();
        this.ARCHIVE_FLAGS = input.readInt();
        this.FOLDER_COUNT = input.readInt();
        this.FILE_COUNT = input.readInt();
        this.TOTAL_FOLDERNAME_LENGTH = input.readInt();
        this.TOTAL_FILENAME_LENGTH = input.readInt();
        this.FILE_FLAGS = input.readInt();
    }

    /**
     * @return True if the include directory names bit is set, false otherwise.
     */
    public boolean isIncludeDirectoryName() {
        return (this.ARCHIVE_FLAGS & 0x1) != 0;
    }

    /**
     * @return True if the include filenames bit is set, false otherwise.
     */
    public boolean isIncludeFilenames() {
        return (this.ARCHIVE_FLAGS & 0x2) != 0;
    }
    /**
     * @return True if the compressed archive bit is set, false otherwise.
     */
    public boolean isCompressedArchive() {
        return (this.ARCHIVE_FLAGS & 0x4) != 0;
    }

    /**
     * @return True if the embed filename bit is set, false otherwise.
     */
    public boolean isEmbedFilename() {
        return (this.ARCHIVE_FLAGS & 0x100) != 0;
    }

    /**
     * @return True if the flag for miscellaneous (ie: scripts) files is set.
     */
    public boolean containsMiscellaneous() {
        return (this.FILE_FLAGS & 0x100) != 0;
    }

    final public byte[] FILEID;
    final public int VERSION;
    final public int FOLDER_OFFSET;
    final public int ARCHIVE_FLAGS;
    final public int FOLDER_COUNT;
    final public int FILE_COUNT;
    final public int TOTAL_FOLDERNAME_LENGTH;
    final public int TOTAL_FILENAME_LENGTH;
    final public int FILE_FLAGS;

}
