/*
 * Copyright 2016 Mark Fairchild
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.bsa;

import restringer.LittleEndianRAF;
import java.io.File;
import java.io.IOException;
import java.util.*;
import restringer.Game;
import restringer.esp.StringsFile;
import restringer.pex.PexFile;

/**
 * Handles the job of reading scripts out of BSA files.
 *
 * @author Mark Fairchild
 * @version 2016/07/07
 */
public class BSAParser {

    /**
     * Creates a new <code>BSAHeader</code> by reading from a
     * <code>LittleEndianDataInput</code>.
     *
     * @param name The name of the BSA.
     * @param input The file from which to readFully.
     * @throws IOException
     */
    public BSAParser(String name, LittleEndianRAF input) throws IOException {
        this.INPUT = Objects.requireNonNull(input);

        // Read the header.
        this.NAME = name;
        this.HEADER = new BSAHeader(input);

        this.FOLDERRECORDS = new ArrayList<>(this.HEADER.FOLDER_COUNT);
        this.FILERECORD_BLOCKS = new ArrayList<>(this.HEADER.FOLDER_COUNT);
        this.FILENAMES = new ArrayList<>(this.HEADER.FILE_COUNT);

        // Read folder records.
        for (int i = 0; i < this.HEADER.FOLDER_COUNT; i++) {
            FolderRecord folder = new FolderRecord(input, this.HEADER);
            this.FOLDERRECORDS.add(folder);
        }

        // Read FileRecordBlocks.
        for (FolderRecord folder : this.FOLDERRECORDS) {
            FileRecordBlock block = new FileRecordBlock(input, folder, this.HEADER);
            this.FILERECORD_BLOCKS.add(block);
        }

        // Read the filename table.
        if (this.HEADER.isIncludeFilenames()) {
            for (int i = 0; i < this.HEADER.FILE_COUNT; i++) {
                String fileName = input.readZString();
                this.FILENAMES.add(fileName);
            }

            // Map filerecords to filenames.
            Iterator<String> nameIterator = this.FILENAMES.iterator();
            this.FILERECORD_BLOCKS.forEach(block -> block.FILERECORDS.forEach(file -> file.name = nameIterator.next()));
        }
    }

    /**
     * Retrieves all of the scripts from the BSA file as <code>PexFile</code>
     * objects.
     *
     * @return A map of <code>PexFile</code> objects.
     * @throws IOException
     *
     */
    public Map<File, PexFile> getScripts() throws IOException {
        final Map<File, PexFile> SCRIPTS = new java.util.LinkedHashMap<>();

        // Find the scripts directory.
        Optional<FileRecordBlock> o = this.FILERECORD_BLOCKS
                .parallelStream()
                .filter(block -> block.NAME.equalsIgnoreCase("scripts"))
                .findFirst();

        if (!o.isPresent()) {
            return SCRIPTS;
        }
        final FileRecordBlock SCRIPTFOLDER = o.get();

        // Scan through the files, load them, and makes PexFile's.
        for (FileRecord SCRIPTFILE : SCRIPTFOLDER.FILERECORDS) {
            // Make sure it's a PEX file and not a PSC file.
            if (!restringer.Mod.PEX_REGEX.matcher(SCRIPTFILE.name).matches()) {
                continue;
            }

            final FileData FILEDATA = new FileData(this.INPUT, SCRIPTFILE, this.HEADER);
            PexFile pex = PexFile.readScript(FILEDATA.DATA);
            File pseudoFile = new File(pex.getFilename().toString());
            SCRIPTS.put(pseudoFile, pex);
        }
        return SCRIPTS;
    }

    /**
     * Retrieves a single script from the BSA file as <code>PexFile</code>
     * object.
     *
     * @param filename The filename of the script to retrieve.
     * @return A <code>PexFile</code> object.
     * @throws IOException
     *
     */
    public PexFile getScript(String filename) throws IOException {
        // Find the scripts directory.
        Optional<FileRecordBlock> o = this.FILERECORD_BLOCKS
                .parallelStream()
                .filter(block -> block.NAME.equalsIgnoreCase("scripts"))
                .findFirst();

        if (!o.isPresent()) {
            return null;
        }
        final FileRecordBlock SCRIPTFOLDER = o.get();

        // Scan through the files, load them, and makes PexFile's.
        for (FileRecord SCRIPTFILE : SCRIPTFOLDER.FILERECORDS) {
            // Make sure it's a PEX file and not a PSC file.
            if (!restringer.Mod.PEX_REGEX.matcher(SCRIPTFILE.name).matches()) {
                continue;
            } else if (SCRIPTFILE.name.equalsIgnoreCase(filename)) {
                continue;
            }

            final FileData FILEDATA = new FileData(this.INPUT, SCRIPTFILE, this.HEADER);
            PexFile pex = PexFile.readScript(FILEDATA.DATA);
            return pex;
        }
        
        return null;
    }

    /**
     * Retrieves all of the stringtables from the BSA file.
     *
     * @param language The language for which to retrieve stringtables.
     * @return A map of <code>PexFile</code> objects.
     * @throws IOException
     *
     */
    public List<StringsFile> getStrings(String language) throws IOException {
        final List<StringsFile> STRINGS = new java.util.LinkedList<>();

        // Find the scripts directory.
        Optional<FileRecordBlock> o = this.FILERECORD_BLOCKS
                .parallelStream()
                .filter(block -> block.NAME.equalsIgnoreCase("strings"))
                .findFirst();

        if (!o.isPresent()) {
            return STRINGS;
        }

        final FileRecordBlock STRINGSFOLDER = o.get();

        // Scan through the files, load them, and makes StringFile objects.
        for (FileRecord STRINGSFILE : STRINGSFOLDER.FILERECORDS) {
            final String FILENAME = STRINGSFILE.name.toLowerCase();

            // Make sure it's a PEX file and not a PSC file.
            if (!restringer.Mod.STR_REGEX.matcher(FILENAME).matches()) {
                continue;
            } else if (!FILENAME.contains(language)) {
                continue;
            }

            final FileData FILEDATA = new FileData(this.INPUT, STRINGSFILE, this.HEADER);

            StringsFile.Type type = StringsFile.Type.match(FILEDATA.NAME);
            StringsFile strings = new StringsFile(FILENAME, FILEDATA.getStream(), type);
            STRINGS.add(strings);
        }
        return STRINGS;
    }

    final String NAME;
    final BSAHeader HEADER;
    final List<FolderRecord> FOLDERRECORDS;
    final List<FileRecordBlock> FILERECORD_BLOCKS;
    final ArrayList<String> FILENAMES;
    final LittleEndianRAF INPUT;

}
