/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.IOException;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import java.util.List;
import restringer.IString;

/**
 * FieldVMAD represents all a VMAD field.
 *
 * @author Mark Fairchild
 * @version 2016/04/23
 */
public class FieldVMAD implements Field {

    /**
     * Creates a new FieldVMAD by reading it from a LittleEndianInput.
     *
     * @param recordCode The record code.
     * @param fieldCode The field code, which must be "VMAD".
     * @param input The LittleEndianInput to readFully.
     * @param big A flag indicating that this is a BIG field.
     * @param ctx
     * @throws IOException Exceptions are not handled at all.
     */
    public FieldVMAD(RecordCode recordCode, IString fieldCode, LittleEndianInput input, boolean big, ESPContext ctx) throws IOException {
        assert input.available() > 0;
        assert fieldCode.equals(IString.get("VMAD"));

        this.RECORDCODE = recordCode;
        this.CODE = fieldCode;
        this.VERSION = input.readShort();
        this.OBJFORMAT = input.readShort();
        this.SCRIPTS = new java.util.LinkedList<>();
        this.FRAGMENTS = new java.util.LinkedList<>();
        this.BIG = big;

        int scriptCount = input.readUnsignedShort();

        for (int i = 0; i < scriptCount; i++) {
            try {
                Script script = new Script(input, ctx);
                this.SCRIPTS.add(script);
            } catch (IOException ex) {
                throw new IOException(String.format("Problem reading VMAD section on script %d: %s", i, ex.getMessage()), ex);
            }
        }

        int i = 0;
        try {
            while (input.available() > 0) {
                switch (recordCode) {
                    case INFO:
                    case PACK:
                        this.FRAGMENTS.add(new FragmentInfoPack(input, ctx));
                        break;
                    case PERK:
                        this.FRAGMENTS.add(new FragmentPerk(input, ctx));
                        break;
                    case QUST:
                        this.FRAGMENTS.add(new FragmentQust(input, ctx));
                        break;
                    case SCEN:
                        this.FRAGMENTS.add(new FragmentScen(input, ctx));
                        break;
                    case TERM:
                        this.FRAGMENTS.add(new FragmentTerm(input, ctx));
                        break;
                    default:
                        throw new IOException("Unexpected fragment type: " + recordCode);
                }
                i++;
            }
        } catch (IOException ex) {
            throw new IOException("Error while reading fragment " + i + ".", ex);
        }
    }

    /**
     * @see Entry#write(transposer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        output.write(this.CODE.getBytes());

        if (this.BIG) {
            output.writeShort(0);
        } else {
            output.writeShort(this.calculateSize() - 6);
        }

        output.writeShort(this.VERSION);
        output.writeShort(this.OBJFORMAT);
        output.writeShort(this.SCRIPTS.size());

        for (Script script : this.SCRIPTS) {
            script.write(output);
        }

        for (FragmentBase fragment : this.FRAGMENTS) {
            fragment.write(output);
        }
    }

    /**
     * @return The calculated size of the field.
     * @see Entry#calculateSize()
     */
    @Override
    public int calculateSize() {
        int sum = 12;
        sum += this.SCRIPTS.stream().mapToInt(v -> v.calculateSize()).sum();
        sum += this.FRAGMENTS.stream().mapToInt(v -> v.calculateSize()).sum();
        return sum;
    }

    /**
     * Returns the field code.
     *
     * @return The field code.
     */
    @Override
    public IString getCode() {
        return this.CODE;
    }

    /**
     * Returns a String representation of the Field, which will just be the code
     * string.
     *
     * @return A string representation.
     *
     */
    @Override
    public String toString() {
        return this.getCode().toString();
    }

    final private RecordCode RECORDCODE;
    final private IString CODE;
    final private short VERSION;
    final private short OBJFORMAT;
    final private List<Script> SCRIPTS;
    final private List<FragmentBase> FRAGMENTS;
    final boolean BIG;

}
