/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.IOException;
import java.util.List;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.IString;

/**
 * Describes script fragments for PERK records.
 *
 * @author Mark Fairchild
 * @version 2016/04/23
 */
public class FragmentPerk extends FragmentBase {

    public FragmentPerk(LittleEndianInput input, ESPContext ctx) throws IOException {
        try {
            this.UNKNOWN = input.readByte();

            if (ctx.GAME.isFO4()) {
                ctx.pushContext("FragmentPerk");
                this.FILENAME = null;
                this.SCRIPT = new Script(input, ctx);
            } else {
                this.FILENAME = input.readUTF();
                this.SCRIPT = null;
                ctx.pushContext("FragmentPerk:" + this.FILENAME);
            }

            int fragmentCount = input.readUnsignedShort();
            this.FRAGMENTS = new java.util.ArrayList<>(fragmentCount);
            for (int i = 0; i < fragmentCount; i++) {
                Fragment fragment = new Fragment(input);
                this.FRAGMENTS.add(fragment);
            }
            
        } catch (IOException ex) {
            throw ex;
        } finally {
            ctx.popContext();
        }
    }

    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        output.writeByte(this.UNKNOWN);
        if (null != this.SCRIPT) {
            this.SCRIPT.write(output);
        }
        if (null != this.FILENAME) {
            output.writeUTF(this.FILENAME);
        }

        output.writeShort(this.FRAGMENTS.size());

        for (Fragment fragment : this.FRAGMENTS) {
            fragment.write(output);
        }
    }

    @Override
    public int calculateSize() {
        int sum = 3;
        sum += (null != this.SCRIPT ? this.SCRIPT.calculateSize() : 0);
        sum += (null != this.FILENAME ? 2 + this.FILENAME.length() : 0);
        sum += this.FRAGMENTS.stream().mapToInt(v -> v.calculateSize()).sum();
        return sum;
    }

    @Override
    public String toString() {
        if (null != this.SCRIPT) {
            return String.format("Perk: %s (%d, %d frags)", this.SCRIPT.NAME, this.UNKNOWN, this.FRAGMENTS.size());
        } else if (null != this.FILENAME) {
            return String.format("Perk: %s (%d, %d frags)", this.FILENAME, this.UNKNOWN, this.FRAGMENTS.size());
        } else {
            return String.format("Perk: (%d, %d frags)", this.UNKNOWN, this.FRAGMENTS.size());
        }
    }

    final private byte UNKNOWN;
    final private String FILENAME;
    final private Script SCRIPT;
    final private List<Fragment> FRAGMENTS;

    /**
     *
     */
    public class Fragment implements Entry {

        public Fragment(LittleEndianInput input) throws IOException {
            this.INDEX = input.readUnsignedShort();
            this.UNKNOWN1 = input.readShort();
            this.UNKNOWN2 = input.readByte();
            this.SCRIPTNAME = IString.get(input.readUTF());
            this.FRAGMENTNAME = IString.get(input.readUTF());
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeShort(this.INDEX);
            output.writeShort(this.UNKNOWN1);
            output.writeByte(this.UNKNOWN2);
            output.writeUTF(this.SCRIPTNAME.toString());
            output.writeUTF(this.FRAGMENTNAME.toString());
        }

        @Override
        public int calculateSize() {
            return 9 + this.SCRIPTNAME.length() + this.FRAGMENTNAME.length();
        }

        final private int INDEX;
        final private short UNKNOWN1;
        final private byte UNKNOWN2;
        final private IString SCRIPTNAME;
        final private IString FRAGMENTNAME;
    }
}
