/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.IOException;
import java.util.List;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.IString;

/**
 * Describes script fragments for QUST records.
 *
 * @author Mark Fairchild
 * @version 2016/04/23
 */
public class FragmentQust extends FragmentBase {

    public FragmentQust(LittleEndianInput input, ESPContext ctx) throws IOException {
        try {
            this.UNKNOWN = input.readByte();
            int fragmentCount = input.readUnsignedShort();

            if (ctx.GAME.isFO4()) {
                ctx.pushContext("FragmentQust");
                this.FILENAME = null;
                this.SCRIPT = new Script(input, ctx);
            } else {
                this.FILENAME = input.readUTF();
                this.SCRIPT = null;
                ctx.pushContext("FragmentQust:" + this.FILENAME);
            }

            this.FRAGMENTS = new java.util.LinkedList<>();
            this.ALIASES = new java.util.LinkedList<>();

            for (int i = 0; i < fragmentCount; i++) {
                Fragment fragment = new Fragment(input, ctx);
                this.FRAGMENTS.add(fragment);
            }

            int aliasCount = input.readUnsignedShort();
            for (int i = 0; i < aliasCount; i++) {
                Alias alias = new Alias(input, ctx);
                this.ALIASES.add(alias);
            }

        } catch (IOException ex) {
            throw ex;
        } finally {
            ctx.popContext();
        }
    }

    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        output.writeByte(this.UNKNOWN);
        output.writeShort(this.FRAGMENTS.size());
        if (null != this.FILENAME) {
            output.writeUTF(this.FILENAME);
        }
        if (null != this.SCRIPT) {
            this.SCRIPT.write(output);
        }

        for (Fragment fragment : this.FRAGMENTS) {
            fragment.write(output);
        }

        output.writeShort(this.ALIASES.size());
        for (Alias alias : this.ALIASES) {
            alias.write(output);
        }
    }

    @Override
    public int calculateSize() {
        int sum = 5;
        sum += (null != this.FILENAME ? 2 + this.FILENAME.length() : 0);
        sum += (null != this.SCRIPT ? this.SCRIPT.calculateSize() : 0);
        sum += this.FRAGMENTS.stream().mapToInt(v -> v.calculateSize()).sum();
        sum += this.ALIASES.stream().mapToInt(v -> v.calculateSize()).sum();
        return sum;
    }

    @Override
    public String toString() {
        if (null != this.SCRIPT) {
            return String.format("Quest: %s (%d, %d frags, %d aliases)", this.SCRIPT.NAME, this.UNKNOWN, this.FRAGMENTS.size(), this.ALIASES.size());
        } else if (null != this.FILENAME) {
            return String.format("Quest: %s (%d, %d frags, %d aliases)", this.FILENAME, this.UNKNOWN, this.FRAGMENTS.size(), this.ALIASES.size());
        } else {
            return String.format("Quest: (%d, %d frags, %d aliases)", this.UNKNOWN, this.FRAGMENTS.size(), this.ALIASES.size());

        }
    }

    final private byte UNKNOWN;
    final private String FILENAME;
    final private Script SCRIPT;
    final private List<Fragment> FRAGMENTS;
    final private List<Alias> ALIASES;

    /**
     *
     */
    public class Fragment implements Entry {

        public Fragment(LittleEndianInput input, ESPContext ctx) throws IOException {
            this.STAGE = input.readUnsignedShort();
            this.UNKNOWN1 = input.readShort();
            this.LOGENTRY = input.readInt();
            this.UNKNOWN2 = input.readByte();
            this.SCRIPTNAME = IString.get(input.readUTF());
            this.FRAGMENTNAME = IString.get(input.readUTF());
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeShort(this.STAGE);
            output.writeShort(this.UNKNOWN1);
            output.writeInt(this.LOGENTRY);
            output.writeByte(this.UNKNOWN2);
            output.writeUTF(this.SCRIPTNAME.toString());
            output.writeUTF(this.FRAGMENTNAME.toString());
        }

        @Override
        public int calculateSize() {
            int sum = 13;
            sum += this.SCRIPTNAME.length();
            sum += this.FRAGMENTNAME.length();
            return sum;
        }

        final private int STAGE;
        final private short UNKNOWN1;
        final private int LOGENTRY;
        final private byte UNKNOWN2;
        final private IString SCRIPTNAME;
        final private IString FRAGMENTNAME;
    }

    /**
     *
     */
    public class Alias implements Entry {

        public Alias(LittleEndianInput input, ESPContext ctx) throws IOException {
            this.OBJECT = input.readLong();
            this.VERSION = input.readShort();
            this.OBJFORMAT = input.readShort();
            this.SCRIPTS = new java.util.LinkedList<>();

            int scriptCount = input.readUnsignedShort();
            for (int i = 0; i < scriptCount; i++) {
                Script script = new Script(input, ctx);
                this.SCRIPTS.add(script);
            }
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeLong(this.OBJECT);
            output.writeShort(this.VERSION);
            output.writeShort(this.OBJFORMAT);
            output.writeShort(this.SCRIPTS.size());

            for (Script script : this.SCRIPTS) {
                script.write(output);
            }
        }

        @Override
        public int calculateSize() {
            int sum = 14;
            sum += this.SCRIPTS.stream().mapToInt(v -> v.calculateSize()).sum();
            return sum;
        }

        final private long OBJECT;
        final private short VERSION;
        final private short OBJFORMAT;
        final private List<Script> SCRIPTS;
    }
}
