/*
 * Copyright 2017 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.IOException;
import java.util.List;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.IString;

/**
 * Describes script fragments for QUST records.
 *
 * @author Mark Fairchild
 */
public class FragmentTerm extends FragmentBase {

    public FragmentTerm(LittleEndianInput input, ESPContext ctx) throws IOException {
        //input = LittleEndianInputStream.debug(input);
        this.UNKNOWN = input.readByte();
        this.SCRIPT = new Script(input, ctx);

        int fragCount = input.readShort();
        this.FRAGMENTS = new java.util.ArrayList<>(fragCount);

        for (int i = 0; i < fragCount; i++) {
            Fragment fragment = new Fragment(input);
            this.FRAGMENTS.add(fragment);
        }
    }

    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        output.writeByte(this.UNKNOWN);
        this.SCRIPT.write(output);
        output.writeShort(this.FRAGMENTS.size());

        for (Fragment fragment : this.FRAGMENTS) {
            fragment.write(output);
        }
    }

    @Override
    public int calculateSize() {
        int sum = 3;
        sum += this.SCRIPT.calculateSize();
        sum += this.FRAGMENTS.stream().mapToInt(v -> v.calculateSize()).sum();
        return sum;
    }

    @Override
    public String toString() {
        return String.format("Term: %s (%d, %d fragments)", this.SCRIPT.NAME, this.UNKNOWN, this.FRAGMENTS.size());
    }

    final private byte UNKNOWN;
    final private Script SCRIPT;
    final private List<Fragment> FRAGMENTS;

    /**
     *
     */
    public class Fragment implements Entry {

        public Fragment(LittleEndianInput input) throws IOException {
            this.INDEX = input.readInt();
            this.UNKNOWN = input.readByte();
            this.SCRIPTNAME = IString.get(input.readUTF());
            this.FRAGMENTNAME = IString.get(input.readUTF());
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeByte(this.INDEX);
            output.writeByte(this.UNKNOWN);
            output.writeUTF(this.SCRIPTNAME.toString());
            output.writeUTF(this.FRAGMENTNAME.toString());
        }

        @Override
        public int calculateSize() {
            return 9 + this.SCRIPTNAME.length() + this.FRAGMENTNAME.length();
        }

        @Override
        public String toString() {
            return String.format("%d: %s [%s] (%d)", this.INDEX, this.SCRIPTNAME, this.FRAGMENTNAME, this.UNKNOWN);
        }

        final private int INDEX;
        final private byte UNKNOWN;
        final private IString SCRIPTNAME;
        final private IString FRAGMENTNAME;
    }

}
