/*
 * Copyright 2017 Mark.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.IOException;
import restringer.IString;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInput;

/**
 * Describes a property entry in a VMAD's scripts.
 *
 * @author Mark Fairchild
 * @version 2016/04/23
 */
final public class Property implements Entry {

    /**
     * Creates a new Property by reading it from a LittleEndianInput.
     *
     * @param input The LittleEndianInput to readFully.
     * @param ctx
     * @throws IOException Exceptions are not handled at all.
     */
    public Property(LittleEndianInput input, ESPContext ctx) throws IOException {
        this.NAME = IString.get(input.readUTF());
        ctx.pushContext("prop:" + this.NAME);
        
        this.TYPE = input.readByte();
        this.STATUS = input.readByte();
        try {
            this.DATA = PropertyData.readPropertyData(this.TYPE, input, ctx);
        } catch (IOException ex) {
            throw new IOException("Error data for reading property " + this.NAME, ex);
        } finally {
            ctx.popContext();
        }
    }

    /**
     * @see Entry#write(transposer.LittleEndianDataOutput)
     * @param output The LittleEndianDataOutput.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        output.writeUTF(this.NAME.toString());
        output.writeByte(this.TYPE);
        output.writeByte(this.STATUS);
        this.DATA.write(output);
    }

    /**
     * @return The calculated size of the Script.
     */
    @Override
    public int calculateSize() {
        int sum = 4 + this.NAME.length() + this.DATA.calculateSize();
        return sum;
    }

    @Override
    public String toString() {
        return String.format("%s: %d (%02x): %s", this.NAME, this.TYPE, this.STATUS, this.DATA);
    }
    
    private final IString NAME;
    private final byte TYPE;
    private final byte STATUS;
    private final PropertyData DATA;

}
