/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.IOException;
import java.util.List;
import restringer.BoundedLittleEndianInput;
import restringer.IString;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInputStream;
import restringer.ess.papyrus.EID;

/**
 * Base class for the records of an ESP file.
 *
 * @author Mark Fairchild
 * @version 2016/04/23
 */
abstract public class Record implements Entry {

    /**
     * Reads a field from an ESP file input and returns it. Usually only one
     * field is readFully, but if the field is an XXXX type, the next field will
     * be returned as well.
     *
     * @param parentCode The recordcode of the containing record.
     * @param input The LittleEndianInput to readFully.
     * @param ctx The mod descriptor.
     * @return A list of fields that were readFully.
     * @throws IOException Exceptions aren't handled at all.
     *
     */
    static final public List<Field> readFields(RecordCode parentCode, LittleEndianInput input, ESPContext ctx) throws IOException {
        final List<Field> FIELDS = new java.util.LinkedList<>();

        while (input.available() > 0) {
            List<Field> fieldsRead = Record.readField(parentCode, input, ctx);
            FIELDS.addAll(fieldsRead);
        }

        return FIELDS;
    }

    /**
     * Reads a field from an ESP file input and returns it. Usually only one
     * field is readFully, but if the field is an XXXX type, the next field will
     * be returned as well.
     *
     * @param parentCode The recordcode of the containing record.
     * @param input The LittleEndianInput to readFully.
     * @param ctx The mod descriptor.
     * @return A list of fields that were readFully.
     * @throws IOException Exceptions aren't handled at all.
     *
     */
    static final public List<Field> readField(RecordCode parentCode, LittleEndianInput input, ESPContext ctx) throws IOException {
        return readFieldAux(parentCode, input, 0, ctx);
    }

    /**
     * Reads a field from an ESP file input and returns it. Usually only one
     * field is readFully, but if the field is an XXXX type, the next field will
     * be returned as well.
     *
     * @param parentCode The recordcode of the containing record.
     * @param input The LittleEndianInput to readFully.
     * @param bigSize The size of the field, if it was specified externally (a
     * "XXXX" record).
     * @param ctx The mod descriptor.
     * @return A list of fields that were readFully.
     * @throws IOException Exceptions aren't handled at all.
     *
     */
    static private List<Field> readFieldAux(RecordCode parentCode, LittleEndianInput input, int bigSize, ESPContext ctx) throws IOException {
        assert input.available() > 0;

        // Read the record identification code.
        final byte[] CODEBYTES = new byte[4];
        input.readFully(CODEBYTES);
        final IString CODE = IString.get(new String(CODEBYTES));
        ctx.pushContext(CODE);

        // Read the record size.
        final boolean BIG = bigSize > 0;
        final int DATASIZE = input.readUnsignedShort();
        final int ACTUALSIZE = (BIG ? bigSize : DATASIZE);

        // 
        // Read the record data.
        final LittleEndianInput FIELDINPUT = new BoundedLittleEndianInput(input, ACTUALSIZE);

        // This list will hold either one or two fields that are readFully.
        List<Field> fields = new java.util.ArrayList<>();

        // Depending on what code we found, pick a subclass to readFully in the
        // rest of the data.
        if (CODE.equals(IString.get("XXXX"))) {
            FieldXXXX xxxx = new FieldXXXX(CODE, FIELDINPUT);
            List<Field> fieldsRead = readFieldAux(parentCode, input, xxxx.getData(), ctx);
            fields.add(xxxx);
            fields.addAll(fieldsRead);

        } else if (CODE.equals(IString.get("VMAD"))) {
            FieldVMAD field = new FieldVMAD(parentCode, CODE, FIELDINPUT, BIG, ctx);
            fields.add(field);

        } else if (CODE.equals(IString.get("EDID"))) {
            FieldEDID field = new FieldEDID(CODE, FIELDINPUT, ACTUALSIZE, BIG, ctx);
            fields.add(field);

        } else if (CODE.equals(IString.get("FULL"))) {
            FieldFull field = new FieldFull(CODE, FIELDINPUT, ACTUALSIZE, BIG, ctx);
            fields.add(field);

        } else if (CODE.equals(IString.get("NAME")) && (parentCode == RecordCode.ACHR
                || parentCode == RecordCode.REFR)) {
            FieldName field = new FieldName(CODE, FIELDINPUT, ACTUALSIZE, BIG, ctx);
            fields.add(field);

        } else {
            Field field = new FieldBasic(CODE, FIELDINPUT, ACTUALSIZE, BIG, ctx);
            fields.add(field);
        }

        ctx.popContext();
        return fields;
    }

    /**
     * Returns the record code.
     *
     * @return The record code.
     */
    abstract public RecordCode getCode();

    /**
     * Reads a record from an ESP file input and returns it.
     *
     * @param input The LittleEndianInput to readFully.
     * @param ctx The mod descriptor.
     * @return The next Record from input.
     * @throws IOException Exceptions aren't handled at all.
     *
     */
    static public Record readRecord(LittleEndianInput input, ESPContext ctx) throws IOException {
        if (input.available() < 24) {
            throw new IOException();
        }

        // Read the record identification code.
        final byte[] CODEBYTES = new byte[4];
        input.readFully(CODEBYTES);
        final String CODESTRING = new String(CODEBYTES);
        final RecordCode CODE = RecordCode.valueOf(CODESTRING);

        // Read the record size.
        final int DATASIZE = input.readInt();

        // GRUPs get handled differently than other records.
        if (CODE == RecordCode.GRUP) {
            // Read the header.
            final byte[] HEADER = new byte[16];
            input.readFully(HEADER);

            // Read the record data.
            final LittleEndianInput RECORDINPUT = new BoundedLittleEndianInput(input, DATASIZE - 24);

            // Read the rest of the record.
            return new RecordGrup(CODE, HEADER, RECORDINPUT, ctx);

        } else {
            // Read the header.
            final Header HEADER = new Header(input, ctx);

            // Read the record data.
            final LittleEndianInput RECORDINPUT = new BoundedLittleEndianInput(input, DATASIZE);

            // Read the rest of the record. Handle compressed records separately.
            if (HEADER.isCompressed()) {
                return new RecordCompressed(CODE, HEADER, RECORDINPUT, ctx);

            } else {
                return new RecordBasic(CODE, HEADER, RECORDINPUT, ctx);
            }
        }
    }

    /**
     * Reads a record from an ESP file input and returns it.
     *
     * @param input The LittleEndianInput to readFully.
     * @param ctx The mod descriptor.
     * @throws IOException Exceptions aren't handled at all.
     *
     */
    static public void skimRecord(LittleEndianInput input, ESPContext ctx) throws IOException {
        if (input.available() < 24) {
            throw new IOException();
        }

        // Read the record identification code.
        final byte[] CODEBYTES = new byte[4];
        input.readFully(CODEBYTES);
        final String CODESTRING = new String(CODEBYTES);
        final RecordCode CODE;

        try {
            CODE = RecordCode.valueOf(CODESTRING);
        } catch (Exception ex) {
            throw ex;
        }

        // Read the record size.
        final int DATASIZE = input.readInt();

        // GRUPs get handled differently than other records.
        if (CODE == RecordCode.GRUP) {
            // Read the header.
            final byte[] HEADER = new byte[16];
            input.readFully(HEADER);
            
            LittleEndianInput headerStream = LittleEndianInputStream.wrap(HEADER);            
            final int PREFIX = headerStream.readInt();
            final int TYPE = headerStream.readInt();
            switch (TYPE) {
                case 0:
                    ctx.pushContext(new String(HEADER,0,4));
                    break;
                case 4:
                case 5:
                    final int X = (TYPE == 4 || TYPE == 5 ? PREFIX&0xFFFF : -1);
                    final int Y = (TYPE == 4 || TYPE == 5 ? PREFIX>>>4 : -1);
                    ctx.pushContext(X + ", " + Y);
                    break;
                case 2:
                    ctx.pushContext("Block " + PREFIX);
                    break;
                case 3:
                    ctx.pushContext("SubBlock " + PREFIX);
                    break;
                default:
                    ctx.pushContext(EID.pad8(PREFIX));
                    break;
            }
            
            // Read the record data.
            final LittleEndianInput RECORDINPUT = new BoundedLittleEndianInput(input, DATASIZE - 24);

            // Read the rest of the record.
            RecordGrup.skimGRUP(CODE, HEADER, RECORDINPUT, ctx);
            ctx.popContext();

        } else {
            // Read the header.
            final Header HEADER = new Header(input, ctx);
            ctx.pushContext(EID.pad8(HEADER.ID));
            
            // Read the record data.
            final LittleEndianInput RECORDINPUT = new BoundedLittleEndianInput(input, DATASIZE);

            // Read the rest of the record. Handle compressed records separately.
            if (HEADER.isCompressed()) {
                RecordCompressed.skimRecord(CODE, HEADER, RECORDINPUT, ctx);

            } else {
                RecordBasic.skimRecord(CODE, HEADER, RECORDINPUT, ctx);
            }
            
            ctx.popContext();
        }
    }

    /**
     * Header represents the standard header for all records except GRUP.
     *
     * @author Mark Fairchild
     * @version 2016/04/23
     */
    static public class Header implements Entry {

        /**
         * Creates a new Header by reading it from a LittleEndianInput.
         *
         * @param input The LittleEndianInput to readFully.
         * @param ctx The mod descriptor.
         * @throws IOException Exceptions are not handled at all.
         */
        public Header(LittleEndianInput input, ESPContext ctx) throws IOException {
            this.FLAGS = input.readInt();

            int id = input.readInt();
            int newID = (null == ctx ? id : ctx.TES4.remapFormID(id));
            this.ID = newID;

            this.REVISION = input.readInt();
            this.VERSION = input.readShort();
            this.UNKNOWN = input.readShort();
        }

        /**
         * @see Entry#write(transposer.LittleEndianDataOutput)
         * @throws IOException
         */
        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeInt(this.FLAGS);
            output.writeInt(this.ID);
            output.writeInt(this.REVISION);
            output.writeShort(this.VERSION);
            output.writeShort(this.UNKNOWN);
        }

        /**
         * @return The calculated size of the field.
         * @see Entry#calculateSize()
         */
        @Override
        public int calculateSize() {
            return 16;
        }

        /**
         * Checks if the header indicates a compressed record.
         *
         * @return True if the field data is compressed, false otherwise.
         */
        public boolean isCompressed() {
            return (this.FLAGS & 0x00040000) != 0;
        }

        /**
         * Checks if the header indicates localization (TES4 record only).
         *
         * @return True if the record is a TES4 and localization is enabled.
         */
        public boolean isLocalized() {
            return (this.FLAGS & 0x00000080) != 0;
        }

        final public int FLAGS;
        final public int ID;
        final public int REVISION;
        final public short VERSION;
        final public short UNKNOWN;

    }
}
