/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.IOException;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import java.util.List;

/**
 * RecordBasic represents all records that are not a GRUP and do not contain
 * compressed fields.
 *
 * @author Mark Fairchild
 * @version 2016/09/08
 */
public class RecordBasic extends Record {

    /**
     * Skims a RecordBasic by reading it from a LittleEndianInput.
     *
     * @param recordCode The record code.
     * @param header The header.
     * @param input The LittleEndianInput to read.
     * @param ctx The mod descriptor.
     * @throws IOException Exceptions are not handled at all.
     */
    static public void skimRecord(RecordCode recordCode, Record.Header header, LittleEndianInput input, ESPContext ctx) throws IOException {
        List<Field> fieldsRead = Record.readFields(recordCode, input, ctx);
        ctx.IDMAP.addRecord(header.ID, fieldsRead);
    }

    /**
     * Creates a new RecordBasic by reading it from a LittleEndianInput.
     *
     * @param recordCode The record code.
     * @param header The header.
     * @param input The LittleEndianInput to read.
     * @param ctx The mod descriptor.
     * @throws IOException Exceptions are not handled at all.
     */
    public RecordBasic(RecordCode recordCode, Record.Header header, LittleEndianInput input, ESPContext ctx) throws IOException {
        this.CODE = recordCode;
        this.HEADER = header;

        List<Field> fieldsRead = Record.readFields(recordCode, input, ctx);
        this.FIELDS = new java.util.ArrayList<>(fieldsRead);
    }

    /**
     * @see Entry#write(transposer.LittleEndianDataOutput)
     * @param output The LittleEndianDataOutput.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        output.write(this.CODE.toString().getBytes());
        output.writeInt(this.calculateSize() - 24);
        this.HEADER.write(output);

        for (Field field : this.FIELDS) {
            field.write(output);
        }
    }

    /**
     * @return The calculated size of the field.
     * @see Entry#calculateSize()
     */
    @Override
    public int calculateSize() {
        int sum = 24;
        sum += this.FIELDS.stream().mapToInt(v -> v.calculateSize()).sum();
        return sum;
    }

    /**
     * Returns the record code.
     *
     * @return The record code.
     */
    @Override
    public RecordCode getCode() {
        return this.CODE;
    }

    /**
     * @return The record header.
     */
    public Record.Header getHeader() {
        return this.HEADER;
    }

    /**
     * Returns a String representation of the Record, which will just be the
     * code string.
     *
     * @return A string representation.
     *
     */
    @Override
    public String toString() {
        return this.getCode().toString();
    }

    final private RecordCode CODE;
    final private Record.Header HEADER;
    final private List<Field> FIELDS;

}
