/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.zip.*;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInputStream;

/**
 * RecordCompressed represents all records that are compressed.
 *
 * @author Mark Fairchild
 * @version 2016/04/23
 */
public class RecordCompressed extends Record {

    /**
     * Skims a RecordCompressed by reading it from a LittleEndianInput.
     *
     * @param recordCode The record code.
     * @param header The header.
     * @param input The LittleEndianInput to readFully.
     * @param ctx The mod descriptor.
     * @throws IOException Exceptions are not handled at all.
     */
    static public void skimRecord(RecordCode recordCode, Record.Header header, LittleEndianInput input, ESPContext ctx) throws IOException {
        assert input.available() > 0;
        List<Field> fieldsRead = decompressFields(recordCode, header, input, ctx);
        ctx.IDMAP.addRecord(header.ID, fieldsRead);
    }

    /**
     * Creates a new RecordCompressed by reading it from a LittleEndianInput.
     *
     * @param recordCode The record code.
     * @param header The header.
     * @param input The LittleEndianInput to readFully.
     * @param ctx The mod descriptor.
     * @throws IOException Exceptions are not handled at all.
     */
    public RecordCompressed(RecordCode recordCode, Record.Header header, LittleEndianInput input, ESPContext ctx) throws IOException {
        assert input.available() > 0;

        this.CODE = recordCode;
        this.HEADER = header;

        List<Field> fieldsRead = decompressFields(recordCode, header, input, ctx);
        this.FIELDS = new java.util.ArrayList<>(fieldsRead);
    }

    /**
     *
     * @return @throws IOException
     */
    private byte[] getUncompressedData() throws IOException {
        final ByteArrayOutputStream BAOS = new ByteArrayOutputStream();
        final LittleEndianDataOutput OUTPUT = new LittleEndianDataOutput(BAOS);
        for (Field field : this.FIELDS) {
            field.write(OUTPUT);
        }

        return BAOS.toByteArray();
    }

    /**
     *
     * @param rawData
     * @return
     * @throws IOException
     */
    private byte[] getCompressedData(byte[] rawData) throws IOException {
        DEFLATER.reset();
        DEFLATER.setInput(rawData);
        DEFLATER.finish();

        final byte[] BUFFER = new byte[1024];
        final ByteArrayOutputStream BAOS = new ByteArrayOutputStream();

        while (!DEFLATER.finished()) {
            int count = DEFLATER.deflate(BUFFER);
            BAOS.write(BUFFER, 0, count);
        }

        return BAOS.toByteArray();
    }

    /**
     * @see Entry#write(transposer.LittleEndianDataOutput)
     * @param output The LittleEndianDataOutput.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        output.write(this.CODE.toString().getBytes());

        byte[] UNCOMPRESSED_DATA = this.getUncompressedData();
        byte[] COMPRESSED_DATA = this.getCompressedData(UNCOMPRESSED_DATA);

        final int UNCOMPRESSED_SIZE = UNCOMPRESSED_DATA.length;
        final int COMPRESSED_SIZE = COMPRESSED_DATA.length;

        output.writeInt(4 + COMPRESSED_SIZE);
        this.HEADER.write(output);
        output.writeInt(UNCOMPRESSED_SIZE);
        output.write(COMPRESSED_DATA);
    }

    /**
     * @return The calculated size of the field.
     * @see Entry#calculateSize()
     */
    @Override
    public int calculateSize() {
        int sum = 28;

        try {
            byte[] UNCOMPRESSED_DATA = this.getUncompressedData();
            byte[] COMPRESSED_DATA = this.getCompressedData(UNCOMPRESSED_DATA);
            sum += COMPRESSED_DATA.length;
        } catch (IOException ex) {
            throw new IllegalStateException(ex);
        }

        return sum;
    }

    /**
     * Returns the record code.
     *
     * @return The record code.
     */
    @Override
    public RecordCode getCode() {
        return this.CODE;
    }

    /**
     * Returns a String representation of the Record, which will just be the
     * code string.
     *
     * @return A string representation.
     *
     */
    @Override
    public String toString() {
        return this.getCode().toString();
    }

    /**
     * Manages the actual decompression.
     *
     * @param code
     * @param header The header.
     * @param input
     * @param ctx The mod descriptor.
     * @return
     * @throws IOException
     */
    static private List<Field> decompressFields(RecordCode code, Record.Header header, LittleEndianInput input, ESPContext ctx) throws IOException {
        final int DECOMPRESSED_SIZE = input.readInt();
        final int COMPRESSED_SIZE = input.available();

        final byte[] COMPRESSED = new byte[COMPRESSED_SIZE];
        int bytesRead = input.read(COMPRESSED);
        assert bytesRead == COMPRESSED_SIZE : "Read " + COMPRESSED.length + " bytes, expected " + COMPRESSED_SIZE;

        final Inflater INFLATER = new Inflater();
        INFLATER.setInput(COMPRESSED);

        int bytesInflated;

        try {
            final ByteArrayOutputStream INFLATION_BUFFER = new ByteArrayOutputStream();
            final byte[] BUF = new byte[1024];

            do {
                bytesInflated = INFLATER.inflate(BUF);
                INFLATION_BUFFER.write(BUF, 0, bytesInflated);
            } while (bytesInflated > 0);

            final byte[] DECOMPRESSED = INFLATION_BUFFER.toByteArray();
            assert DECOMPRESSED.length == DECOMPRESSED_SIZE : "Decompressed " + DECOMPRESSED.length + " bytes, expected " + DECOMPRESSED_SIZE;

            final LittleEndianInput INPUT2 = LittleEndianInputStream.wrap(DECOMPRESSED);
            return Record.readFields(code, INPUT2, ctx);

        } catch (DataFormatException ex) {
            throw new IOException(ex);

        } finally {
            INFLATER.end();
        }
    }

    final private RecordCode CODE;
    final private Record.Header HEADER;
    final private List<Field> FIELDS;
    static final private Deflater DEFLATER = new Deflater(9);
}
