/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.IOException;
import java.util.List;
import java.util.LinkedList;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;

/**
 * Describes GRUP records.
 *
 * @author Mark Fairchild
 * @version 2016/09/08
 */
public class RecordGrup extends Record {

    /**
     * Skims an ESPGRUPRecord by reading it from a LittleEndianInput.
     *
     * @param code The record code, which must be RecordCode.GRUP.
     * @param headerData The header data (unused).
     * @param input The LittleEndianInput to read.
     * @param ctx The mod descriptor.
     * @return The ESPIDMap.
     * @throws IOException Exceptions are not handled at all.
     */
    static void skimGRUP(RecordCode code, byte[] headerData, LittleEndianInput input, ESPContext ctx) throws IOException {
        //ESPIDMap map = new ESPIDMap();

        while (input.available() > 0) {
            Record.skimRecord(input, ctx);
            //ESPIDMap recordMap = Record.skimRecord(input, tes4);
            //map.putAll(recordMap);
        }

        //return map;
    }

    /**
     * Creates a new RecordGRUP by reading it from a LittleEndianInput.
     *
     * @param code The record code, which must be RecordCode.GRUP.
     * @param headerData The header data (unused).
     * @param input The LittleEndianInput to read.
     * @param ctx The mod descriptor.
     * @throws IOException Exceptions are not handled at all.
     */
    public RecordGrup(RecordCode code, byte[] headerData, LittleEndianInput input, ESPContext ctx) throws IOException {
        this.CODE = code;
        this.HEADER = headerData.clone();
        this.RECORDS = new LinkedList<>();

        while (input.available() > 0) {
            Record record = Record.readRecord(input, ctx);
            this.RECORDS.add(record);
        }
    }

    /**
     * @see Entry#write(transposer.LittleEndianDataOutput)
     * @param output The LittleEndianDataOutput.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        output.write(this.CODE.toString().getBytes());
        output.writeInt(this.calculateSize());
        output.write(this.HEADER);

        for (Record record : this.RECORDS) {
            record.write(output);
        }
    }

    /**
     * @return The calculated size of the field.
     * @see Entry#calculateSize()
     */
    @Override
    public int calculateSize() {
        int sum = 24;
        sum += this.RECORDS.stream().mapToInt(v -> v.calculateSize()).sum();
        return sum;
    }

    /**
     * Returns the record code.
     *
     * @return The record code.
     */
    @Override
    public RecordCode getCode() {
        return this.CODE;
    }

    /**
     * Returns a String representation of the Record, which will just be the
     * code string.
     *
     * @return A string representation.
     *
     */
    @Override
    public String toString() {
        return this.getCode().toString();
    }

    final private RecordCode CODE;
    final private byte[] HEADER;
    final private List<Record> RECORDS;

}
