/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedList;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import java.util.List;
import java.util.Map;
import restringer.BoundedLittleEndianInput;
import restringer.IString;

/**
 * RecordTes4 is the first record. It handles its own data and is not read using
 * Record.readRecord().
 *
 * @author Mark Fairchild
 * @version 2016/09/08
 */
public class RecordTes4 extends Record {

    /**
     * Creates a new RecordTes4 by reading it from a LittleEndianInput.
     *
     * @param input The LittleEndianInput to read.
     * @param index The plugin index, for correcting FormIDs.
     * @param name The name of the plugin.
     * @param espList The list of ESPs, for correcting FormIDs.
     * @param ctx The mod descriptor.
     * @throws IOException Exceptions are not handled at all.
     */
    public RecordTes4(LittleEndianInput input, int index, String name, List<String> espList, ESPContext ctx) throws IOException {        
        this.CODE = RecordCode.TES4;
        this.INDEX = index;
        this.NAME = name;

        final byte[] CODEBYTES = new byte[4];
        input.readFully(CODEBYTES);
        final String CODESTRING = new String(CODEBYTES);
        assert CODESTRING.equals("TES4");

        final int DATASIZE = input.readInt();
        this.HEADER = new Record.Header(input, null);

        // Read the record data.
        final LittleEndianInput FIELDINPUT = new BoundedLittleEndianInput(input, DATASIZE);

        List<String> masters = new LinkedList<>();
        this.FIELDS = new java.util.LinkedList<>();

        while (FIELDINPUT.available() > 0) {
            List<Field> fields = Record.readField(RecordCode.TES4, FIELDINPUT, ctx);

            if (fields.get(0).getCode().equals(IString.get("MAST"))) {
                FieldBasic mast = (FieldBasic) fields.get(0);
                String master = mast.getDecoder().readZString();
                masters.add(master);
            }

            this.FIELDS.addAll(fields);
        }

        this.MASTERS = java.util.Collections.unmodifiableList(new ArrayList<>(masters));

        Map<String, Integer> esps = new java.util.HashMap<>(espList.size());
        for (int i = 0; i < espList.size(); i++) {
            esps.put(espList.get(i), i);
        }

        this.ESPs = java.util.Collections.unmodifiableMap(esps);
    }

    /**
     * @see Entry#write(transposer.LittleEndianDataOutput)
     * @param output The LittleEndianDataOutput.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        output.write(this.CODE.toString().getBytes());
        output.writeInt(this.calculateSize() - 24);
        this.HEADER.write(output);

        for (Field field : this.FIELDS) {
            field.write(output);
        }
    }

    /**
     * @return The calculated size of the field.
     * @see Entry#calculateSize()
     */
    @Override
    public int calculateSize() {
        int sum = 24;
        sum += this.FIELDS.stream().mapToInt(v -> v.calculateSize()).sum();
        return sum;
    }

    /**
     * Returns the record code.
     *
     * @return The record code.
     */
    @Override
    public RecordCode getCode() {
        return this.CODE;
    }

    /**
     * @return The record header.
     */
    public Record.Header getHeader() {
        return this.HEADER;
    }

    /**
     * Returns a String representation of the Record, which will just be the
     * code string.
     *
     * @return A string representation.
     *
     */
    @Override
    public String toString() {
        return this.getCode().toString();
    }

    /**
     * Remaps formIDs.
     * If the formID's master is not available, the plugin field of the formid
     * will be set to 255.
     * 
     * @param id The ID to remap.
     * @return 
     */
    public int remapFormID(int id) {
        int originalIndex = id >>> 24;
        assert 0 <= originalIndex && originalIndex < 256;
        
        String originalMod;

        if (originalIndex < this.MASTERS.size()) {
             originalMod = this.MASTERS.get(originalIndex);
             
        } else {
            originalMod = this.NAME;
        }

        if (!this.ESPs.containsKey(originalMod)) {
            return (0xFF<<24) | id;
        }
        
        int newIndex = this.ESPs.getOrDefault(originalMod, -1);        
        int newID = (id & 0xFFFFFF) | (newIndex<<24);
        
        // SHOULDN'T happen, but it does.... even once in Skyrim.esm!!!
        /*if (originalIndex > this.MASTERS.size()) {
            final String MSG = "%s contains an invalid formID; %08x was corrected to be %08x. If you think this is incorrect, please contact the developer of ReSaver: markdfairchild@gmail.com.\n";
            System.out.printf(MSG, this.NAME, id, newID);            
        }*/
        
        return newID;
    }

    final public int INDEX;
    final public String NAME;
    final public List<String> MASTERS;
    final public Map<String, Integer> ESPs;
    
    final private RecordCode CODE;
    final private Record.Header HEADER;
    final private List<Field> FIELDS;
}
