/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.IOException;
import java.util.List;
import restringer.IString;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;

/**
 * Describes a script entry in a VMAD field.
 *
 * @author Mark Fairchild
 * @version 2016/04/23
 */
public class Script implements Entry {

    /**
     * Creates a new Script by reading it from a LittleEndianInput.
     *
     * @param input The LittleEndianInput to readFully.
     * @param ctx
     * @throws IOException Exceptions are not handled at all.
     */
    public Script(LittleEndianInput input, ESPContext ctx) throws IOException {
        this.NAME = IString.get(input.readUTF());
        if (this.NAME.isEmpty()) {
            this.PROPERTIES = null;
            this.STATUS = 0;
            return;
        }

        ctx.pushContext("script:" + this.NAME);

        this.STATUS = input.readByte();

        try {
            int propertyCount = input.readUnsignedShort();
            this.PROPERTIES = new java.util.ArrayList<>(propertyCount);

            for (int i = 0; i < propertyCount; i++) {
                Property prop = new Property(input, ctx);
                this.PROPERTIES.add(prop);
            }

        } catch (IOException ex) {
            throw new IOException("Error reading script " + this.NAME, ex);
        } finally {
            ctx.popContext();

        }
    }

    /**
     * Writes the Script.
     *
     * @param output The LittleEndianDataOutput to write.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        if (this.NAME.isEmpty()) {
            output.writeUTF(this.NAME.toString());
            return;
        }

        output.writeUTF(this.NAME.toString());
        output.writeByte(this.STATUS);
        output.writeShort(this.PROPERTIES.size());

        for (Property prop : this.PROPERTIES) {
            prop.write(output);
        }
    }

    /**
     * @return The calculated size of the Script.
     */
    @Override
    public int calculateSize() {
        if (this.NAME.isEmpty()) {
            return 2;
        }

        int sum = 5 + NAME.length();
        sum += this.PROPERTIES.stream().mapToInt(v -> v.calculateSize()).sum();
        return sum;
    }

    final public IString NAME;
    final private byte STATUS;
    final List<Property> PROPERTIES;

}
