/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.io.IOException;
import java.util.Objects;
import restringer.LittleEndianInput;
import restringer.Profile;

/**
 * Describes a ChangeForm containing an NPC Reference.
 *
 * @author Mark Fairchild
 */
public class ChangeFormACHR extends GeneralElement implements ChangeFormData {

    /**
     * Creates a new <code>ChangeFormACHR</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param flags The ChangeForm flags.
     * @param refid The ChangeForm refid.
     * @param analysis
     * @param ess
     * @throws IOException
     */
    public ChangeFormACHR(LittleEndianInput input, Flags.Int flags, RefID refid, restringer.Analysis analysis, ESS ess) throws IOException {
        Objects.requireNonNull(input);

        int initialType;

        if (refid.getType() == RefID.Type.CREATED) {
            initialType = 5;
        } else if (flags.getFlag(CHANGE_REFR_PROMOTED) || flags.getFlag(CHANGE_REFR_CELL_CHANGED)) {
            initialType = 6;
        } else if (flags.getFlag(CHANGE_REFR_HAVOK_MOVE) || flags.getFlag(CHANGE_REFR_MOVE)) {
            initialType = 4;
        } else {
            initialType = 0;
        }

        super.readElement(input, "Initial", in -> new ChangeFormInitialData(in, initialType));

        if (flags.getFlag(CHANGE_REFR_HAVOK_MOVE)) {
            this.readBytesVS(input, "HavokData").toByteArray();
        } 

        if (flags.getFlag(CHANGE_FORM_FLAGS)) {            
            this.CHANGEFORMFLAGS = super.readElement(input, "ChangeFormFlags", in -> new ChangeFormFlags(in));
        } else {
            this.CHANGEFORMFLAGS = null;
        }

        if (flags.getFlag(CHANGE_REFR_BASEOBJECT)) {
            super.readRefID(input, "BaseObject");
        }

        if (flags.getFlag(CHANGE_REFR_SCALE)) {
            super.readFloat(input, "scale");
        }

        if (flags.getFlag(CHANGE_REFR_EXTRA_OWNERSHIP)
                || flags.getFlag(CHANGE_OBJECT_EXTRA_LOCK)
                || flags.getFlag(CHANGE_REFR_EXTRA_ENCOUNTER_ZONE)
                || flags.getFlag(CHANGE_REFR_EXTRA_GAME_ONLY)
                || flags.getFlag(CHANGE_OBJECT_EXTRA_AMMO)
                || flags.getFlag(CHANGE_DOOR_EXTRA_TELEPORT)
                || flags.getFlag(CHANGE_REFR_PROMOTED)
                || flags.getFlag(CHANGE_REFR_EXTRA_ACTIVATING)
                || flags.getFlag(CHANGE_OBJECT_EXTRA_ITEM_DATA)) {
            super.readElement(input, "ExtraData", in -> new ChangeFormExtraData(in));
        }

        if (flags.getFlag(CHANGE_REFR_INVENTORY)
                || flags.getFlag(CHANGE_REFR_LEVELED_INVENTORY)) {
            super.readVSElemArray(input, "Inventory", in -> new ChangeFormInventoryItem(in));
        }

        if (flags.getFlag(CHANGE_REFR_ANIMATION)) {
            super.readBytesVS(input, "Animations");
        }

       /* try (final PrintStream OUT = new PrintStream("Theft.inv.txt")) {
            for (int i = 0; i < buf.length; i++) {
                ByteArrayInputStream BIS = new ByteArrayInputStream(buf, i, 7);
                LittleEndianInput in = LittleEndianInputStream.wrap(BIS);
                final RefID REF = new RefID(in);
                final int COUNT = in.readInt();
                REF.resolveRefs(ess, this);
                REF.addNames(names, strings);
                if (REF.getName() != null) {
                    OUT.printf("Offset %08x : %06x : %s (x%d)\n", i, REF.getValue(), REF, COUNT);
                }
            }
        } catch (IOException ex) {

        }*/
    }

    /**
     * @return The <code>ChangeFormFlags</code> field.
     */
    public ChangeFormFlags getRefID() {
        return this.CHANGEFORMFLAGS;
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        return super.toString();
    }

    /**
     * @see AnalyzableElement#matches(restringer.Profile.Analysis,
     * restringer.Mod)
     * @param analysis
     * @param mod
     * @return
     */
    @Override
    public boolean matches(Profile.Analysis analysis, String mod) {
        return false;
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(restringer.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();

        BUILDER.append("<hr/><p>NPC:</p>");

        return BUILDER.toString();
    }

    final private ChangeFormFlags CHANGEFORMFLAGS;

    static final int CHANGE_FORM_FLAGS = log2(0x01);
    static final int CHANGE_REFR_MOVE = log2(0x02);
    static final int CHANGE_REFR_HAVOK_MOVE = log2(0x04);
    static final int CHANGE_REFR_CELL_CHANGED = log2(0x08);
    static final int CHANGE_REFR_SCALE = log2(0x10);
    static final int CHANGE_REFR_INVENTORY = log2(0x20);
    static final int CHANGE_REFR_EXTRA_OWNERSHIP = log2(0x40);
    static final int CHANGE_REFR_BASEOBJECT = log2(0x80);
    static final int CHANGE_REFR_PROMOTED = log2(0x2000000);
    static final int CHANGE_REFR_EXTRA_ACTIVATING = log2(0x4000000);
    static final int CHANGE_REFR_LEVELED_INVENTORY = log2(0x8000000);
    static final int CHANGE_REFR_ANIMATION = log2(0x10000000);
    static final int CHANGE_REFR_EXTRA_ENCOUNTER_ZONE = log2(0x20000000);
    static final int CHANGE_REFR_EXTRA_CREATED_ONLY = log2(0x40000000);
    static final int CHANGE_REFR_EXTRA_GAME_ONLY = log2(0x80000000);
    static final int CHANGE_OBJECT_EXTRA_ITEM_DATA = log2(0x400);
    static final int CHANGE_OBJECT_EXTRA_AMMO = log2(0x800);
    static final int CHANGE_OBJECT_EXTRA_LOCK = log2(0x1000);
    static final int CHANGE_DOOR_EXTRA_TELEPORT = log2(0x20000);
    static final int CHANGE_OBJECT_EMPTY = log2(0x200000);
    static final int CHANGE_OBJECT_OPEN_DEFAULT_STATE = log2(0x400000);
    static final int CHANGE_OBJECT_OPEN_STATE = log2(0x800000);

    static private int log2(int n) {
        return 31 - Integer.numberOfLeadingZeros(n);
    }

}
