/*
 * Copyright 2017 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.io.IOException;
import java.util.Objects;
import restringer.LittleEndianInput;

/**
 * Manages the data in one element of a change form's extra data.
 *
 * @author Mark Fairchild
 */
public class ChangeFormExtraDataData extends GeneralElement {

    /**
     * Creates a new <code>ChangeFormInitialData</code>.
     *
     * @param input
     * @param type
     * @throws IOException
     */
    public ChangeFormExtraDataData(LittleEndianInput input, int type) throws IOException {
        Objects.requireNonNull(input);
        
        switch (type) {
            case 22:
                this.NAME = "Worn";
                break;
            case 23:
                this.NAME = "WornLeft";
                break;
            case 24:
                this.NAME = "PackageStartLocation";                
                super.readRefID(input, "UNK");
                super.readFloats(input, "POS", 3);
                super.readFloat(input, "UNK1");
                break;
            case 25:
                this.NAME = "Package";
                super.readRefID(input, "UNK1");
                super.readRefID(input, "UNK2");
                super.readInt(input, "UNK3");
                super.readBytes(input, "UNK4", 3);
                break;
            case 26:
                this.NAME = "TrespassPackage";
                RefID ref = super.readRefID(input, "UNK");
                if (ref.getValue() != 0) {
                    assert false : "INCOMPLETE";
                }
                break;
            case 27:
                this.NAME = "RunOncePacks";
                super.readIntsVS(input, "PACKS");
                break;
            case 28:
                this.NAME = "ReferenceHandle";
                super.readRefID(input, "ID");
                break;
            case 29:
                this.NAME = "Unknown29";
                break;
            case 30:
                this.NAME = "LevCreaModifier";
                super.readInt(input, "MOD");
                break;
            case 31:
                this.NAME = "Ghost";
                super.readByte(input, "UNK");
                break;
            case 32:
                this.NAME = "UNKNOWN32";
                break;
            case 33:
                this.NAME = "Ownership";
                super.readRefID(input, "OWNER");
                break;
            case 34:
                this.NAME = "Global";
                super.readRefID(input, "UNK");
                break;
            case 35:
                this.NAME = "Rank";
                super.readRefID(input, "RANKID");
                break;
            case 36:
                this.NAME = "Count";
                super.readShort(input, "COUNT");
                break;
            case 37:
                this.NAME = "Health";
                super.readFloat(input, "HEALTH");
                break;
            case 39:
                this.NAME = "TimeLeft";
                super.readInt(input, "TIME");
                break;
            case 40:
                this.NAME = "Charge";
                super.readFloat(input, "CHARGE");
                break;
            case 42:
                this.NAME = "Lock";
                super.readBytes(input, "UNKS", 2);
                super.readRefID(input, "UNKID");
                super.readInts(input, "UNKS2", 2);
                break;
            case 43:
                this.NAME = "Teleport";
                super.readFloats(input, "POS", 3);
                super.readFloats(input, "ROT", 3);
                super.readByte(input, "UNK");
                super.readRefID(input, "REF");
                break;
            case 44:
                this.NAME = "MapMarker";
                super.readByte(input, "UNK");
                break;
            case 45:
                this.NAME = "LeveledCreature";
                super.readRefID(input, "UNK1");
                super.readRefID(input, "UNK2");
                Flags.Int flags = super.readElement(input, "NPCChangeFlags", i -> Flags.readIntFlags(i));
                super.readElement(input, "NPC", in -> new ChangeFormNPC(in, flags));
                break;
            case 46:
                this.NAME = "LeveledItem";
                super.readInt(input, "UNK");
                super.readByte(input, "UNK2");
                break;
            case 47:
                this.NAME = "Scale";
                super.readFloat(input, "scale");
                break;
            case 49:
                this.NAME = "NonActorMagicCaster";
                super.readInt(input, "unk1");
                super.readRefID(input, "ref1");
                super.readInt(input, "unk2");
                super.readInt(input, "unk3");
                super.readRefID(input, "ref2");
                super.readFloat(input, "unk4");
                super.readRefID(input, "ref3");
                super.readRefID(input, "ref4");
                break;
            case 50:
                this.NAME = "NonActorMagicTarget";
                super.readRefID(input, "ref");
                super.readVSElemArray(input, "targets", in -> new MagicTarget(in));
                break;
            case 52:
                this.NAME = "PlayerCrimeList";
                super.readLongsVS(input, "list");
                break;
            case 56:
                this.NAME = "ItemDropper";
                super.readRefID(input, "unk");
                break;
            case 61:
                this.NAME = "CannotWear";
                break;
            case 62:
                this.NAME = "ExtraPoison";
                super.readRefID(input, "ref");
                super.readInt(input, "unk");
                break;
            case 68:
                this.NAME = "FriendHits";
                super.readFloatsVS(input, "unk");
                break;
            case 69:
                this.NAME = "HeadingTarget";
                super.readRefID(input, "targetID");
                break;
            case 72:
                this.NAME = "StartingWorldOrCell";
                super.readRefID(input, "worldOrCellID");
                break;
            case 73:
                this.NAME = "HotKey";
                super.readByte(input, "unk");
                break;
            case 112:
                this.NAME = "EncounterZone";
                super.readRefID(input, "REF");
                break;
            case 136:
                this.NAME = "AliasInstanceArray";
                super.readVSElemArray(input, "ALIASES", in -> new AliasInstance(in));
                break;
            case 140:
                this.NAME = "PromotedRef";
                super.readVSElemArray(input, "REFS", in -> new RefID(in));
                break;
            default:
                throw new IOException("Unknown ExtraData: type=" + type);
        }
        
    }
    
    final public String NAME;
    
    static private class AliasInstance extends GeneralElement {

        AliasInstance(LittleEndianInput input) throws IOException {
            super.readRefID(input, "REF");
            super.readInt(input, "FORMID");
        }
    }
    
    static private class MagicTarget extends GeneralElement {

        MagicTarget(LittleEndianInput input) throws IOException {
            super.readRefID(input, "REF");
            super.readByte(input, "unk1");
            super.readVSVal(input, "unk2");
            super.readBytesVS(input, "data");
        }
    }
}
