/*
 * Copyright 2017 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import it.unimi.dsi.fastutil.bytes.ByteArrayList;
import it.unimi.dsi.fastutil.floats.FloatArrayList;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.longs.LongArrayList;
import it.unimi.dsi.fastutil.objects.Object2ObjectLinkedOpenHashMap;
import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;
import it.unimi.dsi.fastutil.shorts.ShortArrayList;
import java.io.IOException;
import java.util.Arrays;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.SortedMap;
import java.util.stream.Collectors;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInput;

/**
 * A very generalized element. It's not quite as efficient or customizable, but
 * it's good for elements that can have a range of different members depending
 * on flags.
 *
 * @author Mark Fairchild
 */
public class GeneralElement implements Element {

    /**
     * Create a new <code>GeneralElement</code>.
     */
    public GeneralElement() {
        this.DATA = new Object2ObjectLinkedOpenHashMap<>();
    }

    final public SortedMap<String, Object> getValues() {
        return new Object2ObjectLinkedOpenHashMap<>(this.DATA);
    }

    final public boolean hasVal(String name) {
        Objects.requireNonNull(name);
        return this.DATA.containsKey(name);
    }

    final public Object getVal(String name) {
        Objects.requireNonNull(name);
        return this.DATA.get(name);
    }

    final public Element getElement(String name) {
        Objects.requireNonNull(name);
        Object val = this.DATA.get(name);
        if (null != val && val instanceof Element) {
            return (Element) val;
        }
        return null;
    }

    /**
     * Adds an <code>Element</code>.
     *
     * @param e The element.
     * @param name The name of the element.
     * @param <T> The element type.
     * @return The element.
     */
    /*final public <T extends Element> T addElement(T e, String name) {
        Objects.requireNonNull(e);
        Objects.requireNonNull(name);
        return this.addValue(name, e);
    }*/

    /**
     * Reads a byte.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @return The byte.
     * @throws IOException
     */
    final public byte readByte(LittleEndianInput input, String name) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        byte val = input.readByte();
        return this.addValue(name, val);
    }

    /**
     * Reads a short.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @return The short.
     * @throws IOException
     */
    final public short readShort(LittleEndianInput input, String name) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        short val = input.readShort();
        return this.addValue(name, val);
    }

    /**
     * Reads an int.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @return The int.
     * @throws IOException
     */
    final public int readInt(LittleEndianInput input, String name) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        int val = input.readInt();
        return this.addValue(name, val);
    }

    /**
     * Reads an long.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @return The long.
     * @throws IOException
     */
    final public long readLong(LittleEndianInput input, String name) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        long val = input.readLong();
        return this.addValue(name, val);
    }

    /**
     * Reads a float.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @return The float.
     * @throws IOException
     */
    final public float readFloat(LittleEndianInput input, String name) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        float val = input.readFloat();
        return this.addValue(name, val);
    }

    /**
     * Reads a zstring.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @return The string.
     * @throws IOException
     */
    final public String readZString(LittleEndianInput input, String name) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        String val = input.readZString(256);
        return this.addValue(name, val);
    }

    /**
     * Reads a VSVal.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @param reader The element reader.
     * @param <T> The element type.
     * @return The element.
     * @throws IOException
     */
    final public <T extends Element> T readElement(LittleEndianInput input, String name, ElementReader<T> reader) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        T element = reader.read(input);
        return this.addValue(name, element);
    }

    /**
     * Reads a refid.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @return The RefID.
     * @throws IOException
     */
    final public RefID readRefID(LittleEndianInput input, String name) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        RefID ref = new RefID(input);
        return this.addValue(name, ref);
    }

    /**
     * Reads a VSVal.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @return The RefID.
     * @throws IOException
     */
    final public VSVal readVSVal(LittleEndianInput input, String name) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        VSVal val = new VSVal(input);
        return this.addValue(name, val);
    }

    /**
     * Reads a fixed-length byte array.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @param size The size of the array.
     * @return The array.
     * @throws IOException
     */
    final public ByteArrayList readBytes(LittleEndianInput input, String name, int size) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        assert 0 < size;
        assert size <= 256;

        ByteArrayList val = new ByteArrayList(size);
        for (int i = 0; i < size; i++) {
            val.add(input.readByte());
        }
        return this.addValue(name, val);
    }

    /**
     * Reads a fixed-length short array.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @param size The size of the array.
     * @return The array.
     * @throws IOException
     */
    final public ShortArrayList readShorts(LittleEndianInput input, String name, int size) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        assert 0 < size;
        assert size <= 256;

        ShortArrayList val = new ShortArrayList(size);
        for (int i = 0; i < size; i++) {
            val.add(input.readShort());
        }
        return this.addValue(name, val);
    }

    /**
     * Reads a fixed-length int array.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @param size The size of the array.
     * @return The array.
     * @throws IOException
     */
    final public IntArrayList readInts(LittleEndianInput input, String name, int size) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        assert 0 < size;
        assert size <= 256;

        IntArrayList val = new IntArrayList(size);
        for (int i = 0; i < size; i++) {
            val.add(input.readInt());
        }
        return this.addValue(name, val);
    }

    /**
     * Reads a fixed-length long array.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @param size The size of the array.
     * @return The array.
     * @throws IOException
     */
    final public LongArrayList readLongs(LittleEndianInput input, String name, int size) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        assert 0 < size;
        assert size <= 256;

        LongArrayList val = new LongArrayList(size);
        for (int i = 0; i < size; i++) {
            val.add(input.readLong());
        }
        return this.addValue(name, val);
    }

    /**
     * Reads a fixed-length float array.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @param size The size of the array.
     * @return The array.
     * @throws IOException
     */
    final public FloatArrayList readFloats(LittleEndianInput input, String name, int size) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        assert 0 < size;
        assert size <= 256;

        FloatArrayList val = new FloatArrayList(size);
        for (int i = 0; i < size; i++) {
            val.add(input.readFloat());
        }
        return this.addValue(name, val);
    }

    /**
     * Reads a fixed-length byte array using a VSVal.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @return The array.
     * @throws IOException
     */
    final public ByteArrayList readBytesVS(LittleEndianInput input, String name) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        final VSVal COUNT = this.readVSVal(input, name + "_COUNT");
        assert 0 <= COUNT.getValue();
        return this.readBytes(input, name, COUNT.getValue());
    }

    /**
     * Reads a fixed-length short array using a VSVal.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @return The array.
     * @throws IOException
     */
    final public ShortArrayList readShortsVS(LittleEndianInput input, String name) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        final VSVal COUNT = this.readVSVal(input, name + "_COUNT");
        assert 0 <= COUNT.getValue();
        return this.readShorts(input, name, COUNT.getValue());
    }

    /**
     * Reads a fixed-length int array using a VSVal.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @return The array.
     * @throws IOException
     */
    final public IntArrayList readIntsVS(LittleEndianInput input, String name) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        final VSVal COUNT = this.readVSVal(input, name + "_COUNT");
        assert 0 <= COUNT.getValue();
        return this.readInts(input, name, COUNT.getValue());
    }

    /**
     * Reads a fixed-length long array using a VSVal.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @return The array.
     * @throws IOException
     */
    final public LongArrayList readLongsVS(LittleEndianInput input, String name) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        final VSVal COUNT = this.readVSVal(input, name + "_COUNT");
        assert 0 <= COUNT.getValue();
        return this.readLongs(input, name, COUNT.getValue());
    }

    /**
     * Reads a fixed-length float array using a VSVal.
     *
     * @param input The inputstream.
     * @param name The name of the new element.
     * @return The array.
     * @throws IOException
     */
    final public FloatArrayList readFloatsVS(LittleEndianInput input, String name) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(name);
        final VSVal COUNT = this.readVSVal(input, name + "_COUNT");
        assert 0 <= COUNT.getValue();
        return this.readFloats(input, name, COUNT.getValue());
    }

    /**
     * Reads an array of elements using a supplier functional.
     *
     * @param input The inputstream.
     * @param reader
     * @param name The name of the new element.
     * @param <T> The element type.
     * @return The array.
     * @throws IOException
     */
    final public <T extends Element> ObjectArrayList<T> readVSElemArray(LittleEndianInput input, String name, ElementReader<T> reader) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(reader);
        Objects.requireNonNull(name);

        final VSVal COUNT = this.readVSVal(input, name + "_COUNT");
        assert 0 <= COUNT.getValue();
        ObjectArrayList<T> val = new ObjectArrayList<>(COUNT.getValue());

        for (int i = 0; i < COUNT.getValue(); i++) {
            T e = reader.read(input);
            val.add(e);
        }
        return this.addValue(name, val);
    }

    /**
     *
     * @param name
     * @param val
     */
    private <T> T addValue(String name, T val) {
        if (!SUPPORTED.stream().anyMatch(type -> type.isInstance(val))) {
            throw new IllegalStateException(String.format("Invalid type for %s: %s", name, val.getClass()));
        }

        this.DATA.put(name, val);
        return val;
    }

    /**
     * @see Element#write(restringer.LittleEndianDataOutput)
     * @param output
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        for (Object v : this.DATA.values()) {
            if (Element.class.isInstance(v)) {
                output.writeESSElement((Element) v);
            } else if (v.getClass() == Byte.class) {
                output.writeByte((Byte) v);
            } else if (v.getClass() == Short.class) {
                output.writeShort((Short) v);
            } else if (v.getClass() == Integer.class) {
                output.writeInt((Integer) v);
            } else if (v.getClass() == Float.class) {
                output.writeFloat((Float) v);
            } else if (v.getClass() == String.class) {
                output.writeZString((String) v);
            } else if (v.getClass() == ByteArrayList.class) {
                final ByteArrayList ARR = (ByteArrayList) v;
                for (byte b : ARR) {
                    output.writeByte(b);
                }
            } else if (v.getClass() == ShortArrayList.class) {
                final ShortArrayList ARR = (ShortArrayList) v;
                for (short s : ARR) {
                    output.writeShort(s);
                }
            } else if (v.getClass() == IntArrayList.class) {
                final IntArrayList ARR = (IntArrayList) v;
                for (int i : ARR) {
                    output.writeInt(i);
                }
            } else if (v.getClass() == FloatArrayList.class) {
                final FloatArrayList ARR = (FloatArrayList) v;
                for (float f : ARR) {
                    output.writeFloat(f);
                }
            } else if (v.getClass() == ObjectArrayList.class) {
                final ObjectArrayList<Element> ARR = (ObjectArrayList<Element>) v;
                for (Element e : ARR) {
                    output.writeESSElement(e);
                }
            } else {
                throw new IllegalStateException("Unknown element: " + v.getClass());
            }
        }
    }

    /**
     * @see Element#calculateSize()
     */
    @Override
    public int calculateSize() {
        int sum = 0;

        for (Object v : this.DATA.values()) {
            if (Element.class.isInstance(v)) {
                sum += ((Element) v).calculateSize();
            } else if (v.getClass() == Byte.class) {
                sum += 1;
            } else if (v.getClass() == Short.class) {
                sum += 2;
            } else if (v.getClass() == Integer.class) {
                sum += 4;
            } else if (v.getClass() == Float.class) {
                sum += 4;
            } else if (v.getClass() == String.class) {
                sum += 1 + ((String) v).getBytes().length;
            } else if (v.getClass() == ByteArrayList.class) {
                sum += 1 * ((ByteArrayList) v).size();
            } else if (v.getClass() == ShortArrayList.class) {
                sum += 2 * ((ShortArrayList) v).size();
            } else if (v.getClass() == IntArrayList.class) {
                sum += 4 * ((IntArrayList) v).size();
            } else if (v.getClass() == FloatArrayList.class) {
                sum += 4 * ((FloatArrayList) v).size();
            } else if (v.getClass() == ObjectArrayList.class) {
                final ObjectArrayList<Element> ARR = (ObjectArrayList<Element>) v;
                sum += ARR.stream().mapToInt(w -> w.calculateSize()).sum();
            } else {
                throw new IllegalStateException("Unknown element: " + v.getClass());
            }
        }

        return sum;
    }

    /**
     * @see Element#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        this.DATA.values()
                .stream()
                .filter(v -> v instanceof Element)
                .map(v -> (Element) v)
                .forEach(v -> v.resolveRefs(ess, owner));
    }

    /**
     * @see Element#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
        this.DATA.values()
                .stream()
                .filter(v -> v instanceof Element)
                .map(v -> (Element) v)
                .forEach(v -> v.addNames(analysis));
    }

    /**
     *
     * @return String representation.
     */
    @Override
    public String toString() {
        return DATA.keySet()
                .stream()
                .map(n -> String.format("%s=%s", n, getVal(n)))
                .collect(Collectors.joining(", ", "[", "]"));
    }

    /**
     * Stores the actual data.
     */
    final private Map<String, Object> DATA;

    static final private Set<Class> SUPPORTED = new ObjectOpenHashSet<>(Arrays.asList(
            new Class[]{
                Element.class,
                Byte.class,
                Short.class,
                Integer.class,
                Float.class,
                String.class,
                ByteArrayList.class,
                ShortArrayList.class,
                IntArrayList.class,
                FloatArrayList.class,
                ObjectArrayList.class
            }
    ));

    @FunctionalInterface
    static public interface ElementReader<T extends Element> {

        T read(LittleEndianInput input) throws IOException;
    }

}
