/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.io.IOException;
import java.util.Objects;
import restringer.LittleEndianDataOutput;
import restringer.ess.papyrus.Papyrus;
import restringer.LittleEndianInput;

/**
 * Describes 3-byte formIDs from Skyrim savegames.
 *
 * @author Mark Fairchild
 * @version 2016/06/19
 */
final public class GlobalData implements Element {

    /**
     * Creates a new <code>GlobalData</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param ctx The ESSContext.
     * @throws IOException
     */
    public GlobalData(LittleEndianInput input, ESSContext ctx) throws IOException {
        assert null != input;
        this.TYPE = input.readInt();

        int size = input.readInt();
        final byte[] DATA = new byte[size];
        int read = input.read(DATA);

        if (read != size) {
            throw new IOException(String.format("The block size should be %d, but only %d bytes could be read.", size, read));
        }

        if (this.TYPE == 1001) {
            Papyrus papyrus;

            try {
                papyrus = new Papyrus(DATA, ctx, false);
            } catch (IOException ex) {
                if (ctx.GAME.isSLE()) {
                    papyrus = new Papyrus(DATA, ctx, true);
                } else {
                    throw ex;
                }
            }
            this.BLOCK = papyrus;
            
        } else {
            this.BLOCK = new DefaultGlobalDataBlock(DATA);
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        assert null != output;
        output.writeInt(this.TYPE);
        output.writeInt(this.BLOCK.calculateSize());
        this.BLOCK.write(output);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        return 8 + this.BLOCK.calculateSize();
    }

    /**
     * @return The value of the type field.
     */
    public int getType() {
        return this.TYPE;
    }

    /**
     * @return The data block.
     */
    public GlobalDataBlock getDataBlock() {
        return this.BLOCK;
    }

    /**
     *
     * @return @throws IOException
     */
    public Papyrus getPapyrus() throws IOException {
        if (!(this.TYPE == 1001 && this.BLOCK instanceof Papyrus)) {
            throw new IllegalStateException("Not a papyrus block.");
        }

        return (Papyrus) this.BLOCK;
    }

    /**
     * @see Element#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
        this.BLOCK.addNames(analysis);
    }

    /**
     * @see Element#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        this.BLOCK.resolveRefs(ess, owner);
    }

    /**
     * @see Object#toString()
     * @return
     */
    @Override
    public String toString() {
        return super.toString() + ": type " + Integer.toString(this.TYPE);
    }

    /**
     * @see Object#hashCode()
     * @return
     */
    @Override
    public int hashCode() {
        int hash = 7;
        hash = 89 * hash + Objects.hashCode(this.BLOCK);
        return hash;
    }

    /**
     * @see Object#equals(java.lang.Object)
     * @return
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final GlobalData other = (GlobalData) obj;
        return Objects.equals(this.BLOCK, other.BLOCK);
    }

    final private int TYPE;
    final private GlobalDataBlock BLOCK;
    
}
