/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import restringer.Game;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.Objects;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;

/**
 * Describes header of Skyrim savegames.
 *
 * @author Mark Fairchild
 * @version 2016/06/19
 */
final public class Header implements Element {

    /**
     * Creates a new <code>Header</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @throws IOException
     */
    public Header(LittleEndianInput input) throws IOException {
        Objects.requireNonNull(input);

        // Read the magic string.
        final byte[] magicBuffer = new byte[20];
        input.read(magicBuffer, 0, 3);

        switch (new String(magicBuffer, 0, 3).toUpperCase()) {
            case "TES":
                input.read(magicBuffer, 3, 10);
                this.MAGIC = new String(magicBuffer, 0, 13);
                break;
            case "FO4":
                input.read(magicBuffer, 3, 9);
                this.MAGIC = new String(magicBuffer, 0, 12);
                break;
            default:
                throw new IllegalArgumentException("Unrecognized header: " + new String(magicBuffer, 0, 3));
        }

        // Read the header size.
        final int HEADERSIZE = input.readInt();
        assert HEADERSIZE < 256;

        // Read the version number.
        this.VERSION = input.readInt();

        // Identify which game produced the savefile.
        if (this.MAGIC.equalsIgnoreCase("TESV_SAVEGAME")) {
            if (8 <= this.VERSION && this.VERSION <= 9) {
                this.GAME = Game.SKYRIM_LE;
            } else if (this.VERSION == 12) {
                this.GAME = Game.SKYRIM_SE;
            } else {
                throw new IllegalArgumentException("Unknown version of Skyrim: " + this.VERSION);
            }
        } else if (this.MAGIC.equalsIgnoreCase("FO4_SAVEGAME")) {
            if (11 <= this.VERSION) {
                this.GAME = Game.FALLOUT4;
            } else {
                throw new IllegalArgumentException("Unknown version of Fallout4: " + this.VERSION);
            }
        } else {
            throw new IllegalArgumentException("Unknown game: " + this.MAGIC);
        }

        this.SAVENUMBER = input.readInt();
        this.NAME = WString.read(input);
        this.LEVEL = input.readInt();
        this.LOCATION = WString.read(input);
        this.GAMEDATE = WString.read(input);
        this.RACEID = WString.read(input);
        this.SEX = input.readShort();
        this.CURRENT_XP = input.readFloat();
        this.NEEDED_XP = input.readFloat();
        this.FILETIME = input.readLong();
        this.SCREENSHOT_WIDTH = input.readInt();
        this.SCREENSHOT_HEIGHT = input.readInt();

        if (this.GAME == Game.SKYRIM_SE) {
            this.UNKNOWN_SHORT = input.readShort();
        } else {
            this.UNKNOWN_SHORT = 0;
        }

        assert HEADERSIZE == this.partialSize() : String.format("Header size should be %d, found %d.", HEADERSIZE, this.partialSize());

        switch (this.GAME) {
            case SKYRIM_LE:
                this.BYPP = 3;
                break;
            case FALLOUT4:
            case SKYRIM_SE:
                this.BYPP = 4;
                break;
            default:
                throw new IllegalArgumentException("Invalid game: " + this.GAME);
        }

        this.SCREENSHOT = new byte[this.BYPP * this.SCREENSHOT_WIDTH * this.SCREENSHOT_HEIGHT];
        input.readFully(this.SCREENSHOT);

        this.IMAGE = new BufferedImage(this.SCREENSHOT_WIDTH, this.SCREENSHOT_HEIGHT, BufferedImage.TYPE_INT_RGB);
        int x = 0;
        int y = 0;

        for (int i = 0; i < this.SCREENSHOT.length; i += this.BYPP) {
            int rgb = 0;
            rgb |= (this.SCREENSHOT[i + 2] & 0xFF);
            rgb |= (this.SCREENSHOT[i + 1] & 0xFF) << 8;
            rgb |= (this.SCREENSHOT[i + 0] & 0xFF) << 16;
            if (this.BYPP == 4) {
                rgb |= (this.SCREENSHOT[i + 3] & 0xFF) << 24;
            }
            IMAGE.setRGB(x, y, rgb);

            x++;
            if (x >= this.SCREENSHOT_WIDTH) {
                x = 0;
                y++;
            }
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);
        output.write(this.MAGIC.getBytes());
        output.writeInt(this.partialSize());
        output.writeInt(this.VERSION);
        output.writeInt(this.SAVENUMBER);
        output.writeESSElement(this.NAME);
        output.writeInt(this.LEVEL);
        output.writeESSElement(this.LOCATION);
        output.writeESSElement(this.GAMEDATE);
        output.writeESSElement(this.RACEID);
        output.writeShort(this.SEX);
        output.writeFloat(this.CURRENT_XP);
        output.writeFloat(this.NEEDED_XP);
        output.writeLong(this.FILETIME);
        output.writeInt(this.SCREENSHOT_WIDTH);
        output.writeInt(this.SCREENSHOT_HEIGHT);

        if (this.GAME == Game.SKYRIM_SE) {
            output.writeShort(this.UNKNOWN_SHORT);
        }

        output.write(this.SCREENSHOT);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 4;
        sum += this.partialSize();
        sum += this.MAGIC.length();
        sum += this.SCREENSHOT.length;
        return sum;
    }

    /**
     * The size of the header, not including the magic string, the size itself,
     * or the screenshot.
     *
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    private int partialSize() {
        int sum = 0;
        sum += 4; // version
        sum += 4; // savenumber
        sum += this.NAME.calculateSize();
        sum += 4; // level
        sum += this.LOCATION.calculateSize();
        sum += this.GAMEDATE.calculateSize();
        sum += this.RACEID.calculateSize();
        sum += 2; // sex
        sum += 4; // current xp
        sum += 4; // needed xp
        sum += 8; // filtime
        sum += 8; // screenshot size
        sum += (this.GAME == Game.SKYRIM_SE ? 2 : 0);
        return sum;
    }

    /**
     * @param width The width for scaling.
     * @return A <code>ImageIcon</code> that can be used to display the
     * screenshot.
     */
    public javax.swing.ImageIcon getImage(int width) {
        double scale = ((double) width) / ((double) this.SCREENSHOT_WIDTH);
        int newWidth = (int) (scale * this.SCREENSHOT_WIDTH);
        int newHeight = (int) (scale * this.SCREENSHOT_HEIGHT);
        
        final BufferedImage IMG = new BufferedImage(newWidth, newHeight, BufferedImage.TYPE_INT_RGB);
        final java.awt.geom.AffineTransform XFORM = AffineTransform.getScaleInstance(scale, scale);
        final java.awt.Graphics2D G = (java.awt.Graphics2D) IMG.createGraphics();        
        G.setRenderingHint(java.awt.RenderingHints.KEY_ANTIALIASING, java.awt.RenderingHints.VALUE_ANTIALIAS_ON);
        G.setRenderingHint(java.awt.RenderingHints.KEY_INTERPOLATION, java.awt.RenderingHints.VALUE_INTERPOLATION_BICUBIC);
        G.drawRenderedImage(this.IMAGE, XFORM);
        return new javax.swing.ImageIcon(IMG);
    }

    /**
     * @see Element#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
    }

    /**
     * @see Element#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
    }

    final public String MAGIC;
    final public int VERSION;
    final public int SAVENUMBER;
    final public WString NAME;
    final public int LEVEL;
    final public WString LOCATION;
    final public WString GAMEDATE;
    final public WString RACEID;
    final public short SEX;
    final public float CURRENT_XP;
    final public float NEEDED_XP;
    final public long FILETIME;
    final public int SCREENSHOT_WIDTH;
    final public int SCREENSHOT_HEIGHT;
    final public int BYPP;
    final public short UNKNOWN_SHORT;
    final public Game GAME;
    final private byte[] SCREENSHOT;
    final private BufferedImage IMAGE;

}
