/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.io.IOException;
import java.util.Objects;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInput;
import restringer.ess.papyrus.EID;
import restringer.ess.papyrus.PapyrusElement;
import restringer.ess.papyrus.ScriptInstance;

/**
 * Describes 3-byte formIDs from Skyrim savegames.
 *
 * @author Mark Fairchild
 * @version 2016/06/19
 */
final public class RefID implements Element, Linkable, Comparable<RefID> {

    /**
     * A reusable <code>RefID</code> that is set to 0.
     */
    static public RefID NONE = new RefID(0);

    /**
     * Creates a new <code>RefID</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @throws IOException
     */
    public RefID(LittleEndianInput input) throws IOException {
        assert null != input;
        int val = 0;
        val += input.readUnsignedByte();
        val <<= 8;
        val += input.readUnsignedByte();
        val <<= 8;
        val += input.readUnsignedByte();
        this.DATA = val;
        this.formID = 0;
        this.plugin = null;
        this.form = null;
        this.name = null;
    }

    /**
     * Creates a new <code>RefID</code> directly.
     *
     * @param newData
     */
    public RefID(int newData) {
        this.DATA = newData;
        this.formID = 0;
        this.plugin = null;
        this.form = null;
        this.name = null;
    }

    /**
     * Creates a new <code>RefID</code> directly.
     *
     * @param newData
     */
    public RefID(EID newData) {
        this.DATA = (int) newData.longValue();
        this.formID = 0;
        this.plugin = null;
        this.form = null;
        this.name = null;
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        assert null != output;
        output.writeByte(this.DATA >> 16);
        output.writeByte(this.DATA >> 8);
        output.writeByte(this.DATA);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        return 3;
    }

    /**
     * @return The type of RefID.
     */
    public Type getType() {
        int index = 0x3 & this.DATA >>> 22;
        switch (index) {
            case 0:
                return Type.FORMIDX;
            case 1:
                return Type.DEFAULT;
            case 2:
                return Type.CREATED;
            case 3:
            default:
                return Type.UNKNOWN;
        }
    }

    /**
     * @return The value of the RefID.
     */
    public int getValue() {
        return (this.DATA & 0x3FFFFF);
    }

    /**
     * @return The form ID, or 0 if the refID has no form id.
     */
    public int getFormID() {
        return this.formID;
    }

    /**
     * @return The <code>ChangeForm</code>, or null if the refID has no form.
     */
    public ChangeForm getForm() {
        return this.form;
    }

    /**
     * @return The plugin, or null if none.
     */
    public Plugin getPlugin() {
        return this.plugin;
    }

    /**
     * @return The name field, if any.
     */
    public String getName() {
        return this.name;
    }

    /**
     * @return Checks if the <code>RefID</code> points to a created form that
     * doesn't exist. The result is invalid unless references have been
     * resolved.
     */
    public boolean isNonexistentCreated() {
        return this.getType() == Type.CREATED && null == this.form;
    }

    /**
     * @see Element#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    public void resolveRefs(ESS ess, Element owner) {
        Objects.requireNonNull(ess);

        // Get a reference to the changeform to which this RefID refers.
        this.form = ess.getChangeForms().get(this);
        if (null != this.form && null != owner && owner instanceof ScriptInstance) {
            this.form.addRefHolder((ScriptInstance) owner);
        }

        // If this is a formid-index refid, get the plugin.
        switch (this.getType()) {
            case FORMIDX:
                int index = this.getValue() - 1;
                if (index >= 0 && index < ess.getFormIDs().length) {
                    this.formID = ess.getFormIDs()[index];
                    int pluginIndex = this.formID >>> 24;
                    this.plugin = ess.getPluginInfo().getPlugins().get(pluginIndex);
                }
                break;
            case DEFAULT:
                this.plugin = ess.getPluginInfo().getPlugins().get(0);
                break;
            case CREATED:
                break;
            case UNKNOWN:
            default:
                break;
        }

        if (null != this.plugin) {
            if (owner instanceof ChangeForm) {
                this.plugin.getForms().add((ChangeForm) owner);
            } else if (owner instanceof ScriptInstance) {
                this.plugin.getInstances().add((ScriptInstance) owner);
            }
        }
    }

    /**
     * @see Element#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
        switch (this.getType()) {
            case FORMIDX: {
                int id = this.getFormID();
                this.name = analysis.getName(id);
                break;
            }
            case DEFAULT: {
                int id = this.getValue();
                this.name = analysis.getName(id);
                break;
            }
            default:
        }
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        String namePart = (this.name != null ? " " + this.name : "");

        if (this.DATA == 0) {
            return "00000000";

        } else if (this.getType() == Type.FORMIDX && this.name != null) {
            return "FormIDX-" + zeroPad8(this.getFormID()) + " (" + this.name + ")";

        } else if (this.getType() == Type.FORMIDX) {
            return "FormIDX-" + zeroPad8(this.getFormID());

        } else if (this.name != null) {
            return this.getType() + "-" + zeroPad6(this.getValue()) + " (" + this.name + ")";

        } else {
            return this.getType() + "-" + zeroPad6(this.getValue());
        }
    }

    /**
     * @see restringer.ess.Linkable#toHTML()
     * @return
     */
    @Override
    public String toHTML() {
        return String.format("<a href=\"refid://%08x\">%s</a>", this.DATA, this.toString());
    }

    @Override
    public int compareTo(RefID other) {
        Objects.requireNonNull(other);
        return Integer.compareUnsigned(this.DATA, other.DATA);
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 89 * hash + this.DATA;
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final RefID other = (RefID) obj;
        return this.DATA == other.DATA;
    }

    final private int DATA;
    private int formID;
    private Plugin plugin;
    private ChangeForm form;
    private String name;

    /**
     * The four types of RefIDs.
     */
    static public enum Type {
        FORMIDX, DEFAULT, CREATED, UNKNOWN;
    }

    /**
     * Zero-pads the hexadecimal representation of an integer so that it is a
     * full 4 bytes long.
     *
     * @param val The value to convert to hexadecimal and pad.
     * @return The zero-padded string.
     */
    static private String zeroPad8(int val) {
        String hex = Integer.toHexString(val);
        int length = hex.length();
        assert length <= ZEROES.length;
        return ZEROES[8 - length] + hex;
    }

    /**
     * Zero-pads the hexadecimal representation of an integer so that it is a
     * full 3 bytes long.
     *
     * @param val The value to convert to hexadecimal and pad.
     * @return The zero-padded string.
     */
    static private String zeroPad6(int val) {
        String hex = Long.toHexString(val);
        int length = hex.length();
        assert length < ZEROES.length;
        return ZEROES[6 - length] + hex;
    }

    /**
     * An array of strings of zeroes with the length matching the index.
     */
    static final private String[] ZEROES = makeZeroes();

    static private String[] makeZeroes() {
        String[] zeroes = new String[16];
        zeroes[0] = "";

        for (int i = 1; i < zeroes.length; i++) {
            zeroes[i] = zeroes[i - 1] + "0";
        }

        return zeroes;
    }

}
