/*
 * Copyright 2016 Mark.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.io.IOException;
import java.util.Objects;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInput;

/**
 * A Skyrim variable-size value.
 *
 * @author Mark Fairchild
 * @version 2016/09/02
 */
final public class VSVal implements Element {

    /**
     * Creates a new <code>ChangeForm</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @throws IOException
     */
    public VSVal(LittleEndianInput input) throws IOException {
        int b1 = input.readUnsignedByte();
        int size = b1 & 0x3;

        switch (size) {
            case 0: {
                this.DATA = new byte[]{(byte) b1};
                this.VALUE = b1 >> 2;
                return;
            }
            case 1: {
                int b2 = input.readUnsignedByte();
                this.DATA = new byte[]{(byte) b1, (byte) b2};
                this.VALUE = (b1 | (b2 << 8)) >> 2;
                return;
            }
            case 2: {
                int b2 = input.readUnsignedByte();
                int b3 = input.readUnsignedByte();
                this.DATA = new byte[]{(byte) b1, (byte) b2, (byte) b3};
                this.VALUE = (b1 | (b2 << 8) | (b3 << 16)) >> 2;
                return;
            }
            default:
                throw new IOException();
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);
        output.write(this.DATA);
    }
    
    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        return this.DATA.length;
    }
    
    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        return String.format("%d", this.VALUE);
    }
    
    /**
     * @return The value stored in the VSVal.
     */
    public int getValue() {
        return this.VALUE;
    }
    
    
    /**
     * @see Element#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
    }

    /**
     * @see Element#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
    }

    final private byte[] DATA;
    final private int VALUE;

}
