/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.ess.ESS;
import restringer.ess.Element;

/**
 * Describes an active script's data in a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/06/21
 */
final public class ActiveScriptData implements PapyrusElement, HasID {

    /**
     * Creates a new <code>ActiveScriptData</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param scripts The list of scripts.
     * @param actives The list of ActiveScripts.
     * @param instances The list of Instances.
     * @param refs The list of Instances.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public ActiveScriptData(LittleEndianInput input, ScriptMap scripts, ActiveScriptMap actives, InstanceMap instances, ReferenceMap refs, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(scripts);
        Objects.requireNonNull(ctx);

        this.ID = EID.read4byte(input);
        if (!actives.containsKey(this.ID)) {
            throw new IOException("Invalid ID on ActiveScriptData.");
        }

        this.MAJORVERSION = input.readByte();
        this.MINORVERSION = input.readByte();
        assert this.MINORVERSION >= 1 && this.MINORVERSION <= 2 : "Wrong minor version = " + this.MINORVERSION;

        this.UNKNOWN = Variable.read(input, ctx);

        this.FLAG = input.readByte();
        this.UNKNOWNBYTE = input.readByte();
        this.UNKNOWN2 = ((this.FLAG & 0x01) == 0x01 ? input.readInt() : 0);
        this.UNKNOWN3 = input.readByte();

        switch (this.UNKNOWN3) {
            case 1:
            case 2:
            case 3:
                this.UNKNOWN4 = new Unknown4(input, this.UNKNOWN3, ctx);
                break;
            default:
                this.UNKNOWN4 = null;
        }

        if (ctx.GAME.isFO4() && (null == this.UNKNOWN4 || !this.UNKNOWN4.STRING.equals("TerminalRunResults"))) {
            this.ATTACHED = EID.read8byte(input);
            /*if (!this.ATTACHED.isZero()) {
                if (instances.containsKey(this.ATTACHED)) {
                    System.out.printf("Mystery ID matched an instance: %s\n", this.ATTACHED);
                } else if (refs.containsKey(this.ATTACHED)) {
                    System.out.printf("Mystery ID matched a reference: %s\n", this.ATTACHED);
                } else {
                    System.out.printf("Mystery ID did not match anything: %s\n", this.ATTACHED);
                }
            }*/
        } else {
            this.ATTACHED = null;
        }

        int stackFrameCount = input.readInt();
        this.STACKFRAMES = new ArrayList<>(stackFrameCount);
        for (int i = 0; i < stackFrameCount; i++) {
            try {
                StackFrame frame = new StackFrame(input, scripts, ctx);
                this.STACKFRAMES.add(frame);
            } catch (IOException ex) {
                throw new IOException(String.format("Problem with StackFrame Data. Processed %d/%d.", i, stackFrameCount), ex);

            }
        }

        this.UNKNOWN5 = (stackFrameCount != 0 ? input.readByte() : 0);
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        assert null != output;
        output.writeESSElement(this.ID);
        output.writeByte(this.MAJORVERSION);
        output.writeByte(this.MINORVERSION);
        output.writeESSElement(this.UNKNOWN);
        output.writeByte(this.FLAG);
        output.writeByte(this.UNKNOWNBYTE);

        if ((this.FLAG & 0x01) == 0x01) {
            output.writeInt(this.UNKNOWN2);
        }

        output.writeByte(this.UNKNOWN3);

        if (null != this.UNKNOWN4) {
            output.writeESSElement(this.UNKNOWN4);
        }

        if (null != this.ATTACHED) {
            output.writeESSElement(this.ATTACHED);
        }

        output.writeInt(this.STACKFRAMES.size());

        for (StackFrame frame : this.STACKFRAMES) {
            output.writeESSElement(frame);
        }

        if (!this.STACKFRAMES.isEmpty()) {
            output.writeByte(this.UNKNOWN5);
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 2;
        sum += this.ID.calculateSize();
        sum += this.UNKNOWN.calculateSize();
        sum += 2;
        sum += ((this.FLAG & 0x01) == 0x01 ? 4 : 0);
        sum += 5;
        sum += (null == this.ATTACHED ? 0 : this.ATTACHED.calculateSize());
        sum += (null == this.UNKNOWN4 ? 0 : this.UNKNOWN4.calculateSize());
        sum += (this.STACKFRAMES.isEmpty() ? 0 : 1);
        sum += this.STACKFRAMES.stream().mapToInt(frame -> frame.calculateSize()).sum();

        return sum;
    }

    /**
     * @return The ID of the papyrus element.
     */
    @Override
    public EID getID() {
        return this.ID;
    }

    /**
     * @return The major version field.
     */
    public byte getMajorVersion() {
        return this.MAJORVERSION;
    }

    /**
     * @return The minor version field.
     */
    public int getMinorVersion() {
        return this.MINORVERSION;
    }

    /**
     * @return The flag field.
     */
    public int getFlag() {
        return this.FLAG;
    }

    /**
     * @return The unknown variable field.
     */
    public Variable getUnknownVar() {
        return this.UNKNOWN;
    }

    /**
     * @return The Unknown4 field.
     */
    public Unknown4 getUnknown4() {
        return this.UNKNOWN4;
    }

    /**
     * @return The Unknown byte field.
     */
    public byte getUnknownByte() {
        return this.UNKNOWNBYTE;
    }

    /**
     * @return The Unknown2 field.
     */
    public int getUnknown2() {
        return this.UNKNOWN2;
    }

    /**
     * @return The Unknown3 field.
     */
    public byte getUnknown3() {
        return this.UNKNOWN3;
    }

    /**
     * @return The Unknown5 field.
     */
    public byte getUnknown5() {
        return this.UNKNOWN5;
    }

    /**
     * @return The attached field.
     */
    public EID getAttached() {
        return this.ATTACHED;
    }
    
    /**
     * @return The script's stack frames.
     */
    public List<StackFrame> getStackFrames() {
        return this.STACKFRAMES;
    }

    /**
     * @see PapyrusElement#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
        this.UNKNOWN.addNames(analysis);

        if (null != this.UNKNOWN4) {
            this.UNKNOWN4.addNames(analysis);
        }

        this.STACKFRAMES.forEach(frame -> frame.addNames(analysis));
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        this.UNKNOWN.resolveRefs(ess, owner);

        if (null != this.UNKNOWN4) {
            this.UNKNOWN4.resolveRefs(ess, owner);
        }

        this.STACKFRAMES.forEach(frame -> frame.resolveRefs(ess, owner));
    }

    final private EID ID;
    final private byte MAJORVERSION;
    final private byte MINORVERSION;
    final private Variable UNKNOWN;
    final private byte FLAG;
    final private byte UNKNOWNBYTE;
    final private int UNKNOWN2;
    final private byte UNKNOWN3;
    final private Unknown4 UNKNOWN4;
    final private EID ATTACHED;
    final private List<StackFrame> STACKFRAMES;
    final private byte UNKNOWN5;

}
