/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import restringer.ess.AnalyzableElement;
import java.io.IOException;
import java.util.Objects;
import java.util.SortedSet;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.Profile;
import restringer.ess.ESS;
import restringer.ess.Element;
import restringer.ess.Linkable;

/**
 * Describes a function message in a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/06/21
 */
final public class FunctionMessage implements PapyrusElement, AnalyzableElement, Linkable {//, HasID {

    /**
     * Creates a new <code>FunctionMessage</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param scripts The list of scripts.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public FunctionMessage(LittleEndianInput input, ScriptMap scripts, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(scripts);
        Objects.requireNonNull(ctx);

        this.UNKNOWN = input.readByte();
        this.ID = (this.UNKNOWN <= 2 ? EID.read4byte(input) : null);
        this.FLAG = input.readByte();

        if (this.FLAG > 0) {
            this.MESSAGE = new FunctionMessageData(input, this, scripts, ctx);
        } else {
            this.MESSAGE = null;
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        assert null != output;

        output.writeByte(this.UNKNOWN);

        if (this.UNKNOWN <= 2) {
            output.writeESSElement(this.ID);
        }

        output.writeByte(this.FLAG);

        if (null != this.MESSAGE) {
            this.MESSAGE.write(output);
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 2;
        sum += (this.UNKNOWN <= 2 ? 4 : 0);
        sum += (null == this.MESSAGE ? 0 : this.MESSAGE.calculateSize());
        return sum;
    }

    /**
     * @return The ID of the papyrus element.
     */
    //@Override
    public EID getID() {
        return this.ID;
    }

    /**
     * @return The message field.
     */
    public FunctionMessageData getMessage() {
        return this.MESSAGE;
    }

    /**
     * @return The corresponding <code>Script</code>.
     */
    public Script getScript() {
        if (null == this.MESSAGE) {
            return null;
        }
        return this.MESSAGE.getScript();
    }

    /**
     * @see restringer.ess.Linkable#toHTML() 
     * @return 
     */
    @Override
    public String toHTML() {
        return String.format("<a href=\"message://%s\">%s</a>", this.ID, this.toString());
    }
    
    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        if (this.FLAG > 0) {
            TString scriptName = (null != this.MESSAGE ? this.MESSAGE.getScriptName() : null);

            if (this.isUndefined()) {
                if (this.UNKNOWN <= 2) {
                    return "MSG #" + scriptName + "# (" + this.ID + ")";
                } else {
                    return "MSG #" + scriptName;
                }
            } else if (this.UNKNOWN <= 2) {
                return "MSG " + scriptName + " (" + this.ID + ")";
            } else {
                return "MSG " + scriptName;
            }

        } else if (this.UNKNOWN <= 2) {
            return "MSG " + this.FLAG + "," + EID.pad8(this.UNKNOWN) + " (" + this.ID + ")";
            
        } else {
            return "MSG " + this.FLAG + "," + EID.pad8(this.UNKNOWN);
        }
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis, restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(restringer.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();
        BUILDER.append(String.format("<html><h3>FUNCTIONMESSAGE</h3>"));

        if (null != analysis && null != this.MESSAGE) {
            SortedSet<String> mods = analysis.SCRIPT_ORIGINS.get(this.MESSAGE.getScriptName());
            if (null != mods) {
                String mod = mods.last();
                BUILDER.append(String.format("<p>Probably running code from mod %s.</p>", mod));
            }
        }

        BUILDER.append("<p>");
        if (null != this.MESSAGE) {
            BUILDER.append(String.format("Function: %s<br/>", this.MESSAGE.getFName()));
        }

        BUILDER.append(String.format("ID: %s<br/>", this.ID));
        BUILDER.append(String.format("Flag: %s<br/>", this.FLAG));
        BUILDER.append(String.format("Unknown: %d<br/>", this.UNKNOWN));       
        BUILDER.append("</p>");
        BUILDER.append("</html>");
        return BUILDER.toString();
    }

    /**
     * @see AnalyzableElement#matches(restringer.Profile.Analysis,
     * restringer.Mod)
     * @param analysis
     * @param mod
     * @return
     */
    @Override
    public boolean matches(Profile.Analysis analysis, String mod) {
        Objects.requireNonNull(analysis);
        Objects.requireNonNull(mod);
        return null != this.MESSAGE && this.MESSAGE.matches(analysis, mod);
    }

    /**
     * @see PapyrusElement#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
        if (null != this.MESSAGE) {
            this.MESSAGE.addNames(analysis);
        }
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        if (null != this.MESSAGE) {
            this.MESSAGE.resolveRefs(ess, this);
        }
    }

    /**
     * @return A flag indicating if the <code>FunctionMessage</code> is
     * undefined.
     *
     */
    public boolean isUndefined() {
        if (null != this.MESSAGE) {
            return this.MESSAGE.isUndefined();
        }

        return false;
    }

    final private byte UNKNOWN;
    final private EID ID;
    final private byte FLAG;
    final private FunctionMessageData MESSAGE;

}
