/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.Objects;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.ess.ESS;
import restringer.ess.Element;

/**
 * Describes a script member in a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/06/19
 */
public class MemberDesc implements PapyrusElement {

    /**
     * Creates a new <code>MemberData</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public MemberDesc(LittleEndianInput input, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(ctx);
        this.NAME = ctx.STRINGS.read(input);
        this.TYPE = ctx.STRINGS.read(input);
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);
        this.NAME.write(output);
        this.TYPE.write(output);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        return this.NAME.calculateSize() + this.TYPE.calculateSize();
    }

    /**
     * @return The ID of the papyrus element.
     */
    public TString getName() {
        return this.NAME;
    }

    /**
     * @return The type of the array.
     */
    public TString getType() {
        return this.TYPE;
    }

    /**
     * @return The corresponding <code>Script</code> or null if the member
     * is not associated with a script.
     */
    public Script getScript() {
        return this.script;
    }

    /**
     * @see PapyrusElement#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
    }
    
    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        this.script = ess.getPapyrus().getScripts().getOrDefault(this.TYPE, null);
        this.NAME.addRefHolder(owner);
        this.TYPE.addRefHolder(owner);
    }
    
    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        return this.TYPE + " " + this.NAME;
    }
    
    final private TString NAME;
    final private TString TYPE;
    private Script script;
}
