/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.ess.ESS;
import restringer.ess.Element;
import restringer.pex.Opcode;

/**
 * Describes a script member in a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/06/19
 */
final public class OpcodeData implements PapyrusElement {

    /**
     * A reusable instance of the NOP instruction.
     */
    static final public OpcodeData NOP = new OpcodeData();

    /**
     * Creates a new <code>OpcodeData</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public OpcodeData(LittleEndianInput input, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(ctx);
                
        int code = input.readByte();
        this.OPCODE = OPCODES[code];
        
        if (this.OPCODE.ARGS > 0) {
            this.PARAMETERS = new ArrayList<>(this.OPCODE.ARGS);
            
            for (int i = 0; i < OPCODE.ARGS; i++) {
                Parameter var = new Parameter(input, ctx.STRINGS);
                this.PARAMETERS.add(var);
            }
        } else if (this.OPCODE.ARGS < 0) {
            this.PARAMETERS = new ArrayList<>(-this.OPCODE.ARGS);
            
            for (int i = 0; i < 1 - OPCODE.ARGS; i++) {
                Parameter var = new Parameter(input, ctx.STRINGS);
                this.PARAMETERS.add(var);
            }
            
            int numVargs = this.PARAMETERS.get(-this.OPCODE.ARGS).getIntValue();
            this.PARAMETERS.ensureCapacity(this.PARAMETERS.size() + numVargs);
            
            for (int i = 0; i < numVargs; i++) {
                Parameter var = new Parameter(input, ctx.STRINGS);
                this.PARAMETERS.add(var);
            }
        } else {
            this.PARAMETERS = new ArrayList<>(0);
        }
    }

    /**
     * Creates a new <code>OpcodeData</code> for the NOP instruction.
     */
    private OpcodeData() {
        this.OPCODE = Opcode.NOP;
        this.PARAMETERS = new ArrayList<>(0);
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        assert null != output;
        output.writeByte(this.OPCODE.ordinal());
        
        for (Parameter var : this.PARAMETERS) {
            var.write(output);
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 1;
        sum += this.PARAMETERS.stream().mapToInt(var -> var.calculateSize()).sum();
        return sum;
    }

    /**
     * @return The opcode.
     */
    public Opcode getOpcode() {
        return this.OPCODE;
    }

    /**
     * @return The list of instruction parameters.
     */
    public List<Parameter> getParameters() {
        return java.util.Collections.unmodifiableList(this.PARAMETERS);
    }

    /**
     * @see PapyrusElement#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
        this.PARAMETERS.forEach(v -> v.addNames(analysis));
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        this.PARAMETERS.forEach(v -> v.resolveRefs(ess, owner));
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {        
        final StringBuilder BUILDER = new StringBuilder();
        BUILDER.append(this.OPCODE);
        this.PARAMETERS.forEach(p -> BUILDER.append(' ').append(p.toValueString()));
        return BUILDER.toString();
    }
    
    @Override
    public int hashCode() {
        int hash = 3;
        hash = 29 * hash + Objects.hashCode(this.OPCODE);
        hash = 29 * hash + Objects.hashCode(this.PARAMETERS);
        return hash;
    }
    
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final OpcodeData other = (OpcodeData) obj;
        if (this.OPCODE != other.OPCODE) {
            return false;
        }
        return Objects.equals(this.PARAMETERS, other.PARAMETERS);
    }

    final private Opcode OPCODE;
    final private ArrayList<Parameter> PARAMETERS;
    static final private Opcode[] OPCODES = Opcode.values();
    
}
