/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.*;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInputStream;
import restringer.ess.GlobalDataBlock;
import restringer.ess.ESS;
import restringer.ess.ESSContext;
import restringer.ess.Element;
import java.util.stream.Collectors;

/**
 * Describes a the data for a <code>GlobalData</code> when it is the Papyrus
 * script section.
 *
 * @author Mark Fairchild
 * @version 2016/06/21
 */
final public class Papyrus implements PapyrusElement, GlobalDataBlock {

    /**
     * Creates a new <code>Papyrus</code> by reading from a byte buffer.
     *
     * @param buffer The data.
     * @param essCTX The ESSContext.
     * @param applySTBCorrection A flag indicating that <code>StringTable</code>
     * should ATTEMPT to correct for the string table bug.
     * @throws IOException
     */
    public Papyrus(byte[] buffer, ESSContext essCTX, boolean applySTBCorrection) throws IOException {
        Objects.requireNonNull(buffer);
        this.ORIGINAL_DATA = buffer;

        if (buffer.length < 20) {
            throw new IOException("The Papyrus block is missing. This can happen if Skyrim is running too many mods or too many scripts.\nUnfortunately, there is literally nothing I can do to help you with this.");
        }

        //System.out.println("DEBUGGING");
        try (LittleEndianInputStream input = LittleEndianInputStream.wrap(buffer)) {
            int sum = 0;
            int i = 0;

            // Read the header.            
            this.HEADER = input.readShort();
            sum += 2;

            // Read the string table.
            this.STRINGS = new StringTable(input, essCTX.GAME, applySTBCorrection);
            sum += this.STRINGS.calculateSize();
            assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());

            // Create the PapyrusContext.
            final PapyrusContext CTX = new PapyrusContext(essCTX, this.STRINGS);

            // Read the script table and struct table.
            if (CTX.GAME.isFO4()) {
                // Fallout4 has scripts and structs, so the data is a bit
                // different right here.

                int scriptCount = input.readInt();
                this.SCRIPTS = new ScriptMap(scriptCount);

                int structDefCount = input.readInt();
                this.STRUCTDEFS = new StructDefMap(structDefCount);

                // Read the scripts.
                try {
                    for (i = 0; i < scriptCount; i++) {
                        Script script = new Script(input, CTX);
                        this.SCRIPTS.put(script.getName(), script);
                    }
                    sum += 4 + this.SCRIPTS.values().stream().mapToInt(v -> v.calculateSize()).sum();
                } catch (IOException ex) {
                    throw new IOException(String.format("Problem with Scripts. Processed %d/%d.", i, scriptCount), ex);
                }

                // Read the structs.
                try {
                    for (i = 0; i < structDefCount; i++) {
                        StructDef struct = new StructDef(input, CTX);
                        this.STRUCTDEFS.put(struct.getName(), struct);
                    }
                    sum += 4 + this.STRUCTDEFS.values().stream().mapToInt(v -> v.calculateSize()).sum();
                } catch (IOException ex) {
                    throw new IOException(String.format("Problem with StructDefs. Processed %d/%d.", i, structDefCount), ex);
                }
                assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());

            } else {
                // Skyrim and SkyrimSE just have scripts.
                // No real differences between them here.
                int scriptCount = input.readInt();
                this.SCRIPTS = new ScriptMap(scriptCount);
                this.STRUCTDEFS = null;

                try {
                    for (i = 0; i < scriptCount; i++) {
                        Script script = new Script(input, CTX);
                        this.SCRIPTS.put(script.getName(), script);

                    }
                    sum += 4 + this.SCRIPTS.values().stream().mapToInt(v -> v.calculateSize()).sum();
                    assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());
                } catch (IOException | AssertionError ex) {
                    throw new IOException(String.format("Problem with Scripts. Processed %d/%d.", i, scriptCount), ex);
                }

            }

            // Read the script instance table.
            int instanceCount = input.readInt();
            this.INSTANCES = new InstanceMap(instanceCount);

            try {
                for (i = 0; i < instanceCount; i++) {
                    ScriptInstance instance = new ScriptInstance(input, this.SCRIPTS, CTX);
                    this.INSTANCES.put(instance.getID(), instance);
                }
                sum += 4 + this.INSTANCES.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();
                assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());
            } catch (IOException | AssertionError ex) {
                throw new IOException(String.format("Problem with Script Instances. Processed %d/%d.", i, instanceCount), ex);
            }

            // Read the reference table.
            int referenceCount = input.readInt();
            this.REFERENCES = new ReferenceMap(referenceCount);

            try {
                for (i = 0; i < referenceCount; i++) {
                    Reference reference = new Reference(input, this.SCRIPTS, CTX);
                    this.REFERENCES.put(reference.getID(), reference);
                }
                sum += 4 + this.REFERENCES.values().stream().mapToInt(v -> v.calculateSize()).sum();
                assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());
            } catch (IOException ex) {
                throw new IOException(String.format("Problem with References. Processed %d/%d.", i, referenceCount), ex);
            }

            // Read the struct body table.
            if (CTX.GAME.isFO4()) {
                // Read the struct table.
                int structCount = input.readInt();
                this.STRUCTS = new StructMap(structCount);

                try {
                    for (i = 0; i < structCount; i++) {
                        Struct struct = new Struct(input, this.STRUCTDEFS, CTX);
                        this.STRUCTS.put(struct.getID(), struct);
                    }
                    sum += 4 + this.STRUCTS.values().stream().mapToInt(v -> v.calculateSize()).sum();
                    assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());
                } catch (IOException ex) {
                    throw new IOException(String.format("Problem with Struct Instances. Processed %d/%d.", i, structCount), ex);
                }
            } else {
                this.STRUCTS = null;
            }

            // Read the array table.
            int arrayCount = input.readInt();
            this.ARRAYS = new ArrayMap(arrayCount);

            try {
                for (i = 0; i < arrayCount; i++) {
                    ArrayInfo info = new ArrayInfo(input, CTX);
                    this.ARRAYS.put(info.getID(), info);
                }
                sum += 4 + this.ARRAYS.values().stream().mapToInt(v -> v.calculateSize()).sum();
                assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());
            } catch (IOException ex) {
                throw new IOException(String.format("Problem with Arrays. Processed %d/%d.", i, arrayCount), ex);
            }

            this.PAPYRUS_RUNTIME = input.readInt();
            sum += 4;
            assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());

            // Read the active script table.
            int activeScriptCount = input.readInt();
            this.ACTIVESCRIPTS = new ActiveScriptMap(activeScriptCount);

            try {
                for (i = 0; i < activeScriptCount; i++) {
                    ActiveScript active = new ActiveScript(input, CTX);
                    this.ACTIVESCRIPTS.put(active.getID(), active);
                }
                sum += 4 + this.ACTIVESCRIPTS.values().stream().mapToInt(v -> v.calculateSize()).sum();
                assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());
            } catch (IOException ex) {
                throw new IOException(String.format("Problem with ActiveScripts. Processed %d/%d.", i, activeScriptCount), ex);
            }

            // Read the script instance data table and associate the data
            // with the relevant script instances.
            try {
                for (i = 0; i < instanceCount; i++) {
                    ScriptData data = new ScriptData(input, CTX);
                    ScriptInstance instance = this.INSTANCES.get(data.getID());
                    instance.setData(data);
                }
                sum += this.INSTANCES.values().parallelStream().mapToInt(v -> v.getData().calculateSize()).sum();
                assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());
            } catch (Exception | Error ex) {
                throw new IOException(String.format("Problem with Script Instance Data. Processed %d/%d.", i, instanceCount), ex);
            }

            // Read the reference data table and associate the data
            // with the relevant references.
            try {
                for (i = 0; i < referenceCount; i++) {
                    ReferenceData data = new ReferenceData(input, CTX);
                    Reference ref = this.REFERENCES.get(data.getID());
                    ref.setData(data);
                }
                sum += this.REFERENCES.values().stream().mapToInt(v -> v.getData().calculateSize()).sum();
                assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());
            } catch (Exception | Error ex) {
                throw new IOException(String.format("Problem with Reference Data. Processed %d/%d.", i, referenceCount), ex);
            }

            // Read the struct body table.
            if (CTX.GAME.isFO4()) {
                int structCount = this.STRUCTS.size();
                try {
                    for (i = 0; i < structCount; i++) {
                        StructData data = new StructData(input, this.STRUCTS, CTX);
                        Struct struct = this.STRUCTS.get(data.getID());
                        struct.setData(data);
                    }
                    sum += this.STRUCTS.values().stream().mapToInt(v -> v.getData().calculateSize()).sum();
                    assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());
                } catch (Exception | Error ex) {
                    throw new IOException(String.format("Problem with Struct Data. Processed %d/%d.", i, structCount), ex);
                }
            }

            // Read the array data table and associate the data
            // with the relevant arrays.
            try {
                for (i = 0; i < arrayCount; i++) {
                    ArrayData data = new ArrayData(input, this.ARRAYS, CTX);
                    ArrayInfo array = this.ARRAYS.get(data.getID());
                    array.setData(data);
                }
                sum += this.ARRAYS.values().stream().mapToInt(v -> v.getData().calculateSize()).sum();
                assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());
            } catch (Exception | Error ex) {
                throw new IOException(String.format("Problem with Array Data. Processed %d/%d.", i, arrayCount), ex);
            }

            // Read the active script data table and associate the data
            // with the relevant script instance.
            try {
                for (i = 0; i < activeScriptCount; i++) {
                    if (i == 67030) {
                        int k = 0;
                    } 
                    ActiveScriptData data = new ActiveScriptData(input, this.SCRIPTS, this.ACTIVESCRIPTS, this.INSTANCES, this.REFERENCES, CTX);
                    ActiveScript script = this.ACTIVESCRIPTS.get(data.getID());
                    script.setData(data);
                }
                sum += this.ACTIVESCRIPTS.values().stream().mapToInt(v -> v.getData().calculateSize()).sum();
                assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());
            } catch (Error | IOException ex) {
                throw new IOException(String.format("Problem with ActiveScript Data. Processed %d/%d.", i, activeScriptCount), ex);
            }

            // Read the function message table.
            int functionMessageCount = input.readInt();
            this.FUNCTIONMESSAGES = new ArrayList<>(functionMessageCount);

            try {
                for (i = 0; i < functionMessageCount; i++) {
                    FunctionMessage message = new FunctionMessage(input, this.SCRIPTS, CTX);
                    this.FUNCTIONMESSAGES.add(message);
                }
                sum += 4 + this.FUNCTIONMESSAGES.stream().mapToInt(v -> v.calculateSize()).sum();
                assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());
            } catch (IOException ex) {
                throw new IOException(String.format("Problem with Function Messages. Processed %d/%d.", i, functionMessageCount), ex);
            }

            // Read the first SuspendedStack table.
            int stack1Count = input.readInt();
            this.SUSPENDEDSTACKS1 = new ArrayList<>(stack1Count);

            try {
                for (i = 0; i < stack1Count; i++) {
                    SuspendedStack stack = new SuspendedStack(input, this.SCRIPTS, CTX);
                    this.SUSPENDEDSTACKS1.add(stack);
                }
                sum += 4 + this.SUSPENDEDSTACKS1.stream().mapToInt(v -> v.calculateSize()).sum();
                assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());
            } catch (IOException ex) {
                throw new IOException(String.format("Problem with Suspended Stacks (list 1). Processed %d/%d.", i, stack1Count), ex);
            }

            // Read the second SuspendedStack table.
            int stack2Count = input.readInt();
            this.SUSPENDEDSTACKS2 = new ArrayList<>(stack2Count);

            try {
                for (i = 0; i < stack2Count; i++) {
                    SuspendedStack stack = new SuspendedStack(input, this.SCRIPTS, CTX);
                    this.SUSPENDEDSTACKS2.add(stack);
                }
                sum += 4 + this.SUSPENDEDSTACKS2.stream().mapToInt(v -> v.calculateSize()).sum();
                assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());
            } catch (IOException ex) {
                throw new IOException(String.format("Problem with Suspended Stacks (list 2). Processed %d/%d.", i, stack2Count), ex);
            }

            // Read the "unknown" fields.
            this.UNK1 = input.readInt();
            this.UNK2 = (this.UNK1 == 0 ? 0 : input.readInt());
            sum += (this.UNK1 == 0 ? 4 : 8);
            assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());

            int unknownCount = input.readInt();
            this.UNKS = new ArrayList<>(unknownCount);

            if (CTX.GAME.isID64()) {
                for (i = 0; i < unknownCount; i++) {
                    EID id = EID.read8byte(input);
                    this.UNKS.add(id);
                }
            } else {
                for (i = 0; i < unknownCount; i++) {
                    EID id = EID.read4byte(input);
                    this.UNKS.add(id);
                }
            }
            sum += 4 + this.UNKS.parallelStream().mapToInt(v -> v.calculateSize()).sum();
            assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());

            // Read the queued unbinds... whatever those are.
            int queuedUnbindCount = input.readInt();
            this.UNBINDS = new ArrayList<>(queuedUnbindCount);

            for (i = 0; i < queuedUnbindCount; i++) {
                QueuedUnbind unbind = new QueuedUnbind(input, this.INSTANCES, CTX);
                this.UNBINDS.add(unbind);
            }
            sum += 4 + this.UNBINDS.stream().mapToInt(v -> v.calculateSize()).sum();
            assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());

            // Read the save file version field.
            this.SAVE_FILE_VERSION = input.readShort();
            sum += 2;
            assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());

            // Stuff the remaining data into a buffer.
            int remaining = input.available();
            this.OTHERDATA = new byte[remaining];
            int read = input.read(this.OTHERDATA);
            sum += this.OTHERDATA.length;
            assert read == remaining;
            assert sum == input.getPosition() : String.format("sum=%d, pos=%d", sum, input.getPosition());
            assert sum == this.calculateSize() : String.format("Summed = %d, calculated = %d", sum, this.calculateSize());
            assert sum == buffer.length : String.format("Summed = %d, buffer length = %d", sum, buffer.length);

        } catch (OutOfMemoryError ex) {
            throw new IOException("Out of memory while reading the Papyrus section.", ex);
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        output = new LittleEndianDataOutput(output);
        int sum = 0;

        assert null != output;
        output.writeShort(this.HEADER);
        sum += 2;
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the string table.
        this.STRINGS.write(output);
        sum += this.STRINGS.calculateSize();
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the script table.
        if (this.STRUCTDEFS != null) {
            output.writeInt(this.SCRIPTS.size());
            output.writeInt(this.STRUCTDEFS.size());

            for (Script script : this.SCRIPTS.values()) {
                script.write(output);
            }
            for (StructDef struct : this.STRUCTDEFS.values()) {
                struct.write(output);
            }

            sum += 4 + this.SCRIPTS.values().stream().mapToInt(v -> v.calculateSize()).sum();
            sum += 4 + this.STRUCTDEFS.values().stream().mapToInt(v -> v.calculateSize()).sum();
            assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());
        } else {
            output.writeInt(this.SCRIPTS.size());
            for (Script script : this.SCRIPTS.values()) {
                script.write(output);
            }
            sum += 4 + this.SCRIPTS.values().stream().mapToInt(v -> v.calculateSize()).sum();
            assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());
        }

        // Write the script instance table.
        output.writeInt(this.INSTANCES.size());
        for (ScriptInstance instance : this.INSTANCES.values()) {
            instance.write(output);
        }
        sum += 4 + this.INSTANCES.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the reference table.
        output.writeInt(this.REFERENCES.size());
        for (Reference ref : this.REFERENCES.values()) {
            ref.write(output);
        }
        sum += 4 + this.REFERENCES.values().stream().mapToInt(v -> v.calculateSize()).sum();
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the struct instance table.
        if (null != this.STRUCTS) {
            output.writeInt(this.STRUCTS.size());

            for (Struct struct : this.STRUCTS.values()) {
                struct.write(output);
            }
            sum += 4 + this.STRUCTS.values().stream().mapToInt(v -> v.calculateSize()).sum();
            assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());
        }

        // Write the array table.
        output.writeInt(this.ARRAYS.size());
        for (ArrayInfo info : this.ARRAYS.values()) {
            info.write(output);
        }
        sum += 4 + this.ARRAYS.values().stream().mapToInt(v -> v.calculateSize()).sum();
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        output.writeInt(this.PAPYRUS_RUNTIME);
        sum += 4;
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the active script table.
        output.writeInt(this.ACTIVESCRIPTS.size());
        for (ActiveScript script : this.ACTIVESCRIPTS.values()) {
            script.write(output);
        }
        sum += 4 + this.ACTIVESCRIPTS.values().stream().mapToInt(v -> v.calculateSize()).sum();
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the script instance data table.
        for (ScriptInstance instance : this.INSTANCES.values()) {
            ScriptData data = instance.getData();
            data.write(output);
        }
        sum += this.INSTANCES.values().parallelStream().mapToInt(v -> v.getData().calculateSize()).sum();
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the reference data table.
        for (Reference ref : this.REFERENCES.values()) {
            ref.getData().write(output);
        }
        sum += this.REFERENCES.values().stream().mapToInt(v -> v.getData().calculateSize()).sum();
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the struct instance table.
        if (null != this.STRUCTS) {
            for (Struct struct : this.STRUCTS.values()) {
                struct.getData().write(output);
            }
            sum += this.STRUCTS.values().stream().mapToInt(v -> v.getData().calculateSize()).sum();
            assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());
        }

        // Write the array data table.
        for (ArrayInfo info : this.ARRAYS.values()) {
            info.getData().write(output);
        }
        sum += this.ARRAYS.values().stream().mapToInt(v -> v.getData().calculateSize()).sum();
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the active script data table.
        for (ActiveScript script : this.ACTIVESCRIPTS.values()) {
            script.getData().write(output);
        }
        sum += this.ACTIVESCRIPTS.values().stream().mapToInt(v -> v.getData().calculateSize()).sum();
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the function message table.
        output.writeInt(this.FUNCTIONMESSAGES.size());
        for (FunctionMessage message : this.FUNCTIONMESSAGES) {
            message.write(output);
        }
        sum += 4 + this.FUNCTIONMESSAGES.stream().mapToInt(v -> v.calculateSize()).sum();
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the first suspended stack table.
        output.writeInt(this.SUSPENDEDSTACKS1.size());
        for (SuspendedStack stack : this.SUSPENDEDSTACKS1) {
            stack.write(output);
        }
        sum += 4 + this.SUSPENDEDSTACKS1.stream().mapToInt(v -> v.calculateSize()).sum();
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the first suspended stack table.
        output.writeInt(this.SUSPENDEDSTACKS2.size());
        for (SuspendedStack stack : this.SUSPENDEDSTACKS2) {
            stack.write(output);
        }
        sum += 4 + this.SUSPENDEDSTACKS2.stream().mapToInt(v -> v.calculateSize()).sum();
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the "unknown" fields.
        output.writeInt(this.UNK1);
        if (this.UNK1 != 0) {
            output.writeInt(this.UNK2);
        }
        sum += (this.UNK1 == 0 ? 4 : 8);
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        output.writeInt(this.UNKS.size());
        for (EID id : this.UNKS) {
            output.writeESSElement(id);
        }
        sum += 4 + this.UNKS.parallelStream().mapToInt(v -> v.calculateSize()).sum();
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the queued unbind table.
        output.writeInt(this.UNBINDS.size());
        for (QueuedUnbind unbind : this.UNBINDS) {
            unbind.write(output);
        }
        sum += 4 + this.UNBINDS.stream().mapToInt(v -> v.calculateSize()).sum();
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the save file version field.
        output.writeShort(this.SAVE_FILE_VERSION);
        sum += 2;
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());

        // Write the remaining data.
        output.write(this.OTHERDATA);
        sum += this.OTHERDATA.length;
        assert sum == output.getPosition() : String.format("sum=%d, pos=%d", sum, output.getPosition());
        assert sum == this.calculateSize() : String.format("Summed = %d, calculated = %d", sum, this.calculateSize());
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 2;

        sum += this.STRINGS.calculateSize();

        sum += 4;
        sum += this.SCRIPTS.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();

        if (null != this.STRUCTDEFS) {
            sum += 4;
            sum += this.STRUCTDEFS.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();
        }

        sum += 4;
        sum += this.INSTANCES.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += 4;
        sum += this.REFERENCES.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();

        if (null != this.STRUCTS) {
            sum += 4;
            sum += this.STRUCTS.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();
            sum += this.STRUCTS.values().parallelStream().mapToInt(v -> v.getData().calculateSize()).sum();
        }

        sum += 4;
        sum += this.ARRAYS.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += 4;

        sum += 4;
        sum += this.ACTIVESCRIPTS.values().parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += this.INSTANCES.values().parallelStream().mapToInt(v -> v.getData().calculateSize()).sum();
        sum += this.REFERENCES.values().parallelStream().mapToInt(v -> v.getData().calculateSize()).sum();
        sum += this.ARRAYS.values().parallelStream().mapToInt(v -> v.getData().calculateSize()).sum();
        sum += this.ACTIVESCRIPTS.values().parallelStream().mapToInt(v -> v.getData().calculateSize()).sum();

        sum += 4;
        sum += this.FUNCTIONMESSAGES.parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += 4;
        sum += this.SUSPENDEDSTACKS1.parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += 4;
        sum += this.SUSPENDEDSTACKS2.parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += (this.UNK1 == 0 ? 4 : 8);
        sum += 4 + this.UNKS.parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += 4;
        sum += this.UNBINDS.parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += 2; // save file version.

        sum += this.OTHERDATA.length;

        return sum;
    }

    /**
     * @return Accessor for the string table.
     */
    public StringTable getStringTable() {
        return this.STRINGS;
    }

    /**
     * @return Accessor for the list of scripts.
     */
    public ScriptMap getScripts() {
        return this.SCRIPTS;
    }

    /**
     * @return Accessor for the list of structdefs.
     */
    public StructDefMap getStructDefs() {
        if (null == this.STRUCTDEFS) {
            return new StructDefMap(0);
        }
        return this.STRUCTDEFS;
    }

    /**
     * @return Accessor for the list of script instances.
     */
    public InstanceMap getInstances() {
        return this.INSTANCES;
    }

    /**
     * @return Accessor for the list of references.
     */
    public ReferenceMap getReferences() {
        return this.REFERENCES;
    }

    /**
     * @return Accessor for the list of structs.
     */
    public StructMap getStructs() {
        if (null == this.STRUCTS) {
            return new StructMap(0);
        }
        return this.STRUCTS;
    }

    /**
     * @return Accessor for the list of arrays.
     */
    public ArrayMap getArrays() {
        return this.ARRAYS;
    }

    /**
     * @return Accessor for the list of active scripts.
     */
    public ActiveScriptMap getActiveScripts() {
        return this.ACTIVESCRIPTS;
    }

    /**
     * @return Accessor for the list of function messages.
     */
    public List<FunctionMessage> getFunctionMessages() {
        return this.FUNCTIONMESSAGES;
    }

    /**
     * @return Accessor for the first list of suspended stacks.
     */
    public List<SuspendedStack> getSuspendedStacks1() {
        return this.SUSPENDEDSTACKS1;
    }

    /**
     * @return Accessor for the second list of suspended stacks.
     */
    public List<SuspendedStack> getSuspendedStacks2() {
        return this.SUSPENDEDSTACKS2;
    }

    /**
     * @return Accessor for the queued unbinds list.
     */
    public List<QueuedUnbind> getUnbinds() {
        return this.UNBINDS;
    }

    /**
     * Counts the number of unattached instances, the
     * <code>ScriptInstance</code> objects whose refID is 0.
     *
     * @return The number of unattached instances.
     */
    public int countUnattachedInstances() {
        return (int) this.INSTANCES.values().stream().filter(v -> v.isUnattached()).count();
    }

    /**
     * Counts the number of <code>ScriptInstance</code>, <code>Reference</code>,
     * and <code>ActiveScript</code> objects whose script is null.
     *
     * @return The number of undefined elements / undefined threads.
     */
    public int[] countUndefinedElements() {
        int count = 0;
        int threads = 0;

        count += this.getScripts().values().stream().filter(v -> v.isUndefined()).count();
        count += this.getInstances().values().parallelStream().filter(v -> v.isUndefined()).count();
        count += this.getReferences().values().stream().filter(v -> v.isUndefined()).count();
        count += this.getStructDefs().values().stream().filter(v -> v.isUndefined()).count();
        count += this.getStructs().values().stream().filter(v -> v.isUndefined()).count();
        threads += this.getActiveScripts().values().stream().filter(v -> v.isUndefined() && !v.isTerminated()).count();
        return new int[]{count, threads};
    }

    /**
     * Removes all <code>ScriptInstance</code> objects whose refID is 0.
     *
     * @return The number of instances removed.
     */
    public int cleanUnattachedInstances() {
        final Set<ScriptInstance> UNATTACHED = this.getInstances().values()
                .stream()
                .filter(v -> v.isUnattached())
                .collect(Collectors.toSet());

        return this.removeElements(UNATTACHED);
    }

    /**
     * Removes all <code>ScriptInstance</code> objects whose script is null.
     * Also checks <code>ActiveScript</code>, <code>FunctionMessage</code>, and
     * <code>SuspendedStack</code>.
     *
     * @return The number of instances removed.
     */
    public int[] cleanUndefinedElements() {
        final int[] COUNTS = this.countUndefinedElements();

        int count = 0;

        count += this.removeElements(this.getScripts().values().stream().filter(v -> v.isUndefined()).collect(Collectors.toSet()));
        count += this.removeElements(this.getStructDefs().values().stream().filter(v -> v.isUndefined()).collect(Collectors.toSet()));
        count += this.removeElements(this.getInstances().values().stream().filter(v -> v.isUndefined()).collect(Collectors.toSet()));
        count += this.removeElements(this.getReferences().values().stream().filter(v -> v.isUndefined()).collect(Collectors.toSet()));
        count += this.removeElements(this.getStructs().values().stream().filter(v -> v.isUndefined()).collect(Collectors.toSet()));

        this.getActiveScripts().values().stream().filter(v -> v.isUndefined() && !v.isTerminated()).forEach(v -> v.zero());

        assert COUNTS[0] <= count;
        COUNTS[0] = count;
        return COUNTS;
    }

    /**
     * Removes a <code>PapyrusElement</code> collection.
     *
     * @param elements The elements to remove.
     * @return The number of elements removed.
     *
     */
    public int removeElements(java.util.Set<? extends PapyrusElement> elements) {
        assert null != elements;
        assert !elements.contains(null);
        final java.util.LinkedList<PapyrusElement> ELEMENTS = new java.util.LinkedList<>(elements);

        int count = 0;

        while (!ELEMENTS.isEmpty()) {
            final PapyrusElement ELEMENT = ELEMENTS.pop();

            if (ELEMENT instanceof Script) {
                final Script DEF = (Script) ELEMENT;
                if (this.getScripts().containsKey(DEF.getName())) {
                    count++;
                    this.getScripts().remove(DEF.getName());
                    ELEMENTS.addAll(this.getInstances().values().parallelStream()
                            .filter(v -> v.getDefinition() == DEF)
                            .collect(Collectors.toSet()));
                }

            } else if (ELEMENT instanceof StructDef) {
                final StructDef DEF = (StructDef) ELEMENT;
                if (this.getStructDefs().containsKey(DEF.getName())) {
                    count++;
                    this.getStructDefs().remove(DEF.getName());
                    ELEMENTS.addAll(this.getStructs().values().parallelStream()
                            .filter(v -> v.getStructDef() == DEF)
                            .collect(Collectors.toSet()));
                }

            } else if (ELEMENT instanceof ScriptInstance) {
                final ScriptInstance INSTANCE = (ScriptInstance) ELEMENT;
                if (this.getInstances().containsKey(INSTANCE.getID())) {
                    this.getInstances().remove(INSTANCE.getID());
                    count++;
                }

            } else if (ELEMENT instanceof Struct) {
                final Struct STRUCT = (Struct) ELEMENT;
                if (this.getStructs().containsKey(STRUCT.getID())) {
                    this.getStructs().remove(STRUCT.getID());
                    count++;
                }

            } else if (ELEMENT instanceof Reference) {
                final Reference REF = (Reference) ELEMENT;
                if (this.getReferences().containsKey(REF.getID())) {
                    this.getReferences().remove(REF.getID());
                    count++;
                }

            } else if (ELEMENT instanceof ArrayInfo) {
                final ArrayInfo ARRAY = (ArrayInfo) ELEMENT;
                if (this.getArrays().containsKey(ARRAY.getID())) {
                    this.getArrays().remove(ARRAY.getID());
                    count++;
                }

            } else if (ELEMENT instanceof ActiveScript) {
                final ActiveScript ACTIVE = (ActiveScript) ELEMENT;
                if (this.getActiveScripts().containsKey(ACTIVE.getID())) {
                    this.getActiveScripts().remove(ACTIVE.getID());
                    count++;
                }

            } else if (ELEMENT instanceof SuspendedStack) {
                final SuspendedStack STACK = (SuspendedStack) ELEMENT;
                if (this.getSuspendedStacks1().contains(STACK)) {
                    this.getSuspendedStacks1().remove(STACK);
                    count++;
                } else if (this.getSuspendedStacks2().contains(STACK)) {
                    this.getSuspendedStacks2().remove(STACK);
                    count++;
                }

            } else {
                System.err.println("Papyrus.removeElements: can't delete this element: " + ELEMENT);
            }

        }

        return count;
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        return "Papyrus-" + super.toString();
    }

    /**
     * @see PapyrusElement#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
        this.getInstances().values().forEach(v -> v.addNames(analysis));
        this.getStructs().values().forEach(v -> v.addNames(analysis));
        this.getReferences().values().forEach(v -> v.addNames(analysis));
        this.getActiveScripts().values().forEach(v -> v.addNames(analysis));
        this.getArrays().values().forEach(v -> v.addNames(analysis));
        this.getSuspendedStacks1().forEach(v -> v.addNames(analysis));
        this.getSuspendedStacks2().forEach(v -> v.addNames(analysis));
        this.getFunctionMessages().forEach(v -> v.addNames(analysis));
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        this.getArrays().values().forEach(array -> array.resolveRefs(ess, null));
        this.getScripts().values().forEach(script -> script.resolveRefs(ess, null));
        this.getInstances().values().forEach(instance -> instance.resolveRefs(ess, null));
        this.getStructDefs().values().forEach(def -> def.resolveRefs(ess, null));
        this.getStructs().values().forEach(struct -> struct.resolveRefs(ess, null));
        this.getReferences().values().forEach(ref -> ref.resolveRefs(ess, null));
        this.getActiveScripts().values().forEach(script -> script.resolveRefs(ess, null));
        this.getSuspendedStacks1().forEach(stack -> stack.resolveRefs(ess, null));
        this.getSuspendedStacks2().forEach(stack -> stack.resolveRefs(ess, null));
        this.getFunctionMessages().forEach(msg -> msg.resolveRefs(ess, null));
    }

    /**
     * Does a very general search for an ID.
     *
     * @param id The ID to search for.
     * @return Any match of any kind.
     */
    public PapyrusElement broadSpectrumMatch(EID id) {
        if (this.getInstances().containsKey(id)) {
            return this.getInstances().get(id);
        }
        if (this.getReferences().containsKey(id)) {
            return this.getReferences().get(id);
        }
        if (this.getArrays().containsKey(id)) {
            return this.getArrays().get(id);
        }
        if (this.getActiveScripts().containsKey(id)) {
            return this.getActiveScripts().get(id);
        }
        if (this.getStructs().containsKey(id)) {
            return this.getStructs().get(id);
        }

        Optional<FunctionMessage> msg = this.getFunctionMessages().stream().filter(v -> v.getID().equals(id)).findAny();
        if (msg.isPresent()) {
            return msg.get();
        }

        Optional<SuspendedStack> susp1 = this.getSuspendedStacks1().stream().filter(v -> v.getID().equals(id)).findAny();
        if (susp1.isPresent()) {
            return susp1.get();
        }

        Optional<SuspendedStack> susp2 = this.getSuspendedStacks2().stream().filter(v -> v.getID().equals(id)).findAny();
        if (susp2.isPresent()) {
            return susp2.get();
        }
        Optional<QueuedUnbind> qu = this.getUnbinds().stream().filter(v -> v.getID().equals(id)).findAny();
        if (qu.isPresent()) {
            return qu.get();
        }

        return null;
    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 83 * hash + Arrays.hashCode(this.ORIGINAL_DATA);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final Papyrus other = (Papyrus) obj;

        for (int i = 0; i < this.ORIGINAL_DATA.length; i++) {
            if (this.ORIGINAL_DATA[i] != other.ORIGINAL_DATA[i]) {
                return false;
            }
        }

        return Arrays.equals(this.ORIGINAL_DATA, other.ORIGINAL_DATA);
    }

    final private short HEADER;
    final private int PAPYRUS_RUNTIME;
    final private short SAVE_FILE_VERSION;
    final private int UNK1;
    final private int UNK2;

    final private StringTable STRINGS;
    final private ScriptMap SCRIPTS;
    final private StructDefMap STRUCTDEFS;
    final private InstanceMap INSTANCES;
    final private ReferenceMap REFERENCES;
    final private StructMap STRUCTS;
    final private ArrayMap ARRAYS;
    final private ActiveScriptMap ACTIVESCRIPTS;
    final private List<FunctionMessage> FUNCTIONMESSAGES;
    final private List<SuspendedStack> SUSPENDEDSTACKS1;
    final private List<SuspendedStack> SUSPENDEDSTACKS2;
    final private List<EID> UNKS;
    final private List<QueuedUnbind> UNBINDS;

    final private byte[] OTHERDATA;
    final private byte[] ORIGINAL_DATA;

}
