/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.Objects;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.ess.ESS;
import restringer.ess.Element;

/**
 * Describes a variable in a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/06/21
 */
final public class Parameter implements PapyrusElement {

    /**
     * Creates a term, a label for doing substitutions.
     *
     * @param value
     * @return
     */
    static public Parameter createTerm(String value) {
        return new Parameter(value);
    }

    /**
     * Creates a new <code>Parameter</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param strings The string table.
     * @throws IOException
     */
    public Parameter(LittleEndianInput input, StringTable strings) throws IOException {
        assert null != input;
        assert null != strings;
        this.TYPE = Type.read(input);
        this.TERM = null;

        switch (this.TYPE) {
            case NULL:
                this.DATA = 0;
                this.STR_VALUE = null;
                this.FLT_VALUE = 0.0f;
                break;
            case IDENTIFIER:
            case STRING:
                this.DATA = 0;
                this.STR_VALUE = strings.read(input);
                this.FLT_VALUE = 0.0f;
                break;
            case UNKNOWN8:
                this.DATA = 0;
                this.STR_VALUE = strings.read(input);
                this.FLT_VALUE = 0.0f;
                break;
            case INTEGER:
                this.DATA = input.readInt();
                this.STR_VALUE = null;
                this.FLT_VALUE = 0.0f;
                break;
            case FLOAT:
                this.DATA = input.readInt();
                this.STR_VALUE = null;
                this.FLT_VALUE = Float.intBitsToFloat(this.DATA);
                break;
            case BOOLEAN:
                this.DATA = input.readByte();
                this.STR_VALUE = null;
                this.FLT_VALUE = 0.0f;
                break;
            case TERM:
                throw new IllegalStateException("Terms cannot be read.");
            default:
                throw new IOException("Illegal Parameter type: " + this.TYPE);
        }

    }

    /**
     * Creates a term with the specified "value".
     *
     * @param value
     */
    private Parameter(String value) {
        this.TYPE = Type.TERM;
        this.DATA = 0;
        this.STR_VALUE = null;
        this.FLT_VALUE = 0.0f;
        this.TERM = value;
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        assert null != output;
        this.TYPE.write(output);

        switch (this.TYPE) {
            case NULL:
                break;
            case IDENTIFIER:
            case STRING:
                this.STR_VALUE.write(output);
                break;
            case UNKNOWN8:
                this.STR_VALUE.write(output);
                break;
            case INTEGER:
            case FLOAT:
                output.writeInt(this.DATA);
                break;
            case BOOLEAN:
                output.writeByte(this.DATA);
                break;
            case TERM:
                throw new IllegalStateException("Terms cannot be written.");
            default:
                throw new IllegalStateException();
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = this.TYPE.calculateSize();

        switch (this.TYPE) {
            case NULL:
                break;
            case IDENTIFIER:
            case STRING:
                sum += this.STR_VALUE.calculateSize();
                break;
            case INTEGER:
            case FLOAT:
            case UNKNOWN8:
                sum += 4;
                break;
            case BOOLEAN:
                sum += 1;
                break;
            case TERM:
                throw new IllegalStateException("Terms cannot be written.");
            default:
                throw new IllegalStateException();
        }

        return sum;
    }

    /**
     * @return The ID of the papyrus element.
     */
    public Type getType() {
        return this.TYPE;
    }

    /**
     * @return The variable data as a RefID.
     */
    public int getRefIDValue() {
        assert this.TYPE == Type.IDENTIFIER;
        return this.DATA;
    }

    /**
     * @return The variable data as a String.
     */
    public TString getStringValue() {
        assert this.TYPE == Type.STRING;
        return this.STR_VALUE;
    }

    /**
     * @return The variable data as an int.
     */
    public int getIntValue() {
        assert this.TYPE == Type.INTEGER;
        return this.DATA;
    }

    /**
     * @return The variable data as a String.
     */
    public float getFloatValue() {
        assert this.TYPE == Type.FLOAT;
        return this.FLT_VALUE;
    }

    /**
     * @return The variable data as a boolean.
     */
    public boolean getBooleanValue() {
        assert this.TYPE == Type.BOOLEAN;
        return this.DATA != 0;
    }

    /**
     * @return A flag indicating if the parameter is an identifier to a temp
     * variable.
     */
    public boolean isTemp() {
        return this.TYPE == Type.IDENTIFIER
                && TEMP_PATTERN.test(this.STR_VALUE.toString())
                && !AUTOVAR_PATTERN.test(this.STR_VALUE.toString())
                && !NONE_PATTERN.test(this.STR_VALUE.toString());
    }

    /**
     * @return A flag indicating if the parameter is an Autovariable.
     */
    public boolean isAutovar() {
        return this.TYPE == Type.IDENTIFIER
                && AUTOVAR_PATTERN.test(this.STR_VALUE.toString());
    }

    /**
     * @return A flag indicating if the parameter is an None variable.
     */
    public boolean isNonevar() {
        return this.TYPE == Type.IDENTIFIER
                && NONE_PATTERN.test(this.STR_VALUE.toString());
    }

    /**
     * @see PapyrusElement#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        if (this.TYPE == Type.STRING) {
            this.STR_VALUE.addRefHolder(owner);
        }
    }

    /**
     * @return Short string representation.
     */
    public String toValueString() {
        switch (this.TYPE) {
            case IDENTIFIER:
                return this.STR_VALUE.toString();
            case STRING:
                return this.STR_VALUE.toString().replace("\n", "\\n");
            case UNKNOWN8:
                return this.STR_VALUE.toString().replace("\n", "\\n");
            case INTEGER:
                return Integer.toString(this.DATA);
            case FLOAT:
                return Float.toString(this.FLT_VALUE);
            case BOOLEAN:
                return Boolean.toString(this.DATA != 0);
            case TERM:
                return this.TERM;
            case VARIANT:
                return "VARIANT";
            case STRUCT:
                return "STRUCT";
            default:
                return "INVALID";
        }
    }

    /**
     * An appropriately parenthesized string form of the parameter.
     *
     * @return
     */
    public String paren() {
        if (this.TYPE == Type.TERM) {
            return "(" + this.toValueString() + ")";
        } else {
            return this.toValueString();
        }
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        return this.TYPE + ":" + this.toValueString();
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 41 * hash + Objects.hashCode(this.TYPE);
        hash = 41 * hash + this.DATA;
        hash = 41 * hash + Objects.hashCode(this.STR_VALUE);
        hash = 41 * hash + Objects.hashCode(this.TERM);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final Parameter other = (Parameter) obj;
        return this.TYPE == other.TYPE
                && this.DATA == other.DATA
                && Objects.equals(this.TERM, other.TERM);
    }

    final private Type TYPE;
    final private int DATA;
    final private float FLT_VALUE;
    final private TString STR_VALUE;
    final private String TERM;
    static final Predicate<String> TEMP_PATTERN = Pattern.compile("^::.+$", Pattern.CASE_INSENSITIVE).asPredicate();
    static final Predicate<String> NONE_PATTERN = Pattern.compile("^::NoneVar$", Pattern.CASE_INSENSITIVE).asPredicate();
    static final Predicate<String> AUTOVAR_PATTERN = Pattern.compile("^::(.+)_var$", Pattern.CASE_INSENSITIVE).asPredicate();

    static public enum Type implements PapyrusElement {
        NULL,
        IDENTIFIER,
        STRING,
        INTEGER,
        FLOAT,
        BOOLEAN,
        VARIANT,
        STRUCT,
        UNKNOWN8,
        TERM;

        static public Type read(LittleEndianInput input) throws IOException {
            Objects.requireNonNull(input);
            int val = input.readUnsignedByte();
            if (val < 0 || val >= VALUES.length) {
                throw new IOException("Invalid type: " + val);
            }
            return Type.values()[val];
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeByte(this.ordinal());
        }

        @Override
        public int calculateSize() {
            return 1;
        }

        @Override
        public void addNames(restringer.Analysis analysis) {
        }

        @Override
        public void resolveRefs(ESS ess, Element owner) {
        }

        static final private Type[] VALUES = Type.values();
    }

}
