/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.Objects;
import java.util.SortedSet;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInput;
import restringer.Profile;
import restringer.ess.AnalyzableElement;
import restringer.ess.ESS;
import restringer.ess.Element;
import restringer.ess.Linkable;

/**
 * Describes a function parameter in a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/07/01
 */
final public class QueuedUnbind implements PapyrusElement, AnalyzableElement, Linkable, HasID {

    /**
     * Creates a new <code>MemberData</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param instances The list of script instances.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public QueuedUnbind(LittleEndianInput input, InstanceMap instances, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(instances);
        Objects.requireNonNull(ctx);

        if (ctx.GAME.isID64()) {
            this.ID = EID.read8byte(input);
        } else {
            this.ID = EID.read4byte(input);
        }

        this.UNKNOWN = input.readInt();
        this.INSTANCE = instances.get(this.ID);
        assert null != this.INSTANCE;
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        assert null != output;
        output.writeESSElement(this.ID);
        output.writeInt(this.UNKNOWN);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        return 4 + this.ID.calculateSize();
    }

    /**
     * @return The ID of the papyrus element.
     */
    @Override
    public EID getID() {
        return this.ID;
    }

    /**
     * @return The unknown field.
     */
    public int getUnknown() {
        return this.UNKNOWN;
    }

    /**
     * @return The corresponding <code>ScriptInstance</code>.
     */
    public ScriptInstance getScriptInstance() {
        return this.INSTANCE;
    }

    /**
     * @see PapyrusElement#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {

    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(restringer.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();
        if (null != this.INSTANCE && null != this.INSTANCE.getScript()) {
            BUILDER.append(String.format("<html><h3>QUEUED UNBIND of %s</h3>", this.INSTANCE.getScript().toHTML()));
        } else if (null != this.INSTANCE) {
            BUILDER.append(String.format("<html><h3>QUEUED UNBIND of %s</h3>", this.INSTANCE.getScriptName()));
        } else {
            BUILDER.append(String.format("<html><h3>QUEUED UNBIND of %s</h3>", this.ID));
        }

        if (null == this.INSTANCE) {
            BUILDER.append(String.format("<p>Instance: %s</p>", this.ID));
        } else {
            BUILDER.append(String.format("<p>Instance: %s</p>", this.INSTANCE.toHTML()));
        }
        
        BUILDER.append(String.format("<p>Unknown: %s</p>", EID.pad8(this.UNKNOWN)));

        Element UNK = save.broadSpectrumMatch(this.UNKNOWN);
        if (null != UNK) {
            BUILDER.append("<p>Potential match for UNKNOWN found using general search:<br/>");
            if (UNK instanceof Linkable) {
                BUILDER.append(((Linkable) UNK).toHTML());
            } else {
                BUILDER.append(UNK.toString());
            }
            BUILDER.append("</p>");
        }

        if (null != analysis) {
            SortedSet<String> providers = analysis.SCRIPT_ORIGINS.get(this.INSTANCE.getScriptName());
            if (null != providers) {
                String probablyProvider = providers.last();
                BUILDER.append(String.format("The queued unbinding probably came from mod \"%s\".\n\n", probablyProvider));

                if (providers.size() > 1) {
                    BUILDER.append("<p>Full list of providers:</p><ul>");
                    providers.forEach(mod -> BUILDER.append(String.format("<li>%s", mod)));
                    BUILDER.append("</ul>");
                }
            }
        }

        BUILDER.append("</html>");
        return BUILDER.toString();
    }

    /**
     * @see AnalyzableElement#matches(restringer.Profile.Analysis,
     * restringer.Mod)
     * @param analysis
     * @param mod
     * @return
     */
    @Override
    public boolean matches(Profile.Analysis analysis, String mod) {
        Objects.requireNonNull(analysis);
        Objects.requireNonNull(mod);
        return this.INSTANCE.matches(analysis, mod);
    }

    /**
     * @return A flag indicating if the <code>ScriptInstance</code> is
     * undefined.
     *
     */
    public boolean isUndefined() {
        return this.INSTANCE.isUndefined();
    }

    /**
     * @see restringer.ess.Linkable#toHTML()
     * @return
     */
    @Override
    public String toHTML() {
        return String.format("<a href=\"%s\">%s</a>", this.ID, this.toString());
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        if (null == this.INSTANCE) {
            return this.ID + ": " + EID.pad8(this.UNKNOWN) + " (MISSING INSTANCE)";
        } else {
            return this.ID + ": " + EID.pad8(this.UNKNOWN) + " (" + INSTANCE.getScriptName() + ")";
        }
    }

    final private EID ID;
    final private int UNKNOWN;
    final private ScriptInstance INSTANCE;

}
