/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import restringer.ess.AnalyzableElement;
import java.io.IOException;
import java.util.List;
import java.util.SortedSet;
import java.util.stream.Collectors;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.ess.Element;
import restringer.ess.ESS;
import restringer.ess.Linkable;

/**
 * Describes a reference in a Skyrim savegame.
 *
 * @author Mark Fairchild
 */
final public class Reference extends GameElement {

    /**
     * Creates a new <code>Reference</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param scripts The list of scripts.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public Reference(LittleEndianInput input, ScriptMap scripts, PapyrusContext ctx) throws IOException {
        super(input, scripts, ctx);
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        super.write(output);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        return super.calculateSize();
    }

    /**
     * @return The <code>ReferenceData</code> for the instance.
     */
    public ReferenceData getData() {
        return this.data;
    }

    /**
     * Sets the data field.
     *
     * @param newData The new value for the data field.
     */
    public void setData(ReferenceData newData) {
        this.data = newData;
    }

    /**
     * @return The name of the corresponding <code>Script</code>.
     */
    public TString getScriptName() {
        return super.getDefinitionName();
    }

    /**
     * @return The corresponding <code>Script</code>.
     */
    public Script getScript() {
        assert super.getDefinition() instanceof Script;
        return (Script) super.getDefinition();
    }

   /**
     * @see restringer.ess.Linkable#toHTML() 
     * @return 
     */
    @Override
    public String toHTML() {
        return String.format("<a href=\"reference://%s\">%s</a>", this.getID(), this.toString());
    }
    
    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        final StringBuilder BUF = new StringBuilder();

        if (this.isUndefined()) {
            BUF.append("#").append(this.getScriptName()).append("# ");
        } else {
            BUF.append(this.getScriptName()).append(": ");
        }

        BUF.append("(").append(this.getID()).append(")");
        return BUF.toString();
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(restringer.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();
        if (null != this.getScript()) {
            BUILDER.append(String.format("<html><h3>REFERENCE of %s</h3>", this.getScript().toHTML()));
        } else {
            BUILDER.append(String.format("<html><h3>REFERENCE of %s</h3>", this.getScriptName()));
        }

        if (null != analysis) {
            SortedSet<String> providers = analysis.SCRIPT_ORIGINS.get(this.getScriptName());
            if (null != providers) {
                String probablyProvider = providers.last();
                BUILDER.append(String.format("<p>This script probably came from \"%s\".</p>", probablyProvider));

                if (providers.size() > 1) {
                    BUILDER.append("<p>Full list of providers:</p><ul>");
                    providers.forEach(mod -> BUILDER.append(String.format("<li>%s", mod)));
                    BUILDER.append("</ul>");
                }
            }
        }

        BUILDER.append(String.format("<p>ID: %s</p>", this.getID()));
        BUILDER.append(String.format("<p>Type2: %s</p>", this.data.getType()));
        BUILDER.append(String.format("<p>Unknown1: %08x</p>", this.data.getUnknown1()));
        BUILDER.append(String.format("<p>Unknown2: %08x</p>", this.data.getUnknown2()));

        Element UNKNOWN1 = save.broadSpectrumMatch(this.getData().getUnknown1());
        if (null != UNKNOWN1) {
            BUILDER.append("<p>Potential match for unknown1 found using general search:<br/>");
            if (UNKNOWN1 instanceof Linkable) {
                BUILDER.append(((Linkable)UNKNOWN1).toHTML());
            } else {
                BUILDER.append(UNKNOWN1.toString());
            }
            BUILDER.append("</p>");
        }
        
        Element UNKNOWN2 = save.broadSpectrumMatch(this.getData().getUnknown2());
        if (null != UNKNOWN2) {
            BUILDER.append("<p>Potential match for unknown2 found using general search:<br/>");
            if (UNKNOWN2 instanceof Linkable) {
                BUILDER.append(((Linkable)UNKNOWN2).toHTML());
            } else {
                BUILDER.append(UNKNOWN2.toString());
            }
            BUILDER.append("</p>");
        }
        
        final List<ScriptInstance> REFERRING_INSTANCES
                = save.getPapyrus().getInstances().values().stream()
                .filter(v -> v.getData().getMembers().stream().anyMatch(m -> m.hasRef(this.getID())))
                .collect(Collectors.toList());

        final List<Reference> REFERRING_REFS
                = save.getPapyrus().getReferences().values().stream()
                .filter(v -> v.getData().getMembers().stream().anyMatch(m -> m.hasRef(this.getID())))
                .collect(Collectors.toList());

        final List<ArrayInfo> REFERRING_ARRAYS
                = save.getPapyrus().getArrays().values().stream()
                .filter(v -> v.getData().getMembers().stream().anyMatch(m -> m.hasRef(this.getID())))
                .collect(Collectors.toList());

        final List<ActiveScript> REFERRING_THREADS
                = save.getPapyrus().getActiveScripts().values().stream()
                .filter(t -> t.getData().getStackFrames().stream().anyMatch(s -> s.getOwner().hasRef(this.getID())))
                .collect(Collectors.toList());

        final List<Struct> REFERRING_STRUCTS
                = save.getPapyrus().getStructs().values().stream()
                .filter(v -> v.getData().getMembers().stream().anyMatch(m -> m.hasRef(this.getID())))
                .collect(Collectors.toList());


        BUILDER.append(String.format("<p>There are %d threads with stackframes attached to this reference.</p>", REFERRING_THREADS.size()));
        if (0 < REFERRING_THREADS.size() && REFERRING_THREADS.size() < 50) {
            BUILDER.append("<ul>");
            REFERRING_THREADS.forEach(i -> BUILDER.append(String.format("<li>%s</a>", i.toHTML())));
            BUILDER.append("</ul>");
        }

        BUILDER.append(String.format("<p>There are %d instances holding refs to this reference.</p>", REFERRING_INSTANCES.size()));
        if (0 < REFERRING_INSTANCES.size() && REFERRING_INSTANCES.size() < 50) {
            BUILDER.append("<ul>");
            REFERRING_INSTANCES.forEach(i -> BUILDER.append(String.format("<li>%s</a>", i.toHTML())));
            BUILDER.append("</ul>");
        }

        BUILDER.append(String.format("<p>There are %d references holding refs to this reference.</p>", REFERRING_REFS.size()));
        if (0 < REFERRING_REFS.size() && REFERRING_REFS.size() < 50) {
            BUILDER.append("<ul>");
            REFERRING_REFS.forEach(i -> BUILDER.append(String.format("<li>%s</a>", i.toHTML())));
            BUILDER.append("</ul>");
        }

        BUILDER.append(String.format("<p>There are %d arrays holding refs to this reference.</p>", REFERRING_ARRAYS.size()));
        if (0 < REFERRING_ARRAYS.size() && REFERRING_ARRAYS.size() < 50) {
            BUILDER.append("<ul>");
            REFERRING_ARRAYS.forEach(i -> BUILDER.append(String.format("<li>%s</a>", i.toHTML())));
            BUILDER.append("</ul>");
        }

        BUILDER.append(String.format("<p>There are %d struct holding refs to this reference.</p>", REFERRING_STRUCTS.size()));
        if (0 < REFERRING_STRUCTS.size() && REFERRING_STRUCTS.size() < 50) {
            BUILDER.append("<ul>");
            REFERRING_STRUCTS.forEach(i -> BUILDER.append(String.format("<li>%s</a>", i.toHTML())));
            BUILDER.append("</ul>");
        }

        BUILDER.append("</html>");
        return BUILDER.toString();
    }

    /**
     * @see PapyrusElement#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
        assert null != this.data;
        this.data.addNames(analysis);
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        super.resolveRefs(ess, owner);
        this.data.resolveRefs(ess, this);
    }

    private ReferenceData data;

}
