/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInputStream;
import restringer.ess.ESS;
import restringer.ess.Element;

/**
 * Describes a script instance's data in a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/06/21
 */
final public class ScriptData implements PapyrusElement, HasID {

    /**
     * Creates a new <code>ScriptData</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public ScriptData(LittleEndianInputStream input, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(ctx);

        if (ctx.GAME.isID64()) {
            this.ID = EID.read8byte(input);
        } else {
            this.ID = EID.read4byte(input);
        }

        this.FLAG = input.readByte();
        this.TYPE = ctx.STRINGS.read(input);
        this.UNKNOWN1 = input.readInt();
        this.UNKNOWN2 = ((this.FLAG & 0x04) != 0 ? input.readInt() : 0);

        int memberCount = input.readInt();
        this.MEMBERS.ensureCapacity(memberCount);

        int i = 0;

        try {
            for (i = 0; i < memberCount; i++) {
                Variable var = Variable.read(input, ctx);
                this.MEMBERS.add(var);
            }
        } catch (IOException ex) {
            throw new IOException(String.format("Error; read %d/%d members.", i, memberCount), ex);
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);

        output.writeESSElement(this.ID);
        output.writeByte(this.FLAG);
        output.writeESSElement(this.TYPE);
        output.writeInt(this.UNKNOWN1);

        if ((this.FLAG & 0x04) != 0) {
            output.writeInt(this.UNKNOWN2);
        }

        output.writeInt(this.MEMBERS.size());

        int i = 0;
        for (Variable var : this.MEMBERS) {
            output.writeESSElement(var);
            i++;
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 9;
        sum += this.ID.calculateSize();
        sum += ((this.FLAG & 0x04) != 0 ? 4 : 0);
        sum += this.TYPE.calculateSize();
        sum += this.MEMBERS.stream().mapToInt(var -> var.calculateSize()).sum();
        return sum;
    }

    /**
     * @return The ID of the papyrus element.
     */
    @Override
    public EID getID() {
        return this.ID;
    }

    /**
     * @return The type of the script.
     */
    public TString getType() {
        return this.TYPE;
    }

    /**
     * @return The list of <code>Variable</code>.
     */
    public List<Variable> getMembers() {
        return this.MEMBERS;
    }

    /**
     * @see PapyrusElement#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
        this.MEMBERS.forEach(v -> v.addNames(analysis));
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        this.MEMBERS.forEach(var -> var.resolveRefs(ess, owner));
        this.TYPE.addRefHolder(owner);
    }

    /**
     * @see Object#toString()
     * @return
     */
    @Override
    public String toString() {
        final StringBuilder BUILDER = new StringBuilder();
        BUILDER.append("SCRIPTDATA\n");
        BUILDER.append(String.format("ID = %s\n", this.ID));
        BUILDER.append(String.format("flag= %d\n", this.FLAG));
        BUILDER.append(String.format("type = %s\n", this.TYPE));
        BUILDER.append(String.format("unknown1 = %d\n", this.UNKNOWN1));
        BUILDER.append(String.format("unknown2 = %d\n\n", this.UNKNOWN2));
        this.MEMBERS.forEach(var -> BUILDER.append(String.format("%s\n", var)));
        return BUILDER.toString();
    }

    final private EID ID;
    final private byte FLAG;
    final private TString TYPE;
    final private int UNKNOWN1;
    final private int UNKNOWN2;
    final private ArrayList<Variable> MEMBERS = new ArrayList<>();

}
