/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import restringer.ess.AnalyzableElement;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.SortedSet;
import java.util.stream.Collectors;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.ess.Element;
import restringer.ess.ESS;
import restringer.ess.Plugin;
import restringer.ess.RefID;

/**
 * Describes a script instance in a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/06/21
 */
final public class ScriptInstance extends GameElement {

    /**
     * Creates a new <code>ScriptInstances</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param scripts The list of scripts.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public ScriptInstance(LittleEndianInput input, ScriptMap scripts, PapyrusContext ctx) throws IOException {
        super(input, scripts, ctx);
        this.UNKNOWN2BITS = input.readShort();
        this.UNKNOWN = input.readShort();
        this.REFID = new RefID(input);
        this.UNKNOWN_BYTE = input.readByte();
        if (ctx.GAME.isFO4()) {
            //if (this.REFID.getValue() == 0x10000 || this.REFID.getValue() == 0x0) {
            if ((this.UNKNOWN2BITS & 0x3) == 0x3) {
                this.UNKNOWN_FO_BYTE = input.readByte();
            } else {
                this.UNKNOWN_FO_BYTE = null;
            }
        } else {
            this.UNKNOWN_FO_BYTE = null;
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        super.write(output);
        output.writeShort(this.UNKNOWN2BITS);
        output.writeShort(this.UNKNOWN);
        output.writeESSElement(this.REFID);
        output.writeByte(this.UNKNOWN_BYTE);
        if (null != this.UNKNOWN_FO_BYTE) {
            output.writeByte(this.UNKNOWN_FO_BYTE);
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 5;
        sum += super.calculateSize();
        sum += this.REFID.calculateSize();
        if (null != this.UNKNOWN_FO_BYTE) {
            sum++;
        }
        return sum;
    }

    /**
     * @return The RefID of the papyrus element.
     */
    public RefID getRefID() {
        return this.REFID;
    }

    /**
     * @return The unknown short field; if it's -1, the RefID field may not be
     * valid.
     */
    public short getUnknown() {
        return this.UNKNOWN;
    }

    /**
     * @return The mystery flag; equivalent to <code>this.getUnknown==-1</code>.
     */
    public boolean isMysteryFlag() {
        return this.UNKNOWN == -1;
    }

    /**
     * @return The <code>ScriptData</code> for the instance.
     */
    public ScriptData getData() {
        return this.data;
    }

    /**
     * Sets the data field.
     *
     * @param newData The new value for the data field.
     */
    public void setData(ScriptData newData) {
        this.data = newData;
    }

    /**
     * @return The name of the corresponding <code>Script</code>.
     */
    public TString getScriptName() {
        return super.getDefinitionName();
    }

    /**
     * @return The corresponding <code>Script</code>.
     */
    public Script getScript() {
        assert super.getDefinition() instanceof Script;
        return (Script) super.getDefinition();
    }

    /**
     * @return A list of the script's member variables and their descriptors. If
     * the script has not been resolved, null will be returned.
     */
    public List<Member> getMembers() {
        if (null == this.getScript()) {
            return null;
        }

        final List<MemberDesc> DESCS = this.getScript().getExtendedMembers();
        final List<Variable> VARS = this.getData().getMembers();
        assert DESCS.size() == VARS.size();

        final List<Member> MEMBERS = new ArrayList<>(DESCS.size());
        final Iterator<MemberDesc> I1 = DESCS.iterator();
        final Iterator<Variable> I2 = VARS.iterator();

        while (I1.hasNext() && I2.hasNext()) {
            MemberDesc desc = I1.next();
            Variable var = I2.next();
            MEMBERS.add(new Member(desc, var));
        }

        return MEMBERS;
    }

    /**
     * @see restringer.ess.Linkable#toHTML()
     * @return
     */
    @Override
    public String toHTML() {
        return String.format("<a href=\"instance://%s\">%s</a>", this.getID(), this.toString());
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        final StringBuilder BUF = new StringBuilder();

        if (this.isUndefined()) {
            BUF.append("#").append(this.getScriptName()).append("#: ");
        } else {
            BUF.append(this.getScriptName()).append(": ");
        }

        if (this.REFID.isNonexistentCreated()) {
            BUF.append(isMysteryFlag() ? "@" : "&");
        } else if (this.isMysteryFlag()) {
            BUF.append("*");
        }

        BUF.append("refid=").append(this.REFID).append(" (").append(this.getID()).append(")");
        return BUF.toString();
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(restringer.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();
        if (null != this.getScript()) {
            BUILDER.append(String.format("<html><h3>INSTANCE of %s</h3>", this.getScript().toHTML()));
        } else {
            BUILDER.append(String.format("<html><h3>INSTANCE of %s</h3>", this.getScriptName()));
        }

        final Plugin PLUGIN = this.REFID.getPlugin();
        if (PLUGIN != null) {
            BUILDER.append(String.format("<p>This instance is attached to an object from %s.</p>", PLUGIN.toHTML()));
        } else if (this.REFID.getType() == RefID.Type.CREATED) {
            BUILDER.append("<p>This instance was created in-game.</p>");
        }

        if (this.isUndefined()) {
            BUILDER.append("<p><em>WARNING: SCRIPT MISSING!</em><br/>Remove Undefined Instances\" will delete this.</p>");
        }

        if (this.isUnattached()) {
            BUILDER.append("<p><em>WARNING: OBJECT MISSING!</em><br/>Selecting \"Remove Unattached Instances\" will delete this.</p>");
        }

        if (this.REFID.isNonexistentCreated()) {
            BUILDER.append("<p><em>REFID POINTS TO NONEXISTENT CREATED FORM.</em><br/>Remove non-existent form instances\" will delete this. However, some mods create these instances deliberately. </p>");
        }

        if (null != analysis) {
            final SortedSet<String> PROVIDERS = analysis.SCRIPT_ORIGINS.get(this.getScriptName());
            if (null != PROVIDERS) {
                String probableProvider = PROVIDERS.last();
                BUILDER.append(String.format("<p>The script probably came from mod \"%s\".</p>", probableProvider));

                if (PROVIDERS.size() > 1) {
                    BUILDER.append("<p>Full list of providers:</p><ul>");
                    PROVIDERS.forEach(mod -> BUILDER.append(String.format("<li>%s", mod)));
                    BUILDER.append("</ul>");
                }
            }
        }

        BUILDER.append(String.format("<p>ID: %s</p>", this.getID()));
        BUILDER.append(String.format("<p>Type: %s</p>", this.data.getType()));

        boolean mysteryFlag = this.UNKNOWN == -1;

        if (save.getChangeForms().containsKey(this.REFID)) {
            BUILDER.append(String.format("<p>RefID%s: %s</p>", (mysteryFlag ? "@" : ""), this.REFID.toHTML()));
        } else {
            BUILDER.append(String.format("<p>RefID%s: %s</p>", (mysteryFlag ? "@" : ""), this.REFID.toString()));
        }

        BUILDER.append(String.format("<p>Unknown2bits: %01X</p>", this.UNKNOWN2BITS));
        BUILDER.append(String.format("<p>UnknownShort: %04X</p>", this.UNKNOWN));
        BUILDER.append(String.format("<p>UnknownByte: %02x</p>", this.UNKNOWN_BYTE));

        final List<ScriptInstance> REFERRING_INSTANCES
                = save.getPapyrus().getInstances().values().stream()
                .filter(v -> v.getData().getMembers().stream().anyMatch(m -> m.hasRef(this.getID())))
                .collect(Collectors.toList());

        final List<Reference> REFERRING_REFS
                = save.getPapyrus().getReferences().values().stream()
                .filter(v -> v.getData().getMembers().stream().anyMatch(m -> m.hasRef(this.getID())))
                .collect(Collectors.toList());

        final List<ArrayInfo> REFERRING_ARRAYS
                = save.getPapyrus().getArrays().values().stream()
                .filter(v -> v.getData().getMembers().stream().anyMatch(m -> m.hasRef(this.getID())))
                .collect(Collectors.toList());

        final List<ActiveScript> REFERRING_THREADS
                = save.getPapyrus().getActiveScripts().values().stream()
                .filter(t -> t.getData().getStackFrames().stream().anyMatch(s -> s.getOwner().hasRef(this.getID())))
                .collect(Collectors.toList());

        final List<Struct> REFERRING_STRUCTS
                = save.getPapyrus().getStructs().values().stream()
                .filter(v -> v.getData().getMembers().stream().anyMatch(m -> m.hasRef(this.getID())))
                .collect(Collectors.toList());

        BUILDER.append(String.format("<p>There are %d threads with stackframes attached to this instance.</p>", REFERRING_THREADS.size()));
        if (0 < REFERRING_THREADS.size() && REFERRING_THREADS.size() < 50) {
            BUILDER.append("<ul>");
            REFERRING_THREADS.forEach(i -> BUILDER.append(String.format("<li>%s</a>", i.toHTML())));
            BUILDER.append("</ul>");
        }

        BUILDER.append(String.format("<p>There are %d instances holding refs to this instance.</p>", REFERRING_INSTANCES.size()));
        if (0 < REFERRING_INSTANCES.size() && REFERRING_INSTANCES.size() < 50) {
            BUILDER.append("<ul>");
            REFERRING_INSTANCES.forEach(i -> BUILDER.append(String.format("<li>%s</a>", i.toHTML())));
            BUILDER.append("</ul>");
        }

        BUILDER.append(String.format("<p>There are %d references holding refs to this instance.</p>", REFERRING_REFS.size()));
        if (0 < REFERRING_REFS.size() && REFERRING_REFS.size() < 50) {
            BUILDER.append("<ul>");
            REFERRING_REFS.forEach(i -> BUILDER.append(String.format("<li>%s</a>", i.toHTML())));
            BUILDER.append("</ul>");
        }

        BUILDER.append(String.format("<p>There are %d arrays holding refs to this instance.</p>", REFERRING_ARRAYS.size()));
        if (0 < REFERRING_ARRAYS.size() && REFERRING_ARRAYS.size() < 50) {
            BUILDER.append("<ul>");
            REFERRING_ARRAYS.forEach(i -> BUILDER.append(String.format("<li>%s</a>", i.toHTML())));
            BUILDER.append("</ul>");
        }

        BUILDER.append(String.format("<p>There are %d struct holding refs to this instance.</p>", REFERRING_STRUCTS.size()));
        if (0 < REFERRING_STRUCTS.size() && REFERRING_STRUCTS.size() < 50) {
            BUILDER.append("<ul>");
            REFERRING_STRUCTS.forEach(i -> BUILDER.append(String.format("<li>%s</a>", i.toHTML())));
            BUILDER.append("</ul>");
        }

        BUILDER.append("</html>");
        return BUILDER.toString();
    }

    /**
     * @see PapyrusElement#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
        assert null != this.data;
        this.data.addNames(analysis);
        this.REFID.addNames(analysis);
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        super.resolveRefs(ess, owner);
        this.data.resolveRefs(ess, this);
        this.REFID.resolveRefs(ess, this);
    }

    /**
     * @return A flag indicating if the <code>ScriptInstance</code> is
     * unattached.
     *
     */
    public boolean isUnattached() {
        return this.REFID.equals(RefID.NONE);
    }

    final private short UNKNOWN2BITS;
    final private short UNKNOWN;
    final private RefID REFID;
    final private byte UNKNOWN_BYTE;
    final private Byte UNKNOWN_FO_BYTE;
    private ScriptData data;

}
