/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Objects;
import java.util.Optional;
import restringer.ess.WString;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.ess.ESS;
import restringer.ess.Element;
import restringer.Game;

/**
 * An abstraction describing a string table.
 *
 * @author Mark Fairchild
 * @version 2016/06/19
 */
public class StringTable extends ArrayList<TString> implements PapyrusElement {

    /**
     * Creates a new <code>TString</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @return The new <code>TString</code>.
     * @throws IOException
     */
    public TString read(LittleEndianInput input) throws IOException {
        Objects.requireNonNull(input);

        int index;

        if (this.GAME.isStr32()) {
            // SkyrimSE uses 32bit string indices.            
            index = input.readInt();

        } else {
            index = input.readUnsignedShort();
            // SkyrimLegendary and Fallout4 use 16bit string indices.
            // Various corrections are possible though.

            if (index == 0xFFFF && !this.STBCORRECTION) {
                index = input.readInt();
            }
        }

        if (index < 0 || index >= this.size()) {
            throw new IOException(String.format("Invalid TString index: %d / %d", index, this.size()));
        }

        TString newString = this.get(index);
        return newString;
    }

    /**
     * Creates a new <code>StringTable</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param game The type of save.
     * @param applySTBCorrection A flag indicating that <code>StringTable</code>
     * should ATTEMPT to correct for the string table bug.
     * @throws IOException
     */
    public StringTable(LittleEndianInput input, Game game, boolean applySTBCorrection) throws IOException {
        this.GAME = Objects.requireNonNull(game);

        int strCount;

        if (this.GAME.isStr32()) {
            // SkyrimSE uses 32bit string indices.            
            strCount = input.readInt();
            STBCORRECTION = false;

        } else {
            // SkyrimLegendary and Fallout4 use 16bit string indices.
            // Various corrections are possible though.           
            strCount = input.readUnsignedShort();

            // Large string table version.
            if (strCount == 0xFFFF) {
                strCount = input.readInt();
            }

            // Fallback for catching the stringtable bug.
            if (strCount < 20000 && applySTBCorrection) {
                strCount |= 0x10000;
                STBCORRECTION = true;
            } else {
                STBCORRECTION = false;
            }
        }

        // Read the actual strings.
        try {
            this.ensureCapacity(strCount);
            for (int i = 0; i < strCount; i++) {
                try {
                    final WString WSTR = WString.read(input);
                    final TString TSTR = (this.GAME.isStr32()
                            ? new TString32(WSTR, i)
                            : new TString16(WSTR, i));
                    this.add(TSTR);
                } catch (IOException | RuntimeException | Error ex) {
                    throw new IOException("Error reading string #" + i, ex);
                }
            }
        } catch (IOException ex) {
            throw new IOException(String.format("Error; read %d/%d strings.", this.size(), strCount), ex);
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        if (this.STBCORRECTION) {
            throw new IOException("String-Table-Bug correction in effect. Cannot write save.");
        }

        if (this.GAME.isStr32()) {
            // SkyrimSE uses 32bit string indices.
            output.writeInt(this.size());

        } else {
            // SkyrimLegendary and Fallout4 use 16bit string indices.
            // Various corrections are possible though.           

            // Large string table version.
            if (this.size() > 0xFFF0) {
                output.writeShort(0xFFFF);
                output.writeInt(this.size());
            } else {
                output.writeShort(this.size());                
            }
        }

        // Write the actual strings.
        for (TString tstr : this) {
            try {
                tstr.writeFull(output);
            } catch (IOException ex) {
                throw new IOException("Error writing string #" + tstr.getINDEX(), ex);
            }
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 0;

        if (this.GAME.isStr32()) {
            sum += 4;
        } else if (this.size() > 0xFFF0) {
            sum += 6;
        } else {
            sum += 2;
        }

        sum += this.parallelStream().mapToInt(v -> v.calculateFullSize()).sum();
        return sum;
    }

    /**
     * @see PapyrusElement#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner
    ) {
    }

    /**
     * Checks if the <code>StringTable</code> contains a <code>TString</code>
     * that matches a specified string value.
     *
     * @param val The value to match against.
     * @return True if the <code>StringTable</code> contains a matching
     * <code>TString</code>, false otherwise.
     */
    public boolean containsMatching(String val) {
        return this.stream().anyMatch(v -> v.equals(val));
    }

    /**
     * Adds a new string to the <code>StringTable</code> and returns the
     * corresponding <code>TString</code>.
     *
     * @param val The value of the new string.
     * @return The new <code>TString</code>, or the existing one if the
     * <code>StringTable</code> already contained a match.
     */
    public TString addString(String val) {
        Optional<TString> match = this.stream().filter(v -> v.equals(val)).findFirst();
        if (match.isPresent()) {
            return match.get();
        }

        TString tstr = (this.GAME.isStr32()
                ? new TString32(val, this.size())
                : new TString16(val, this.size()));
        this.add(tstr);
        return tstr;
    }

    /**
     * A flag indicating that the string-table-bug correction is in effect. This
     * means that the table is NOT SAVABLE.
     *
     * @return
     */
    public boolean isSTBCorrection() {
        return this.STBCORRECTION;
    }

    /**
     * A flag indicating that the string-table-bug correction is in effect.
     */
    final private boolean STBCORRECTION;

    /**
     * Stores the game.
     */
    final private Game GAME;

    /**
     * TString implementation for 16 bit TStrings.
     */
    final private class TString16 extends TString {

        /**
         * Creates a new <code>TString16</code> from a <code>WString</code> and
         * an index.
         *
         * @param wstr The <code>WString</code>.
         * @param index The index of the <code>TString</code>.
         */
        private TString16(WString wstr, int index) {
            super(wstr, index);
        }

        /**
         * Creates a new <code>TString16</code> from a character sequence and an
         * index.
         *
         * @param cs The <code>CharSequence</code>.
         * @param index The index of the <code>TString</code>.
         */
        private TString16(CharSequence cs, int index) {
            super(cs, index);
        }

        /**
         * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
         * @param output The output stream.
         * @throws IOException
         */
        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            if (this.getINDEX() > 0xFFF0) {
                output.writeShort(0xFFFF);
                output.writeInt(this.getINDEX());
            } else {
                output.writeShort(this.getINDEX());
            }
        }

        /**
         * @see restringer.ess.Element#calculateSize()
         * @return The size of the <code>Element</code> in bytes.
         */
        @Override
        public int calculateSize() {
            return (this.getINDEX() > 0xFFF0 ? 6 : 2);
        }

    }

    /**
     * TString implementation for 32 bit TStrings.
     */
    final private class TString32 extends TString {

        /**
         * Creates a new <code>TString32</code> from a <code>WString</code> and
         * an index.
         *
         * @param wstr The <code>WString</code>.
         * @param index The index of the <code>TString</code>.
         */
        private TString32(WString wstr, int index) {
            super(wstr, index);
        }

        /**
         * Creates a new <code>TString32</code> from a character sequence and an
         * index.
         *
         * @param cs The <code>CharSequence</code>.
         * @param index The index of the <code>TString</code>.
         */
        private TString32(CharSequence cs, int index) {
            super(cs, index);
        }

        /**
         * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
         * @param output The output stream.
         * @throws IOException
         */
        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeInt(this.getINDEX());
        }

        /**
         * @see restringer.ess.Element#calculateSize()
         * @return The size of the <code>Element</code> in bytes.
         */
        @Override
        public int calculateSize() {
            return 4;
        }

    }
}
