/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.List;
import java.util.ArrayList;
import java.util.Objects;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.ess.ESS;
import restringer.ess.Element;
import restringer.ess.Flags;

/**
 * Describes struct data in a Skyrim savegame.
 *
 * @author Mark Fairchild
 */
final public class StructData implements PapyrusElement, HasID {

    /**
     * Creates a new <code>StructData</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param structs The Structs.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public StructData(LittleEndianInput input, StructMap structs, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(structs);
        Objects.requireNonNull(ctx);

        if (ctx.GAME.isID64()) {
            this.ID = EID.read8byte(input);
        } else {
            this.ID = EID.read4byte(input);
        }
        
        this.FLAG = Flags.readByteFlags(input);

        int memberCount = input.readInt();
        this.MEMBERS.ensureCapacity(memberCount);

        int i = 0;

        try {
            for (i = 0; i < memberCount; i++) {
                Variable var = Variable.read(input, ctx);
                this.MEMBERS.add(var);
            }
        } catch (IOException ex) {
            throw new IOException(String.format("Error; read %d/%d members.", i, memberCount), ex);
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);        
        output.writeESSElement(this.ID);
        output.writeESSElement(this.FLAG);
        output.writeInt(this.MEMBERS.size());

        int i = 0;
        for (Variable var : this.MEMBERS) {
            output.writeESSElement(var);
            i++;
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 4;
        sum += this.FLAG.calculateSize();
        sum += this.ID.calculateSize();
        sum += this.MEMBERS.stream().mapToInt(var -> var.calculateSize()).sum();
        return sum;
    }

    /**
     * @return The ID of the papyrus element.
     */
    @Override
    public EID getID() {
        return this.ID;
    }

    /**
     * @return The flag field.
     */
    public Flags.Byte getFlag() {
        return this.FLAG;
    }

    /**
     * @return The contents of the array.
     */
    public List<Variable> getMembers() {
        return this.MEMBERS;
    }

    /**
     * @see PapyrusElement#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
        this.MEMBERS.forEach(v -> v.addNames(analysis));
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        this.MEMBERS.forEach(var -> var.resolveRefs(ess, owner));
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        return this.ID.toString() + this.MEMBERS;
    }

    final private EID ID;
    final private Flags.Byte FLAG;
    final private ArrayList<Variable> MEMBERS = new ArrayList<>();

}
