/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import restringer.ess.AnalyzableElement;
import java.io.IOException;
import java.util.List;
import java.util.Objects;
import java.util.SortedSet;
import java.util.stream.Collectors;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.Profile;
import restringer.ess.Element;
import restringer.ess.ESS;
import restringer.ess.Linkable;

/**
 * Describes a structure in a Fallout 4 savegame.
 *
 * @author Mark Fairchild
 * @version 2016/10/02
 */
final public class StructDef extends Definition {

    /**
     * Creates a new <code>Structure</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public StructDef(LittleEndianInput input, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(ctx);

        this.NAME = ctx.STRINGS.read(input);

        int memberCount = input.readInt();
        this.MEMBERS = new java.util.ArrayList<>(memberCount);
        for (int i = 0; i < memberCount; i++) {
            MemberDesc member = new MemberDesc(input, ctx);
            this.MEMBERS.add(member);
        }
        this.instanceCount = 0;
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        assert null != output;
        this.NAME.write(output);
        output.writeInt(this.MEMBERS.size());

        for (MemberDesc member : this.MEMBERS) {
            member.write(output);
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 4;
        sum += this.NAME.calculateSize();
        sum += this.MEMBERS.stream().mapToInt(member -> member.calculateSize()).sum();
        return sum;
    }

    /**
     * @return The ID of the papyrus element.
     */
    @Override
    public TString getName() {
        return this.NAME;
    }

    /**
     * @return The list of <code>MemberDesc</code>.
     */
    public List<MemberDesc> getMembers() {
        return java.util.Collections.unmodifiableList(this.MEMBERS);
    }

    /**
     * @see PapyrusElement#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        this.MEMBERS.forEach(member -> member.resolveRefs(ess, this));
        this.NAME.addRefHolder(this);
    }

    /**
     * @see restringer.ess.Linkable#toHTML()
     * @return
     */
    @Override
    public String toHTML() {
        return String.format("<a href=\"structdef://%s\">%s</a>", this.NAME, this.NAME);
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        return this.NAME.toString();
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(restringer.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();
        BUILDER.append("<html>");

        BUILDER.append(String.format("<h3>STRUCTURE DEFINITION %ss</h3>", this.NAME));

        if (null != analysis) {
            SortedSet<String> mods = analysis.SCRIPT_ORIGINS.get(this.NAME);

            if (null != mods) {
                if (mods.size() > 1) {
                    BUILDER.append("<p>WARNING: MORE THAN ONE MOD PROVIDES THIS SCRIPT!<br />Exercise caution when editing or deleting this script!</p>");
                }

                String probablyProvider = mods.last();
                BUILDER.append(String.format("<p>This script probably came from \"%s\".</p>", probablyProvider));
                BUILDER.append("<p>Full list of providers:</p>");
                BUILDER.append("<ul>");
                mods.forEach(mod -> BUILDER.append(String.format("<li>%s", mod)));
                BUILDER.append("</ul>");
            }
        }

        BUILDER.append(String.format("<p>Contains %d member variables.</p>", this.MEMBERS.size()));

        final List<Struct> STRUCTS = save.getPapyrus()
                .getStructs()
                .values()
                .stream()
                .filter(instance -> instance.getStructDef() == this)
                .collect(Collectors.toList());

        BUILDER.append(String.format("<p>There are %d instances of this structure definition.</p>", STRUCTS.size()));
        if (STRUCTS.size() < 20) {
            BUILDER.append("<ul>");
            STRUCTS.forEach(i -> {
                String s = String.format("<li>%s</a>", i.toHTML());
                BUILDER.append(s);
            });
            BUILDER.append("</ul>");
        }
        
        /*if (null != analysis && analysis.STRUCT_ORIGINS.containsKey(this.NAME)) {
            final java.io.File PEXFILE = analysis.SCRIPTS.get(this.NAME);
            final java.io.File PARENT = PEXFILE.getParentFile();

            BUILDER.append("");
            BUILDER.append(String.format("<hr /><p>Disassembled source code:<br />(from %s)</p>", PEXFILE.getPath()));

            if (PEXFILE.exists() && PEXFILE.canRead()) {
                try {
                    final restringer.pex.Pex SCRIPT = restringer.pex.Pex.readScript(PEXFILE);

                    java.io.StringWriter code = new java.io.StringWriter();
                    SCRIPT.disassemble(code, restringer.pex.AssemblyLevel.STRIPPED);

                    BUILDER.append("<p<code><pre>");
                    BUILDER.append(code.getBuffer());
                    BUILDER.append("</pre></code></p>");

                } catch (RuntimeException ex) {
                    BUILDER.append("<p><em>Error: disassembly failed.</em></p>");
                } catch (java.io.IOException ex) {
                    BUILDER.append("<p><em>Error: couldn't read the script file.</em></p>");
                } catch (Error ex) {
                    BUILDER.append("<p><em>Error: unexpected error while reading script file.</em></p>");
                }

            } else if (PARENT.exists() && PARENT.isFile()) {
                try (restringer.LittleEndianRAF input = new restringer.LittleEndianRAF(PARENT, "r")) {
                    restringer.bsa.BSAParser BSA = new restringer.bsa.BSAParser(PARENT.getName(), input);
                    final restringer.pex.Pex SCRIPT = BSA.getScript(PEXFILE.getName());

                    java.io.StringWriter code = new java.io.StringWriter();
                    SCRIPT.disassemble(code, restringer.pex.AssemblyLevel.STRIPPED);

                    BUILDER.append("<p<code><pre>");
                    BUILDER.append(code.getBuffer());
                    BUILDER.append("</pre></code></p>");

                } catch (RuntimeException ex) {
                    BUILDER.append("<p><em>Error: disassembly failed.</em></p>");
                } catch (java.io.IOException ex) {
                    BUILDER.append("<p><em>Error: couldn't read the script file.</em></p>");
                } catch (Error ex) {
                    BUILDER.append("<p><em>Error: unexpected error while reading script file.</em></p>");
                }
            }
        }
         */
        BUILDER.append("</html>");
        return BUILDER.toString();
    }

    /**
     * @see AnalyzableElement#matches(restringer.Profile.Analysis,
     * restringer.Mod)
     * @param analysis
     * @param mod
     * @return
     */
    @Override
    public boolean matches(Profile.Analysis analysis, String mod) {
        Objects.requireNonNull(analysis);
        Objects.requireNonNull(mod);

        //final SortedSet<String> OWNERS = analysis.SCRIPT_ORIGINS.get(this.NAME);
        //return null != OWNERS && OWNERS.contains(mod);
        return false;
    }

    /**
     * @return A flag indicating if the <code>Script</code> is undefined.
     *
     */
    @Override
    public boolean isUndefined() {
        return false;
    }

    /**
     * Increments the instance count.
     */
    void incrementInstanceCount() {
        this.instanceCount++;
    }

    final private TString NAME;
    final private List<MemberDesc> MEMBERS;
    private int instanceCount;

}
