/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.Objects;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.ess.ESS;
import restringer.ess.Element;
import restringer.ess.Linkable;

/**
 * Describes a variable in a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/06/29
 */
abstract public class Variable implements PapyrusElement, Linkable {

    /**
     * Creates a new <code>Variable</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param ctx The PapyrusContext.
     * @return The new <code>Variable</code>.
     * @throws IOException
     */
    static public Variable read(LittleEndianInput input, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(ctx);

        final Type TYPE = Type.read(input);

        switch (TYPE) {
            case NULL:
                return new Null(input);
            case REF:
                return new Ref(input, ctx);
            case STRING:
                return new Str(input, ctx);
            case INTEGER:
                return new Int(input);
            case FLOAT:
                return new Flt(input);
            case BOOLEAN:
                return new Bool(input);
            case VARIANT:
                return new Variant(input, ctx);
            case STRUCT:
                return new Struct(input, ctx);
            case REF_ARRAY:
            case STRING_ARRAY:
            case INTEGER_ARRAY:
            case FLOAT_ARRAY:
            case BOOLEAN_ARRAY:
            case VARIANT_ARRAY:
            case STRUCT_ARRAY:
                return new Array(TYPE, input, ctx);
            default:
                throw new IOException();
        }
    }

    /**
     * @see PapyrusElement#addNames(restringer.Analysis)
     * @param analysis The analysis data.
     */
    @Override
    public void addNames(restringer.Analysis analysis) {
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
    }

    /**
     * @return The EID of the papyrus element.
     */
    abstract public Type getType();

    /**
     * @see restringer.ess.Linkable#toHTML()
     * @return
     */
    @Override
    public String toHTML() {
        return this.toString();
    }

    /**
     * @return A string representation that only includes the type field.
     */
    public String toTypeString() {
        return this.getType().toString();
    }

    /**
     * Checks if the variable stores a reference to something.
     *
     * @return
     */
    public boolean hasRef() {
        return false;
    }

    /**
     * Checks if the variable stores a reference to a particular something.
     *
     * @param id
     * @return
     */
    public boolean hasRef(EID id) {
        return false;
    }

    /**
     * Returns the variable's refid or null if the variable isn't a reference
     * type.
     *
     * @return
     */
    public EID getRef() {
        return null;
    }

    /**
     * Returns the variable's referent or null if the variable isn't a reference
     * type.
     *
     * @return
     */
    public GameElement getReferent() {
        return null;
    }

    /**
     * @return A string representation that doesn't include the type field.
     */
    abstract public String toValueString();

    /**
     * Variable that stores nothing.
     */
    static final public class Null extends Variable {

        public Null(LittleEndianInput input) throws IOException {
            this.VALUE = input.readInt();
        }

        @Override
        public int calculateSize() {
            return 5;
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            this.getType().write(output);
            output.writeInt(this.VALUE);
        }

        @Override
        public Type getType() {
            return Type.NULL;
        }

        @Override
        public String toValueString() {
            return "NULL";
        }

        @Override
        public String toString() {
            return "NULL";
        }

        final private int VALUE;
    }

    /**
     * ABT for a variable that stores some type of ref.
     */
    static abstract private class AbstractRef extends Variable {

        public AbstractRef(LittleEndianInput input, PapyrusContext ctx) throws IOException {
            Objects.requireNonNull(input);
            this.REFTYPE = ctx.STRINGS.read(input);

            if (ctx.GAME.isID64()) {
                this.REF = EID.read8byte(input);
            } else {
                this.REF = EID.read4byte(input);
            }

            this.ref = new restringer.ess.RefID(this.REF);
        }

        public AbstractRef(TString type, EID id) {
            this.REF = Objects.requireNonNull(id);
            this.REFTYPE = Objects.requireNonNull(type);
            this.ref = new restringer.ess.RefID(this.REF);
        }

        public boolean isNull() {
            return this.REF.isZero();
        }

        public TString getRefType() {
            return this.REFTYPE;
        }

        @Override
        public boolean hasRef() {
            return true;
        }

        @Override
        public boolean hasRef(EID id) {
            return Objects.equals(this.REF, id);
        }

        @Override
        public EID getRef() {
            return this.REF;
        }

        @Override
        public GameElement getReferent() {
            return this.referent;
        }

        @Override
        public int calculateSize() {
            int sum = 1;
            sum += this.REFTYPE.calculateSize();
            sum += this.REF.calculateSize();
            return sum;
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            this.getType().write(output);
            this.REFTYPE.write(output);
            output.writeESSElement(this.REF);
        }

        /**
         * @see PapyrusElement#addNames(restringer.Analysis)
         * @param analysis The analysis data.
         */
        @Override
        public void addNames(restringer.Analysis analysis) {
            this.ref.addNames(analysis);
        }

        /**
         * @see PapyrusElement#resolveRefs(ESS, Element)
         * @param ess The full savegame.
         * @param owner The owner of the element, or null if it is not owned.
         */
        @Override
        public void resolveRefs(ESS ess, Element owner) {
            if (ess.getPapyrus().getInstances().containsKey(this.REF)) {
                this.referent = ess.getPapyrus().getInstances().get(this.REF);

            } else if (ess.getPapyrus().getReferences().containsKey(this.REF)) {
                this.referent = ess.getPapyrus().getReferences().get(this.REF);

            } else if (ess.getPapyrus().getStructs().containsKey(this.REF)) {
                this.referent = ess.getPapyrus().getStructs().get(this.REF);

            } else if (this.REF.isZero()) {
                this.referent = null;

            } else {
                System.err.printf("A variable's referent was missing: \nVar = %s\nOwner = %s\n", this, owner);
                this.referent = null;
            }

            if (null != owner) {
                this.REFTYPE.addRefHolder(owner);
            }
        }

        /**
         * @see Variable#toTypeString()
         * @return
         */
        @Override
        public String toTypeString() {
            return this.REFTYPE.toString();
        }

        @Override
        public String toValueString() {
            return this.REF.toString() + " (" + this.REFTYPE + ")";
        }

        @Override
        public String toHTML() {
            if (null != this.referent) {
                if (this.referent instanceof ScriptInstance) {
                    return String.format("%s : <a href=\"instance://%s\">%s</a> (<a href=\"script://%s\">%s</a>)", this.getType(), this.REF, this.REF, this.REFTYPE, this.REFTYPE);
                } else if (this.referent instanceof Reference) {
                    return String.format("%s : <a href=\"reference://%s\">%s</a> (<a href=\"script://%s\">%s</a>)", this.getType(), this.REF, this.REF, this.REFTYPE, this.REFTYPE);
                }
            }

            return String.format("%s : %s (<a href=\"script://%s\">%s</a>)", this.getType(), this.REF, this.REFTYPE, this.REFTYPE);
        }

        @Override
        public String toString() {
            return this.getType() + " : " + this.toValueString();
        }

        final private TString REFTYPE;
        final private EID REF;
        final private restringer.ess.RefID ref;
        private GameElement referent;
    }

    /**
     * Variable that stores a ref. Note to self: a ref is a pointer to a papyrus
     * element, unlike a RefID which points to a form or changeform.
     *
     */
    static final public class Ref extends AbstractRef {

        public Ref(LittleEndianInput input, PapyrusContext ctx) throws IOException {
            super(input, ctx);
        }

        public Ref(TString type, EID id) {
            super(type, id);
        }

        public Ref derive(long id, ESS ess) {
            Ref derivative = new Ref(this.getRefType(), this.getRef().derive(id));
            derivative.resolveRefs(ess, null);
            return derivative;
        }

        @Override
        public Type getType() {
            return Type.REF;
        }

    }

    /**
     * Variable that stores a Variant.
     *
     */
    static final public class Variant extends Variable {

        public Variant(LittleEndianInput input, PapyrusContext ctx) throws IOException {
            Objects.requireNonNull(input);
            final Variable var = Variable.read(input, ctx);
            this.VALUE = var;
        }

        public Variable getValue() {
            return this.VALUE;
        }

        @Override
        public int calculateSize() {
            return 1 + this.VALUE.calculateSize();
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeESSElement(this.getType());
            output.writeESSElement(this.VALUE);
        }

        @Override
        public Type getType() {
            return Type.VARIANT;
        }

        @Override
        public boolean hasRef() {
            return this.VALUE.hasRef();
        }

        @Override
        public boolean hasRef(EID id) {
            return this.VALUE.hasRef(id);
        }

        @Override
        public EID getRef() {
            return this.VALUE.getRef();
        }

        @Override
        public GameElement getReferent() {
            return this.VALUE.getReferent();
        }

        @Override
        public String toValueString() {
            return this.VALUE.toValueString();
        }

        @Override
        public String toString() {
            return this.getType() + ":" + this.VALUE.toString();
        }

        @Override
        public void resolveRefs(ESS ess, Element owner) {
            this.VALUE.resolveRefs(ess, owner);
        }

        final private Variable VALUE;
    }

    /**
     * Variable that stores an UNKNOWN7.
     *
     */
    static final public class Struct extends AbstractRef {

        public Struct(LittleEndianInput input, PapyrusContext ctx) throws IOException {
            super(input, ctx);
        }

        public Struct(TString type, EID id) {
            super(type, id);
        }

        public Struct derive(long id, ESS ess) {
            Struct derivative = new Struct(this.getRefType(), this.getRef().derive(id));
            derivative.resolveRefs(ess, null);
            return derivative;
        }

        @Override
        public Type getType() {
            return Type.STRUCT;
        }

    }

    /**
     * Variable that stores a string.
     */
    static final public class Str extends Variable {

        public Str(LittleEndianInput input, PapyrusContext ctx) throws IOException {
            Objects.requireNonNull(input);
            this.STRINGS = Objects.requireNonNull(ctx.STRINGS);
            this.VALUE = this.STRINGS.read(input);
        }

        public Str(Str template, String newValue) {
            Objects.requireNonNull(template);
            Objects.requireNonNull(newValue);
            this.STRINGS = template.STRINGS;
            this.VALUE = this.STRINGS.addString(newValue);
        }

        public TString getValue() {
            return this.VALUE;
        }

        @Override
        public int calculateSize() {
            return 1 + this.VALUE.calculateSize();
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            this.getType().write(output);
            this.VALUE.write(output);
        }

        @Override
        public Type getType() {
            return Type.STRING;
        }

        /**
         * @see PapyrusElement#resolveRefs(ESS, Element)
         * @param ess The full savegame.
         * @param owner The owner of the element, or null if it is not owned.
         */
        @Override
        public void resolveRefs(ESS ess, Element owner) {
            this.VALUE.addRefHolder(owner);
        }

        @Override
        public String toValueString() {
            //return String.format("\"%s\"", this.VALUE);
            return "\"" + this.VALUE + "\"";
        }

        @Override
        public String toString() {
            //return String.format("%s:\"%s\"", this.getType(), this.VALUE);
            return this.getType() + ":" + this.toValueString();
        }

        final private StringTable STRINGS;
        final private TString VALUE;
    }

    /**
     * Variable that stores an integer.
     */
    static final public class Int extends Variable {

        public Int(LittleEndianInput input) throws IOException {
            Objects.requireNonNull(input);
            this.VALUE = input.readInt();
        }

        public Int(int val) {
            this.VALUE = val;
        }

        public int getValue() {
            return this.VALUE;
        }

        @Override
        public int calculateSize() {
            return 5;
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            this.getType().write(output);
            output.writeInt(this.VALUE);
        }

        @Override
        public Type getType() {
            return Type.INTEGER;
        }

        @Override
        public String toValueString() {
            //return String.format("%d", this.VALUE);
            return Integer.toString(this.VALUE);
        }

        @Override
        public String toString() {
            //return String.format("%s:%d", this.getType(), this.VALUE);
            return this.getType() + ":" + this.toValueString();
        }

        final private int VALUE;
    }

    /**
     * Variable that stores a float.
     */
    static final public class Flt extends Variable {

        public Flt(LittleEndianInput input) throws IOException {
            Objects.requireNonNull(input);
            this.VALUE = input.readFloat();
        }

        public Flt(float val) {
            this.VALUE = val;
        }

        public float getValue() {
            return this.VALUE;
        }

        @Override
        public int calculateSize() {
            return 5;
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            this.getType().write(output);
            output.writeFloat(this.VALUE);
        }

        @Override
        public Type getType() {
            return Type.FLOAT;
        }

        @Override
        public String toValueString() {
            //return String.format("%f", this.VALUE);
            return Float.toString(this.VALUE);
        }

        @Override
        public String toString() {
            //return String.format("%s:%f", this.getType(), this.VALUE);
            return this.getType() + ":" + this.toValueString();
        }

        final private float VALUE;
    }

    /**
     * Variable that stores a boolean.
     */
    static final public class Bool extends Variable {

        public Bool(LittleEndianInput input) throws IOException {
            Objects.requireNonNull(input);
            this.VALUE = input.readInt();
        }

        public Bool(boolean val) {
            this.VALUE = (val ? 1 : 0);
        }

        public boolean getValue() {
            return this.VALUE != 0;
        }

        @Override
        public int calculateSize() {
            return 5;
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            this.getType().write(output);
            output.writeInt(this.VALUE);
        }

        @Override
        public Type getType() {
            return Type.BOOLEAN;
        }

        @Override
        public String toValueString() {
            //return String.format("%s", Boolean.toString(this.VALUE != 0));
            return Boolean.toString(this.VALUE != 0);
        }

        @Override
        public String toString() {
            //return String.format("%s:%s", this.getType(), Boolean.toString(this.VALUE != 0));
            return this.getType() + ":" + this.toValueString();
        }

        final private int VALUE;
    }

    /**
     * Variable that stores an array.
     */
    static final public class Array extends Variable {

        protected Array(Type type, LittleEndianInput input, PapyrusContext ctx) throws IOException {
            Objects.requireNonNull(type);
            Objects.requireNonNull(input);
            this.TYPE = type;

            if (this.TYPE.isRefType()) {
                this.REFTYPE = ctx.STRINGS.read(input);
            } else {
                this.REFTYPE = null;
            }

            if (ctx.GAME.isID64()) {
                this.ARRAYID = EID.read8byte(input);
            } else {
                this.ARRAYID = EID.read4byte(input);
            }

            this.array = null;
        }

        public EID getArrayID() {
            return this.ARRAYID;
        }

        public ArrayInfo getArray() {
            return this.array;
        }

        public Type getElementType() {
            return Type.values()[this.TYPE.ordinal() - 7];
        }

        @Override
        public Type getType() {
            return this.TYPE;
        }

        @Override
        public boolean hasRef() {
            return true;
        }

        @Override
        public boolean hasRef(EID id) {
            return Objects.equals(this.ARRAYID, id);
        }

        @Override
        public EID getRef() {
            return this.getArrayID();
        }

        @Override
        public GameElement getReferent() {
            return null;
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            this.getType().write(output);

            if (this.TYPE.isRefType()) {
                this.REFTYPE.write(output);
            }

            output.writeESSElement(this.ARRAYID);
        }

        @Override
        public int calculateSize() {
            int sum = 1;
            sum += (this.TYPE.isRefType() ? this.REFTYPE.calculateSize() : 0);
            sum += this.ARRAYID.calculateSize();
            return sum;
        }

        /**
         * @see PapyrusElement#resolveRefs(ESS, Element)
         * @param ess The full savegame.
         * @param owner The owner of the element, or null if it is not owned.
         */
        @Override
        public void resolveRefs(ESS ess, Element owner) {
            if (!this.getArrayID().isZero()) {
                this.array = ess.getPapyrus().getArrays().getOrDefault(this.getArrayID(), null);
                assert null != this.array;
                assert owner instanceof PapyrusElement;
                this.array.addRefHolder((PapyrusElement) owner);
            }
        }

        @Override
        public String toTypeString() {
            if (null == this.array) {
                if (this.TYPE.isRefType()) {
                    return "" + this.REFTYPE + "[ ]";
                } else {
                    return this.getElementType() + "[ ]";
                }
            }

            if (this.TYPE.isRefType()) {
                return this.TYPE + ":" + "" + this.REFTYPE + "[" + Integer.toString(this.array.getLength()) + "]";
            } else {
                return this.TYPE + ":" + this.getElementType() + "[" + Integer.toString(this.array.getLength()) + "]";
            }
        }

        @Override
        public String toValueString() {
            if (null != this.getArray()) {
                return "" + this.ARRAYID + ": " + this.getArray().toValueString();
            } else {
                return this.ARRAYID.toString();
            }
        }

        @Override
        public String toHTML() {
            return String.format("%s : <a href=\"%s\">%s</a>", this.toTypeString(), this.ARRAYID, this.ARRAYID);
        }

        @Override
        public String toString() {
            return this.toTypeString() + " " + this.ARRAYID;
        }

        final private Type TYPE;
        final private EID ARRAYID;
        final private TString REFTYPE;
        private ArrayInfo array;
    }

    /**
     * Variable that stores an integer.
     */
    /*static final public class Array6 extends Variable {

        public Array6(LittleEndianInput input, PapyrusContext ctx) throws IOException {
            Objects.requireNonNull(input);
            Objects.requireNonNull(ctx);
            this.VALUE = new byte[8];
            int read = input.read(this.VALUE);
            assert read == 8;
        }

        public byte[] getValue() {
            return this.VALUE;
        }

        @Override
        public int calculateSize() {
            return 1 + VALUE.length;
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            this.getType().write(output);
            output.write(this.VALUE);
        }

        @Override
        public Type getType() {
            return Type.UNKNOWN6_ARRAY;
        }

        @Override
        public String toValueString() {
            //return String.format("%d", this.VALUE);
            return Arrays.toString(this.VALUE);
        }

        @Override
        public String toString() {
            //return String.format("%s:%d", this.getType(), this.VALUE);
            return this.getType() + ":" + this.toValueString();
        }

        final private byte[] VALUE;
    }*/
}
