/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.gui;

import java.io.*;
import java.util.List;
import java.util.logging.Logger;
import java.util.prefs.Preferences;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javafx.stage.FileChooser;
import javafx.stage.FileChooser.ExtensionFilter;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import org.ini4j.Ini;
import restringer.Mod;
import restringer.Game;

/**
 * Displays dialog boxes for configuring <code>ModChooser</code> and
 * <code>SaveWindow</code>.
 *
 * @author Mark Fairchild
 * @version 2016/07/07
 */
abstract public class Configurator {

    /**
     * Shows a file chooser dialog to allow the user to export a plugins list.
     *
     * @param parent The parent component.
     * @param savefile The savefile for which the list is being generated.
     * @return A <code>File</code> pointing to the export file, or
     * <code>null</code> if a file was not selected.
     */
    static public File selectPluginsExport(JFrame parent, File savefile) {
        LOG.info("Choosing an export file.");

        javafx.stage.FileChooser CHOOSER = new javafx.stage.FileChooser();
        CHOOSER.setTitle(RES.getString("ENTER_EXPORT_NAME"));
        CHOOSER.getExtensionFilters().add(new FileChooser.ExtensionFilter(RES.getString("PLUGINS_LIST"), "*.txt"));

        File previousExport = getPreviousPluginsExport();

        if (null != savefile) {
            CHOOSER.setInitialFileName(String.format("Plugins - %s.txt", savefile.getName()));
        }

        if (null != previousExport) {
            LOG.fine("Trying to use the pre-existing export directory.");
            if (Configurator.validatePluginsExport(previousExport)) {
                CHOOSER.setInitialDirectory(previousExport.getParentFile());
                if (null == savefile) {
                    CHOOSER.setInitialFileName(previousExport.getName());
                }
            }
        }

        for (;;) {
            File selection = CHOOSER.showSaveDialog(null);

            if (null == selection) {
                LOG.fine("User cancelled.");
                return null;
            }

            // Append the ".txt" if necessary.
            if (!selection.getName().matches(TXT_PATTERN)) {
                LOG.fine("Correcting file extension.");
                selection = new File(selection.getParent(), selection.getName() + ".txt");
            }

            if (selection.exists() && !selection.canWrite()) {
                final String MSG = String.format(RES.getString("FILE_NOT_WRITEABLE"), selection.getAbsolutePath());
                final String TITLE = RES.getString("UNWRITEABLE");
                LOG.warning(MSG);
                JOptionPane.showMessageDialog(parent, MSG, TITLE, JOptionPane.ERROR_MESSAGE);

            } else if (selection.exists()) {
                final String MSG = RES.getString("FILE_EXISTS_REPLACE");
                final String TITLE = RES.getString("FILE_EXISTS");
                LOG.warning(MSG);
                int overwrite = JOptionPane.showConfirmDialog(parent, MSG, TITLE, JOptionPane.OK_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE);

                if (overwrite == JOptionPane.OK_OPTION) {
                    setPreviousPluginsExport(selection);
                    return selection;
                }

            } else {
                setPreviousPluginsExport(selection);
                return selection;
            }
        }
    }

    /**
     * Shows a file chooser dialog to allow the user to select a new savegame
     * file.
     *
     * @param parent The parent component.
     * @param game Which game the save is for.
     * @return A <code>File</code> pointing to the savegame file, or
     * <code>null</code> if a file was not selected.
     */
    static public File selectNewSaveFile(JFrame parent, Game game) {
        LOG.info("Choosing a savegame.");

        javafx.stage.FileChooser CHOOSER = new javafx.stage.FileChooser();
        CHOOSER.setTitle(RES.getString("SELECT_NEW_SAVEGAME"));
        CHOOSER.getExtensionFilters().add(game.FX_FILTER);

        File previousSave = getPreviousSave();

        if (null != previousSave) {
            LOG.fine("Trying to use the pre-existing save directory.");
            if (Configurator.validateNewSavegame(previousSave)) {
                CHOOSER.setInitialDirectory(previousSave.getParentFile());
            } else if (Configurator.validateSaveDirectory(previousSave.getParentFile())) {
                CHOOSER.setInitialDirectory(previousSave.getParentFile());
            }
        } else {
            final String HOME = System.getProperty("user.home");
            final File f = new File(HOME);
            CHOOSER.setInitialDirectory(f);
        }

        for (;;) {
            File selection = CHOOSER.showSaveDialog(null);
            if (null == selection) {
                LOG.fine("User cancelled.");
                return null;
            }

            // Append the file extension if necessary.
            if (!game.testFilename(selection)) {
                LOG.fine("Correcting file extension.");
                selection = new File(selection.getParent(), selection.getName() + "." + game.EXT);
            }

            if (selection.exists() && !selection.canWrite()) {
                final String MSG = String.format(RES.getString("FILE_NOT_WRITEABLE"), selection.getAbsolutePath());
                final String TITLE = RES.getString("UNWRITEABLE");
                LOG.warning(MSG);
                JOptionPane.showMessageDialog(parent, MSG, TITLE, JOptionPane.ERROR_MESSAGE);

            } else {
                setPreviousSave(selection);
                return selection;
            }
        }
    }

    /**
     * Confirms that the user wishes to overwrite a savefile.
     *
     * @param parent The parent component.
     * @param game Which game the save is for.
     * @return A <code>File</code> pointing to the savegame file, or
     * <code>null</code> if a file was not selected.
     */
    static public File confirmSaveFile(JFrame parent, Game game) {
        LOG.info("Choosing a savegame.");

        File previousSave = getPreviousSave();

        if (previousSave.exists() && !previousSave.canWrite()) {
            final String MSG = String.format(RES.getString("FILE_NOT_WRITEABLE"), previousSave.getAbsolutePath());
            final String TITLE = RES.getString("UNWRITEABLE");
            JOptionPane.showMessageDialog(parent, MSG, TITLE, JOptionPane.ERROR_MESSAGE);
            return null;

        } else if (previousSave.exists()) {
            final String MSG = RES.getString("FILE_EXISTS_REPLACE");
            final String TITLE = RES.getString("FILE_EXISTS");
            LOG.warning(MSG);
            int overwrite = JOptionPane.showConfirmDialog(parent, MSG, TITLE, JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE);

            switch (overwrite) {
                case JOptionPane.YES_OPTION:
                    return previousSave;
                case JOptionPane.CANCEL_OPTION:
                    return null;
                default:
                    return selectNewSaveFile(parent, game);
            }

        } else {
            return previousSave;
        }
    }

    /**
     * Shows a file chooser dialog to allow the user to select a savegame file.
     *
     * @param parent The parent component.
     * @return A <code>File</code> pointing to the savegame file, or
     * <code>null</code> if a file was not selected.
     */
    static public File selectSaveFile(JFrame parent) {
        LOG.info("Choosing a savegame.");

        javafx.stage.FileChooser CHOOSER = new javafx.stage.FileChooser();
        CHOOSER.setTitle(RES.getString("SELECT_SAVEGAME"));
        CHOOSER.getExtensionFilters().add(FX_FILTER);

        File previousSave = getPreviousSave();

        if (null != previousSave) {
            LOG.fine("Trying to use the pre-existing savegame.");
            if (Configurator.validateSavegame(previousSave)) {
                CHOOSER.setInitialDirectory(previousSave.getParentFile());
                CHOOSER.setInitialFileName(previousSave.getName());
            } else if (Configurator.validateSaveDirectory(previousSave.getParentFile())) {
                CHOOSER.setInitialDirectory(previousSave.getParentFile());
            }
        } else {
            final String HOME = System.getProperty("user.home");
            final File f = new File(HOME);
            CHOOSER.setInitialDirectory(f);
        }

        for (;;) {
            File selection = CHOOSER.showOpenDialog(null);

            if (null == selection) {
                LOG.fine("User cancelled.");
                return null;

            } else if (!validateSavegame(selection)) {
                final String MSG = RES.getString("INVALID_SAVEGAME");
                final String TITLE = RES.getString("INVALID");
                LOG.warning(MSG);
                LOG.warning(MSG);
                JOptionPane.showMessageDialog(parent, MSG, TITLE, JOptionPane.ERROR_MESSAGE);

            } else {
                setPreviousSave(selection);
                return selection;
            }
        }
    }

    /**
     * Shows a file chooser dialog to allow the user to select where the a mod
     * is located.
     *
     * @param parent The parent component.
     * @return A list of files.
     */
    static public File selectMod(JFrame parent) {
        LOG.info("Choosing a mod directory.");

        javafx.stage.DirectoryChooser CHOOSER = new javafx.stage.DirectoryChooser();
        CHOOSER.setTitle(RES.getString("SELECT_MOD"));

        if (!Configurator.validateModPath(getUserModDirectory())) {
            if (Configurator.validateModOrganizerDirectory(getModOrganizerDirectory())) {
                setUserModDirectory(new File(getModOrganizerDirectory(), MODS_PATH));
            } else if (Configurator.validateModOrganizerDirectory(MO_DEFAULT_DIR)) {
                setUserModDirectory(new File(MO_DEFAULT_DIR, MODS_PATH));
                setModOrganizerDirectory(MO_DEFAULT_DIR);
            } else {
                setUserModDirectory(MOD_DEFAULT_DIR);
            }
        }

        for (;;) {
            File selection = CHOOSER.showDialog(null);

            if (null == selection) {
                LOG.fine("User cancelled.");
                return null;

            } else if (!validateModPath(selection)) {
                final String MSG = RES.getString("INVALID_MOD");
                final String TITLE = RES.getString("INVALID");
                LOG.warning(MSG);
                JOptionPane.showMessageDialog(parent, MSG, TITLE, JOptionPane.ERROR_MESSAGE);

            } else {
                setUserModDirectory(selection.getParentFile());
                return selection;
            }
        }
    }

    /**
     * Shows a file chooser dialog to allow the user to select where the patch
     * will be generated.
     *
     * @param prevDir The previous value of the field, if any.
     * @param parent The parent component.
     * @return The selected directory or null if one was not selected.
     */
    static public File selectOutputDirectory(File prevDir, JFrame parent) {
        LOG.info("Choosing an output path.");

        final JFileChooser CHOOSER = new JFileChooser();
        CHOOSER.setDialogTitle(RES.getString("SELECT_PATCH_DIR"));
        CHOOSER.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        CHOOSER.setMultiSelectionEnabled(false);

        if (validateOutputPath(prevDir)) {
            LOG.fine("Trying to use the pre-existing output path.");
            CHOOSER.setSelectedFile(prevDir);

        } else if (null != prevDir) {
            LOG.fine("Searching up the tree from the pre-existing output path.");
            while (!prevDir.exists() || !prevDir.canRead()) {
                prevDir = prevDir.getParentFile();
            }
            CHOOSER.setCurrentDirectory(prevDir);

        } else if (validateOutputPath(getUserOutputDirectory())) {
            LOG.fine("Trying to use previous profile's output directory.");
            CHOOSER.setCurrentDirectory(getUserOutputDirectory());

        } else if (validateModOrganizerDirectory(getModOrganizerDirectory())) {
            LOG.fine("Trying to use the Mod Organizer path.");
            CHOOSER.setCurrentDirectory(new File(getModOrganizerDirectory(), MODS_PATH));

        } else if (validateModOrganizerDirectory(MO_DEFAULT_DIR)) {
            LOG.fine("Trying to use the default Mod Organizer path.");
            setModOrganizerDirectory(MO_DEFAULT_DIR);
            CHOOSER.setCurrentDirectory(new File(MO_DEFAULT_DIR, MODS_PATH));
        }

        for (;;) {
            loadChooserPrefs(CHOOSER);
            int result = CHOOSER.showDialog(parent, RES.getString("SELECT"));
            File selection = CHOOSER.getSelectedFile();
            saveChooserPrefs(CHOOSER);

            if (null == selection || result == JFileChooser.CANCEL_OPTION) {
                LOG.fine("User cancelled.");
                return null;

            } else if (!validateOutputPath(selection)) {
                if (!selection.exists()) {
                    final String MSG = String.format(RES.getString("DIR_NONEXISTENT"), selection.getAbsolutePath());
                    final String TITLE = RES.getString("NONEXISTENT");
                    LOG.warning(MSG);
                    JOptionPane.showMessageDialog(parent, MSG, TITLE, JOptionPane.ERROR_MESSAGE);

                } else if (!selection.canWrite()) {
                    final String MSG = String.format(RES.getString("DIRECTORY_NOT_WRITEABLE"), selection.getAbsolutePath());
                    final String TITLE = RES.getString("UNWRITEABLE");
                    LOG.warning(MSG);
                    JOptionPane.showMessageDialog(parent, MSG, TITLE, JOptionPane.ERROR_MESSAGE);

                } else {
                    final String MSG = String.format(RES.getString("INVALID_PATCH_DIR"), selection.getAbsolutePath());
                    final String TITLE = RES.getString("INVALID");
                    LOG.warning(MSG);
                    JOptionPane.showMessageDialog(parent, MSG, TITLE, JOptionPane.ERROR_MESSAGE);
                }

            } else {
                assert Configurator.validateOutputPath(selection);
                String MSG = String.format("User selected outut path directory \"%s\".", selection.getAbsolutePath());
                LOG.info(MSG);
                setUserOutputDirectory(selection);
                return selection;
            }
        }
    }

    /**
     * Shows a file chooser dialog to allow the user to select where
     * ModOrganizer is installed. The result (if any) will be stored in the
     * settings.
     *
     * @param parent The parent component.
     * @return A <code>File</code> pointing to the selected ModOrganizer
     * directory, or <code>null</code> if a directory was not selected.
     */
    static public File selectModOrganizerDirectory(JFrame parent) {
        LOG.info("Choosing the ModOrganizer path.");

        final JFileChooser CHOOSER = new JFileChooser();
        CHOOSER.setDialogTitle(RES.getString("SELECT_MO_DIR"));
        CHOOSER.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        CHOOSER.setMultiSelectionEnabled(false);

        if (validateModOrganizerDirectory(getModOrganizerDirectory())) {
            LOG.fine("Choosing a ModOrganizer path: trying the pre-existing path.");
            CHOOSER.setSelectedFile(getModOrganizerDirectory());
        } else if (validateModOrganizerDirectory(MO_DEFAULT_DIR)) {
            LOG.fine("Choosing a ModOrganizer path: trying a default value.");
            CHOOSER.setSelectedFile(MO_DEFAULT_DIR);
        }

        for (;;) {
            loadChooserPrefs(CHOOSER);
            int result = CHOOSER.showDialog(parent, RES.getString("SELECT"));
            File selection = CHOOSER.getSelectedFile();
            saveChooserPrefs(CHOOSER);

            if (null == selection || result == JFileChooser.CANCEL_OPTION) {
                LOG.fine("User cancelled.");
                return null;

            } else if (!validateModOrganizerDirectory(selection)) {
                if (!selection.exists()) {
                    final String MSG = String.format(RES.getString("DIR_NONEXISTENT"), selection.getAbsolutePath());
                    final String TITLE = RES.getString("NONEXISTENT");
                    LOG.warning(MSG);
                    JOptionPane.showMessageDialog(parent, MSG, TITLE, JOptionPane.ERROR_MESSAGE);

                } else if (!selection.canRead()) {
                    final String MSG = String.format(RES.getString("DIR_UNREADABLE"), selection.getAbsolutePath());
                    final String TITLE = RES.getString("MO DIRECTORY");
                    LOG.warning(MSG);
                    JOptionPane.showMessageDialog(parent, MSG, TITLE, JOptionPane.ERROR_MESSAGE);

                } else if (selection.listFiles(MOD_FILTER).length == 0) {
                    final String MSG = RES.getString("NO_MODS_DIR");
                    final String TITLE = RES.getString("INVALID");
                    LOG.warning(MSG);
                    JOptionPane.showMessageDialog(parent, MSG, TITLE, JOptionPane.ERROR_MESSAGE);

                } else {
                    final String MSG = String.format(RES.getString("NOT_MO"), selection.getAbsolutePath());
                    final String TITLE = RES.getString("INVALID");
                    LOG.warning(MSG);
                    JOptionPane.showMessageDialog(parent, MSG, TITLE, JOptionPane.ERROR_MESSAGE);
                }

            } else {
                final String MSG = String.format("User selected ModOrganizer path directory \"%s\".", selection.getAbsolutePath());
                LOG.fine(MSG);

                assert validateModOrganizerDirectory(selection);
                setModOrganizerDirectory(selection);
                return selection;
            }
        }
    }

    /**
     * Shows a file chooser dialog to allow the user to select where a game is
     * located.
     *
     * @param parent The parent component.
     * @param game The game whose directory should be selected.
     * @return A <code>File</code> pointing to the selected game directory, or
     * <code>null</code> if a directory was not selected.
     */
    static public File selectGameDirectory(JFrame parent, Game game) {
        LOG.info(String.format("Choosing the %s directory.", game));

        final JFileChooser CHOOSER = new JFileChooser();
        CHOOSER.setDialogTitle(String.format(RES.getString("SELECT_GAME_DIR"), game.NAME));
        CHOOSER.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        CHOOSER.setMultiSelectionEnabled(false);

        final File PREV_DIR = getGameDirectory(game);
        if (Configurator.validateGameDirectory(game, PREV_DIR)) {
            LOG.fine("Trying to use the stored value for the game directory.");
            CHOOSER.setSelectedFile(PREV_DIR);
            CHOOSER.setCurrentDirectory(PREV_DIR);
        }

        for (;;) {
            loadChooserPrefs(CHOOSER);
            int result = CHOOSER.showOpenDialog(parent);
            File selection = CHOOSER.getSelectedFile();
            saveChooserPrefs(CHOOSER);

            if (null == selection || result == JFileChooser.CANCEL_OPTION) {
                return null;

            } else if (!validateGameDirectory(game, selection)) {
                final String MSG = String.format(RES.getString("NOT_GAMEDIR"), game.NAME);
                final String TITLE = RES.getString("INVALID");
                LOG.warning(MSG);
                JOptionPane.showMessageDialog(parent, MSG, TITLE, JOptionPane.ERROR_MESSAGE);

            } else {
                assert validateGameDirectory(game, selection);
                setGameDirectory(game, selection);
                return selection;
            }
        }
    }

    /**
     * Validates a directory, checking if it contains a valid installation of
     * ModOrganizer.
     *
     * @param dir The directory to validate.
     * @return True if the directory contains ModOrganizer, false otherwise.
     */
    static public boolean validateModOrganizerDirectory(File dir) {
        if (null == dir) {
            return false;
        } else if (!dir.isDirectory()) {
            return false;
        } else if (!dir.exists()) {
            return false;
        } else if (!dir.canRead()) {
            return false;
        } else {
            return dir.listFiles(MOD_FILTER).length > 0;
        }
    }

    /**
     * Validates a directory, checking if it contains a valid installation of a
     * game.
     *
     * @param game The game to check for.
     * @param dir The directory to validate.
     * @return True if the directory contains the game, false otherwise.
     */
    static public boolean validateGameDirectory(Game game, File dir) {
        // Filter that should only match the correct directory.
        FileFilter filter = file -> file.isDirectory()
                && file.getName().equalsIgnoreCase("data")
                && file.getParentFile().getName().equalsIgnoreCase(game.DIR)
                && new File(file.getParentFile(), game.EXE).exists();

        if (null == dir || !dir.isDirectory() || !dir.exists() || !dir.canRead()) {
            return false;
        } else if (filter.accept(dir)) {
            return true;
        } else {
            return 0 < dir.listFiles(filter).length;
        }
    }

    /**
     * Validates a directory, checking if it is a valid place to write a
     * Restringer patch.
     *
     * @param dir The directory to validate.
     * @return True if the directory exists and is writable, false otherwise.
     */
    static public boolean validateOutputPath(File dir) {
        if (null == dir) {
            return false;
        }

        if (dir.exists() && dir.isFile()) {
            return false;
        }

        if (dir.exists() && dir.isDirectory()) {
            return dir.canWrite();
        }

        dir.mkdir();
        dir.delete();
        return true;
    }

    /**
     * Validates a directory, checking if it contains a mod. In practice this
     * just means that is a directory, it exists, and it is readable.
     *
     * @param dir The directory to validate.
     * @return True if the directory exists and is readable, false otherwise.
     */
    static public boolean validateModPath(File dir) {
        if (null == dir) {
            return false;
        } else if (!dir.isDirectory()) {
            return false;
        } else if (!dir.exists()) {
            return false;
        } else {
            return dir.canRead();
        }
    }

    /**
     * Validates a file, checking if it is a valid place to save a game. In
     * practice this just means that is a file, it's parent is writable, and if
     * it exists it is also writable.
     *
     * @param file The file to validate.
     * @return True if the file either does not exist or is writable, and its
     * parent is writable.
     */
    static public boolean validateNewSavegame(File file) {
        if (null == file) {
            return false;
        } else if (file.exists() && !file.isFile()) {
            return false;
        } else if (file.exists() && !file.canWrite()) {
            return false;
        } else {
            return file.getParentFile().canWrite();
        }
    }

    /**
     * Validates a file, checking if it is a savegame. In practice this just
     * means that is a file, it exists, it is readable, and it has the "ESS" or
     * "FOS" extension.
     *
     * @param file The file to validate.
     * @return True if the file is probably a savegame.
     */
    static public boolean validateSavegame(File file) {
        if (null == file) {
            return false;
        } else if (!file.isFile()) {
            return false;
        } else if (!file.exists()) {
            return false;
        } else if (!file.canRead()) {
            return false;
        } else {
            return SAVE_FILTER.accept(file);
        }
    }

    /**
     * Validates a file, checking if it is a valid place to export a plugin
     * list. In practice this just means that is a file, it's parent is
     * writable, and if it exists it is also writable.
     *
     * @param file The file to validate.
     * @return True if the file either does not exist or is writable, and its
     * parent is writable.
     */
    static public boolean validateNewExport(File file) {
        if (null == file) {
            return false;
        } else if (file.exists() && !file.isFile()) {
            return false;
        } else if (file.exists() && !file.canWrite()) {
            return false;
        } else {
            return file.getParentFile().canWrite();
        }
    }

    /**
     * Validates a file, checking if it is a plugin list export. In practice
     * this just means that is a file, it exist, and it is readable.
     *
     * @param file The file to validate.
     * @return True if the file is probably an export.
     */
    static public boolean validatePluginsExport(File file) {
        if (null == file) {
            return false;
        } else if (file.exists() && !file.isFile()) {
            return false;
        } else if (file.exists() && !file.canWrite()) {
            return false;
        } else {
            return file.getParentFile().canWrite();
        }
    }

    /**
     * Validates a directory, checking if it is a valid place to read and write
     * savegames.. In practice this just means that is a directory, it exists,
     * and it is readable.
     *
     * @param dir The directory to validate.
     * @return True if the directory exists and is readable.
     */
    static public boolean validateSaveDirectory(File dir) {
        if (null == dir) {
            return false;
        } else if (!dir.isDirectory()) {
            return false;
        } else if (!dir.exists()) {
            return false;
        } else {
            return dir.canRead();
        }
    }

    /**
     * Analyzes the ModOrganizer directory and returns a list of mod names, in
     * the order they appear in the currently selected profile's mod list.
     *
     * @param game The game to analyze.
     * @param moDir The ModOrganizer directory.
     * @return The list of Mods, or null if the modlist file could not be read
     * for any reason.
     *
     */
    static public List<Mod> analyzeModOrganizer(Game game, File moDir) {
        assert Configurator.validateModOrganizerDirectory(moDir);
        LOG.info(RES.getString("ATTEMPTING TO ANALYZE MO"));

        final File PROFILE_DIR = new File(moDir, Configurator.PROFILES_PATH);
        final File INI_FILE = new File(moDir, Configurator.INI_PATH);
        final File MOD_DIR = new File(moDir, Configurator.MODS_PATH);

        try {
            LOG.info(RES.getString("LOADING MODORGANIZER.INI"));
            final Ini INI = new Ini();
            INI.getConfig().setLowerCaseOption(true);
            INI.load(INI_FILE);
            LOG.fine(RES.getString("MODORGANIZER.INI LOADED"));

            LOG.info(RES.getString("IMPORTING SELECTED_PROFILE"));
            String profilePath = INI.get("General", "selected_profile", String.class
            );
            LOG.fine(String.format(RES.getString("SELECTED PROFILE"), profilePath));

            final File PROFILE = new File(PROFILE_DIR, profilePath);
            final File MOD_LIST = new File(PROFILE, Configurator.MODLIST_PATH);

            LOG.info(RES.getString("READING PROFILE"));
            final List<String> MODNAMES = new java.util.LinkedList<>();

            try (BufferedReader input = new BufferedReader(new FileReader(MOD_LIST))) {
                LOG.fine(RES.getString("READING  MODLIST.TXT"));

                while (input.ready()) {
                    String line = input.readLine();
                    Matcher matcher = MODLIST_REGEX.matcher(line);
                    if (matcher.matches()) {
                        if (matcher.group(1).equals("+")) {
                            MODNAMES.add(matcher.group(2));
                        }
                    }
                }

                LOG.fine(String.format(RES.getString("CONTAINED MOD NAMES"), MODNAMES.size()));
            }

            final List<Mod> MODS = MODNAMES.stream()
                    .map(name -> new Mod(game, new File(MOD_DIR, name)))
                    .collect(Collectors.toList());

            return MODS;

        } catch (IOException ex) {
            LOG.severe(RES.getString("SOMETHING WENT WRONG"));
            return null;
        }
    }

    /**
     *
     * @param chooser
     */
    static private void saveChooserPrefs(JFileChooser chooser) {
        PREFS.putInt("chooserWidth", chooser.getSize().width);
        PREFS.putInt("chooserHeight", chooser.getSize().height);
        PREFS.putInt("chooserX", chooser.getLocation().x);
        PREFS.putInt("chooserY", chooser.getLocation().y);
    }

    /**
     *
     * @param chooser
     */
    static private void loadChooserPrefs(JFileChooser chooser) {
        int width = PREFS.getInt("chooserWidth", chooser.getSize().width);
        int height = PREFS.getInt("chooserHeight", chooser.getSize().height);
        int x = PREFS.getInt("chooserX", chooser.getLocation().x);
        int y = PREFS.getInt("chooserY", chooser.getLocation().y);
        chooser.setSize(width, height);
        chooser.setLocation(x, y);
    }

    /**
     * Getter for a game's directory field.
     *
     * @param game The game.
     * @return The directory.
     */
    static File getGameDirectory(Game game) {
        String path = "";

        switch (game) {
            case SKYRIM_LE:
                path = PREFS.get("skyrimLEDirectory", "");
                break;
            case SKYRIM_SE:
                path = PREFS.get("skyrimSEDirectory", "");
                break;
            case FALLOUT4:
                path = PREFS.get("fallout4Directory", "");
                break;
        }

        if (path.isEmpty()) {
            return null;
        }

        return new File(path);
    }

    /**
     * Setter for the a game's directory field.
     *
     * @param game The game.
     * @param dir The new directory.
     */
    static void setGameDirectory(Game game, File dir) {
        switch (game) {
            case SKYRIM_LE:
                PREFS.put("skyrimLEDirectory", dir.getPath());
                break;
            case SKYRIM_SE:
                PREFS.put("skyrimSEDirectory", dir.getPath());
                break;
            case FALLOUT4:
                PREFS.put("fallout4Directory", dir.getPath());
                break;
            default:
                throw new IllegalStateException();
        }
    }

    /**
     * Getter for the skyrim SE directory field.
     *
     * @return The directory.
     */
    static File getSkyrimSEDirectory() {
        String path = PREFS.get("skyrimSEDirectory", "");
        if (path.isEmpty()) {
            return null;
        }

        return new File(path);
    }

    /**
     * Setter for the skyrim SE directory field.
     *
     * @param dir The new directory.
     */
    static void setSkyrimSEDirectory(File dir) {
        PREFS.put("skyrimSEDirectory", dir.getPath());
    }

    /**
     * Getter for the Fallout 4 directory field.
     *
     * @return The directory.
     */
    static File getFO4Directory() {
        String path = PREFS.get("fo4Directory", "");
        if (path.isEmpty()) {
            return null;
        }

        return new File(path);
    }

    /**
     * Setter for the Fallout 4 directory field.
     *
     * @param dir The new directory.
     */
    static void setFO4Directory(File dir) {
        PREFS.put("fo4Directory", dir.getPath());
    }

    /**
     * Getter for the user mod directory field.
     *
     * @return The directory.
     */
    static File getUserModDirectory() {
        String path = PREFS.get("userModDirectory", "");
        if (path.isEmpty()) {
            return null;
        }

        return new File(path);
    }

    /**
     * Setter for the user mod directory field.
     *
     * @param dir The new directory.
     */
    static void setUserModDirectory(File dir) {
        PREFS.put("userModDirectory", dir.getPath());
    }

    /**
     * Getter for the user output directory field.
     *
     * @return The directory.
     */
    static File getUserOutputDirectory() {
        String path = PREFS.get("userOutputDirectory", "");
        if (path.isEmpty()) {
            return null;
        }

        return new File(path);
    }

    /**
     * Setter for the user output directory field.
     *
     * @param dir The new directory.
     */
    static void setUserOutputDirectory(File dir) {
        PREFS.put("userOutputDirectory", dir.getPath());
    }

    /**
     * Getter for the mod organizer directory field.
     *
     * @return The directory.
     */
    static File getModOrganizerDirectory() {
        String path = PREFS.get("modOrganizerDirectory", "");
        if (path.isEmpty()) {
            return null;
        }

        return new File(path);
    }

    /**
     * Setter for the mod organizer directory field.
     *
     * @param dir The new directory.
     */
    static void setModOrganizerDirectory(File dir) {
        PREFS.put("modOrganizerDirectory", dir.getPath());
    }

    /**
     * Getter for the previous save field.
     *
     * @return The file.
     */
    static File getPreviousSave() {
        String path = PREFS.get("previousSave", "");
        if (path.isEmpty()) {
            return null;
        }

        return new File(path);
    }

    /**
     * Setter for the previous save field.
     *
     * @param file The new file.
     */
    static void setPreviousSave(File file) {
        PREFS.put("previousSave", file.getPath());
    }

    /**
     * Getter for the previous plugins export field.
     *
     * @return The file.
     */
    static File getPreviousPluginsExport() {
        String path = PREFS.get("previousPluginsExport", "");
        if (path.isEmpty()) {
            return null;
        }

        return new File(path);
    }

    /**
     * Setter for the previous plugins export field.
     *
     * @param file The new file.
     */
    static void setPreviousPluginsExport(File file) {
        PREFS.put("previousPluginsExport", file.getPath());
    }

    /**
     * Displays an error message.
     *
     * @param parent
     * @param msgKey
     * @param titleKey
     */
    static private void error(JFrame parent, String msgKey, String titleKey) {
        final String MSG = RES.getString(msgKey);
        final String TITLE = RES.getString(titleKey);
        LOG.warning(MSG);
        JOptionPane.showMessageDialog(parent, MSG, TITLE, JOptionPane.ERROR_MESSAGE);
    }

    /**
     * Displays an error message.
     *
     * @param parent
     * @param msgKey
     * @param titleKey
     */
    static private void warning(JFrame parent, String msgKey, String titleKey) {
        final String MSG = RES.getString(msgKey);
        final String TITLE = RES.getString(titleKey);
        LOG.warning(MSG);
        JOptionPane.showMessageDialog(parent, MSG, TITLE, JOptionPane.WARNING_MESSAGE);
    }

    static final private Logger LOG = Logger.getLogger(Configurator.class.getCanonicalName());
    static final File MO_DEFAULT_DIR = new File("C:\\Program Files (x86)\\Mod Organizer");
    static final File MOD_DEFAULT_DIR = new File(System.getProperty("user.home"));
    static final String MODS_PATH = "mods";
    static final String PROFILES_PATH = "profiles";
    static final String INI_PATH = "ModOrganizer.ini";
    static final String MODLIST_PATH = "modlist.txt";
    static final String MODLIST_PATTERN = "^([+-])(.+)$";
    static final Pattern MODLIST_REGEX = Pattern.compile(MODLIST_PATTERN);

    static final private FileFilter SAVE_FILTER = (File f)
            -> Game.VALUES.stream().anyMatch(v -> v.REGEX.matcher(f.getName()).matches());

    static final private String TXT_PATTERN = ".+\\.(txt)";
    static final private Pattern TXT_REGEX = Pattern.compile(TXT_PATTERN);

    static final private FileFilter MOD_FILTER = file -> file.isDirectory() && file.getName().equalsIgnoreCase("mods");
    static final private ExtensionFilter FX_FILTER = new ExtensionFilter("Skyrim / Fallout 4 saves", "*.ESS", "*.FOS");
    static final private Preferences PREFS = Preferences.userNodeForPackage(Configurator.class);
    static final private java.util.ResourceBundle RES = java.util.ResourceBundle.getBundle("restringer/gui/Configurator");

    /**
     * Taken from StackOverflow.com
     * http://stackoverflow.com/questions/62289/read-write-to-windows-registry-using-java
     * 2016/09/07
     */
    /*static class StreamReader extends Thread {

        private InputStream is;
        private StringWriter sw = new StringWriter();

        public StreamReader(InputStream is) {
            this.is = is;
        }

        public void run() {
            try {
                int c;
                while ((c = is.read()) != -1) {
                    sw.write(c);
                }
            } catch (IOException e) {
            }
        }

        public String getResult() {
            return sw.toString();
        }
    }*/
}
