/*
 * Copyright 2016 Mark.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.gui;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import restringer.Mod;
import restringer.Profile.Analysis;
import restringer.ess.*;
import restringer.ess.papyrus.*;

/**
 *
 * @author Mark
 */
public class FilterMaker {

    /**
     * Setup a Mod analysis filter.
     *
     * @param mod
     * @param plugins
     * @param analysis
     * @return
     */
    static public Predicate<FilterTreeModel.Node> createModFilter(Mod mod, PluginInfo plugins, Analysis analysis) {
        Objects.requireNonNull(mod);
        Objects.requireNonNull(analysis);
        LOG.info(String.format("Filtering: mod = \"%s\"", mod));

        final String MODNAME = mod.getName();

        final Set<Plugin> PLUGINS = new HashSet<>();
        mod.getESPNames().stream().forEach((espName) -> plugins
                .getPlugins()
                .stream()
                .filter(p -> p.NAME.equalsIgnoreCase(espName))
                .findAny()
                .ifPresent(p -> PLUGINS.add(p)));

        Predicate<FilterTreeModel.Node> filter = node -> {
            // We need to find out what script the element came from.
            if (node.hasElement() && node.getElement() instanceof AnalyzableElement) {
                AnalyzableElement elem = (AnalyzableElement) node.getElement();
                return elem.matches(analysis, MODNAME);
            }
            return false;
        };

        for (Plugin plugin : PLUGINS) {
            filter = filter.or(createPluginFilter(plugin));
        }

        return filter;
    }

    /**
     * Setup a plugin filter.
     *
     * @param plugin
     * @return
     */
    static public Predicate<FilterTreeModel.Node> createPluginFilter(Plugin plugin) {
        Objects.requireNonNull(plugin);
        LOG.info(String.format("Filtering: plugin = \"%s\"", plugin));

        return node -> {
            // If the node doesn't contain an element, it automatically fails.
            if (!node.hasElement()) {
                return false;

            } // Check if the element is the plugin itself.
            else if (node.getElement() instanceof Plugin) {
                Plugin otherPlugin = (Plugin) node.getElement();
                return otherPlugin.equals(plugin);

            } // Check if the element is an instance with a matching refid.
            else if (node.getElement() instanceof ScriptInstance) {
                ScriptInstance instance = (ScriptInstance) node.getElement();
                RefID refID = instance.getRefID();
                if (instance.isMysteryFlag() || null == refID) {
                    return false;
                }
                return refID.getPlugin() == plugin;

            } // Check if the element is a ChangeForm with a matching refid.
            else if (node.getElement() instanceof ChangeForm) {
                ChangeForm form = (ChangeForm) node.getElement();
                RefID refID = form.getRefID();
                return (refID == null ? false : refID.getPlugin() == plugin);

            } // If the element is not an instance, it automatically fails.
            return false;
        };
    }

    /**
     * Setup a regex filter.
     *
     * @param regex
     * @return
     */
    static public Predicate<FilterTreeModel.Node> createRegexFilter(String regex) {
        Objects.requireNonNull(regex);
        LOG.info(String.format("Filtering: regex = \"%s\"", regex));

        if (!regex.isEmpty()) {
            try {
                LOG.info(String.format("Filtering: regex = \"%s\"", regex));
                Pattern pattern = Pattern.compile(regex, Pattern.CASE_INSENSITIVE);
                return node -> pattern.matcher(node.getName()).find();
            } catch (PatternSyntaxException ex) {
            }
        }

        return node -> true;
    }

    /**
     * Setup an undefined element filter.
     *
     * @return
     */
    static public Predicate<FilterTreeModel.Node> createUndefinedFilter() {
        return node -> {
            if (node.hasElement()) {
                Element e = node.getElement();
                if (e instanceof Script) {
                    return ((Script) e).isUndefined();
                } else if (e instanceof ScriptInstance) {
                    return ((ScriptInstance) e).isUndefined();
                } else if (e instanceof Reference) {
                    return ((Reference) e).isUndefined();
                } else if (e instanceof ActiveScript) {
                    return ((ActiveScript) e).isUndefined();
                } else if (e instanceof FunctionMessage) {
                    return ((FunctionMessage) e).isUndefined();
                } else if (e instanceof StackFrame) {
                    return ((StackFrame) e).isUndefined();
                } else if (e instanceof SuspendedStack) {
                    return ((SuspendedStack) e).isUndefined();
                }
            }
            return false;
        };
    }

    /**
     * Setup an unattached element filter.
     *
     * @return
     */
    static public Predicate<FilterTreeModel.Node> createUnattachedFilter() {
        return node -> {
            if (node.hasElement() && node.getElement() instanceof ScriptInstance) {
                return ((ScriptInstance) node.getElement()).isUnattached();
            }
            return false;
        };
    }

    /**
     * Setup a nullref filter.
     *
     * @return
     */
    static public Predicate<FilterTreeModel.Node> createNullRefFilter() {
        return node -> {
            if (!node.hasElement() || !(node.getElement() instanceof ChangeForm)) {
                return false;
            }

            final ChangeForm FORM = (ChangeForm) node.getElement();
            final ChangeFormData DATA = FORM.getData();
            if (!(DATA instanceof ChangeFormFLST)) {
                return false;
            }

            return ((ChangeFormFLST) DATA).containsNullrefs();
        };
    }

    /**
     * Setup a non-existent element filter.
     *
     * @return
     */
    static public Predicate<FilterTreeModel.Node> createNonExistentFilter() {
        return node -> {
            if (node.hasElement() && node.getElement() instanceof ScriptInstance) {
                ScriptInstance instance = (ScriptInstance) node.getElement();
                RefID refID = instance.getRefID();
                return refID.getType() == RefID.Type.CREATED && refID.getForm() == null;
            }
            return false;
        };
    }

    /**
     *
     * @param mod
     * @param plugins
     * @param plugin
     * @param regex
     * @param analysis
     * @param undefined
     * @param unattached
     * @param nullrefs
     * @param nonexistent
     * @param playerRefs
     * @return
     */
    static public Predicate<FilterTreeModel.Node> createFilter(Mod mod,
            Plugin plugin, String regex, PluginInfo plugins, Analysis analysis,
            boolean undefined, boolean unattached, boolean nullrefs,
            boolean nonexistent) {

        LOG.info("Updating filter.");
        ArrayList<Predicate<FilterTreeModel.Node>> FILTERS = new ArrayList<>(4);
        ArrayList<Predicate<FilterTreeModel.Node>> SUBFILTERS = new ArrayList<>(4);

        // Setup a Mod analysis filter.
        if (null != mod && null != analysis && null != plugins) {
            FILTERS.add(createModFilter(mod, plugins, analysis));
        }

        // Setup a plugin filter.
        if (null != plugin) {
            FILTERS.add(createPluginFilter(plugin));
        }

        // Setup a regex filter.
        if (!regex.isEmpty()) {
            FILTERS.add(createRegexFilter(regex));
        }

        // Filter undefined.
        if (undefined) {
            SUBFILTERS.add(createUndefinedFilter());
        }

        // Filter unattached.
        if (unattached) {
            SUBFILTERS.add(createUnattachedFilter());
        }

        // Filter formlists containing nullrefs.
        if (nullrefs) {
            SUBFILTERS.add(createNullRefFilter());
        }

        // Filter instances attached to nonexistent created forms.
        if (nonexistent) {
            SUBFILTERS.add(createNonExistentFilter());
        }

        // Combine the filters.
        // OR the subfilters together.
        Predicate<FilterTreeModel.Node> subfilter = null;

        for (Predicate<FilterTreeModel.Node> f : SUBFILTERS) {
            subfilter = (null == subfilter ? f : subfilter.or(f));
        }

        if (null != subfilter) {
            FILTERS.add(subfilter);
        }

        // AND the main filters together.
        Predicate<FilterTreeModel.Node> mainfilter = null;

        for (Predicate<FilterTreeModel.Node> f : FILTERS) {
            mainfilter = (null == mainfilter ? f : mainfilter.and(f));
        }

        return mainfilter;
    }

    /**
     *
     */
    static final private Logger LOG = Logger.getLogger(FilterMaker.class.getCanonicalName());
}
